/*
 * Copyright (C) 2016-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Reset and clock control (RCC) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	struct process process;

	unsigned int count_systickclk;
	unsigned int count_fclk;
	unsigned int count_apb1clk;
	unsigned int count_apb2clk;
	unsigned int count_apb1timclk;
	unsigned int count_apb2timclk;

	/* Clock control register (RCC_CR) */
	uint8_t plli2son;
	uint8_t pllon;
	uint8_t csson;
	uint8_t hsebyp;
	uint8_t hseon;
	uint8_t hsical;
	uint8_t hsitrim;
	uint8_t hsion;

	/* PLL configuration register (RCC_PLLCFGR) */
	uint8_t pllq;
	uint8_t pllsrc;
	uint8_t pllp;
	uint16_t plln;
	uint8_t pllm;

	/* Clock configuration register (RCC_CFGR) */
	uint8_t mco2;
	uint8_t mco2pre;
	uint8_t mco1pre;
	uint8_t i2sscr;
	uint8_t mco1;
	uint8_t rtcpre;
	uint8_t ppre2;
	uint8_t ppre1;
	uint8_t hpre;
	uint8_t sws;
	uint8_t sw;

	/* Clock interrupt register (RCC_CIR) */
	uint8_t plli2srdyie;
	uint8_t pllrdyie;
	uint8_t hserdyie;
	uint8_t hsirdyie;
	uint8_t lserdyie;
	uint8_t lsirdyie;
	uint8_t cssf;
	uint8_t plli2srdyf;
	uint8_t pllrdyf;
	uint8_t hserdyf;
	uint8_t hsirdyf;
	uint8_t lserdyf;
	uint8_t lsirdyf;

	/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
	uint8_t dma2en;
	uint8_t dma1en;
	uint8_t crcen;
	uint8_t gpiohen;
	uint8_t gpioeen;
	uint8_t gpioden;
	uint8_t gpiocen;
	uint8_t gpioben;
	uint8_t gpioaen;

	/* APB1 peripheral clock enable register (RCC_APB1ENR) */
	uint8_t pwren;
	uint8_t i2c3en;
	uint8_t i2c2en;
	uint8_t i2c1en;
	uint8_t usart2en;
	uint8_t spi3en;
	uint8_t spi2en;
	uint8_t wwdgen;
	uint8_t tim5en;
	uint8_t tim4en;
	uint8_t tim3en;
	uint8_t tim2en;

	/* APB2 peripheral clock enable register (RCC_APB1ENR) */
	uint8_t spi5en;
	uint8_t tim11en;
	uint8_t tim10en;
	uint8_t tim9en;
	uint8_t syscfgen;
	uint8_t spi4en;
	uint8_t spi1en;
	uint8_t sdioen;
	uint8_t adc1en;
	uint8_t usart6en;
	uint8_t usart1en;
	uint8_t tim1en;

	/* APB1 peripheral reset register (RCC_APB1RSTR) */
	uint32_t rst1;

	/* APB2 peripheral reset register (RCC_APB2RSTR) */
	uint32_t rst2;

	/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
	uint8_t plli2sr;
	uint16_t plli2sn;
	uint8_t plli2sm;

	/* Dedicated Clocks Configuration Register (RCC_DCKCFGR) */
	uint8_t timpre;

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	int irq;

	irq = cpssp->NAME.cssf;
	irq |= cpssp->NAME.plli2srdyf & cpssp->NAME.plli2srdyie;
	irq |= cpssp->NAME.pllrdyf & cpssp->NAME.plli2srdyie;
	irq |= cpssp->NAME.hserdyf & cpssp->NAME.hserdyie;
	irq |= cpssp->NAME.hsirdyf & cpssp->NAME.hsirdyie;
	irq |= cpssp->NAME.lserdyf & cpssp->NAME.lserdyie;
	irq |= cpssp->NAME.lsirdyf & cpssp->NAME.lsirdyie;

	// NAME_(irq_out_set)(cpssp, irq);
}

static void
NAME_(rst1_update)(struct cpssp *cpssp)
{
	unsigned int n;

	for (n = 0; n < 32; n++) {
		NAME_(apb1_resetN_set)(cpssp, n, (cpssp->NAME.rst1 >> n) & 1);
	}
}

static void
NAME_(rst2_update)(struct cpssp *cpssp)
{
	unsigned int n;

	for (n = 0; n < 32; n++) {
		NAME_(apb2_resetN_set)(cpssp, n, (cpssp->NAME.rst2 >> n) & 1);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* Clock control register (RCC_CR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-28: Reserved */
			/* Bit 27: Read-only */
			cpssp->NAME.plli2son = (val >> 26) & 1;
			/* Bit 25: Read-only */
			cpssp->NAME.pllon = (val >> 24) & 1;
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-20: Reserved */
			cpssp->NAME.csson = (val >> 19) & 1;
			cpssp->NAME.hsebyp = (val >> 18) & 1;
			/* Bit 17: Read-only */
			cpssp->NAME.hseon = (val >> 16) & 1;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.hsical = (val >> 8) & 0xff;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.hsitrim = (val >> 3) & 0x1f;
			/* Bit 2: Reserved */
			/* Bit 1: Read-only */
			cpssp->NAME.hsion = (val >> 0) & 1;
		}
		break;
	case 0x004:
		/* PLL configuration register (RCC_PLLCFGR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-28: Reserved */
			cpssp->NAME.pllq = (val >> 24) & 0xf;
		}
		if ((bs >> 2) & 1) {
			/* Bit 23: Reserved */
			cpssp->NAME.pllsrc = (val >> 22) & 1;
			/* Bit 21-18: Reserved */
			cpssp->NAME.pllp = (val >> 16) & 0x3;
		}
		if ((bs >> 1) & 1) {
			/* Bit 15: Reserved */
			cpssp->NAME.plln &= ~(0x7f << 2);
			cpssp->NAME.plln |= ((val >> 8) & 0x7f) << 2;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.plln &= ~3;
			cpssp->NAME.plln |= (val >> 6) & 0x3;
			cpssp->NAME.pllm = (val >> 0) & 0x3f;
		}
		break;
	case 0x008:
		/* Clock configuration register (RCC_CFGR) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.mco2 = (val >> 30) & 0x3;
			cpssp->NAME.mco2pre = (val >> 27) & 0x7;
			cpssp->NAME.mco1pre = (val >> 24) & 0x7;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.i2sscr = (val >> 23) & 1;
			cpssp->NAME.mco1 = (val >> 21) & 0x3;
			cpssp->NAME.rtcpre = (val >> 16) & 0x1f;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ppre2 = (val >> 13) & 0x7;
			cpssp->NAME.ppre1 = (val >> 10) & 0x7;
			/* Bit 9-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.hpre = (val >> 4) & 0xf;
			/* Bit 3-2: Read-only */
			cpssp->NAME.sw = (val >> 0) & 0x3;
		}
		break;
	case 0x00c:
		/* Clock interrupt register (RCC_CIR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.cssf &= ~((val >> 23) & 1);
			/* Bit 22: Reserved */
			cpssp->NAME.plli2srdyf &= ~((val >> 21) & 1);
			cpssp->NAME.pllrdyf &= ~((val >> 20) & 1);
			cpssp->NAME.hserdyf &= ~((val >> 19) & 1);
			cpssp->NAME.hsirdyf &= ~((val >> 18) & 1);
			cpssp->NAME.lserdyf &= ~((val >> 17) & 1);
			cpssp->NAME.lsirdyf &= ~((val >> 16) & 1);
		}
		if ((bs >> 1) & 1) {
			/* Bit 15-14: Reserved */
			cpssp->NAME.plli2srdyie = (val >> 13) & 1;
			cpssp->NAME.pllrdyie = (val >> 12) & 1;
			cpssp->NAME.hserdyie = (val >> 11) & 1;
			cpssp->NAME.hsirdyie = (val >> 10) & 1;
			cpssp->NAME.lserdyie = (val >> 9) & 1;
			cpssp->NAME.lsirdyie = (val >> 8) & 1;
		}
		if ((bs >> 0) & 1) {
			/* Bit 7: Read-only */
			/* Bit 6: Reserved */
			/* Bit 5-0: Read-only */
		}
		NAME_(irq_update)(cpssp);
		break;
	case 0x020:
		/* APB1 peripheral reset register (RCC_APB1RSTR) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.rst1 &= ~(0xff << 24);
			cpssp->NAME.rst1 |= val & (0xff << 24);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.rst1 &= ~(0xff << 16);
			cpssp->NAME.rst1 |= val & (0xff << 16);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.rst1 &= ~(0xff << 8);
			cpssp->NAME.rst1 |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.rst1 &= ~(0xff << 0);
			cpssp->NAME.rst1 |= val & (0xff << 0);
		}
		NAME_(rst1_update)(cpssp);
		break;
	case 0x024:
		/* APB2 peripheral reset register (RCC_APB2RSTR) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.rst2 &= ~(0xff << 24);
			cpssp->NAME.rst2 |= val & (0xff << 24);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.rst2 &= ~(0xff << 16);
			cpssp->NAME.rst2 |= val & (0xff << 16);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.rst2 &= ~(0xff << 8);
			cpssp->NAME.rst2 |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.rst2 &= ~(0xff << 0);
			cpssp->NAME.rst2 |= val & (0xff << 0);
		}
		NAME_(rst2_update)(cpssp);
		break;
	case 0x030:
		/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23: Reserved */
			cpssp->NAME.dma2en = (val >> 22) & 1;
			cpssp->NAME.dma1en = (val >> 21) & 1;
			/* Bit 20-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Bit 15-13: Reserved */
			cpssp->NAME.crcen = (val >> 12) & 1;
			/* Bit 11-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.gpiohen = (val >> 7) & 1;
			/* Bit 6-5: Reserved */
			cpssp->NAME.gpioeen = (val >> 4) & 1;
			cpssp->NAME.gpioden = (val >> 3) & 1;
			cpssp->NAME.gpiocen = (val >> 2) & 1;
			cpssp->NAME.gpioben = (val >> 1) & 1;
			cpssp->NAME.gpioaen = (val >> 0) & 1;
		}
		break;
	case 0x040:
		/* APB1 peripheral clock enable register (RCC_APB1ENR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-29: Reserved */
			cpssp->NAME.pwren = (val >> 28) & 1;
			/* Bit 27-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.i2c3en = (val >> 23) & 1;
			cpssp->NAME.i2c2en = (val >> 22) & 1;
			cpssp->NAME.i2c1en = (val >> 21) & 1;
			/* Bit 20-18: Reserved */
			cpssp->NAME.usart2en = (val >> 17) & 1;
			/* Bit 16: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.spi3en = (val >> 15) & 1;
			cpssp->NAME.spi2en = (val >> 14) & 1;
			/* Bit 13-12: Reserved */
			cpssp->NAME.wwdgen = (val >> 11) & 1;
			/* Bit 10-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-4: Reserved */
			cpssp->NAME.tim5en = (val >> 3) & 1;
			cpssp->NAME.tim4en = (val >> 2) & 1;
			cpssp->NAME.tim3en = (val >> 1) & 1;
			cpssp->NAME.tim2en = (val >> 0) & 1;
		}
		break;
	case 0x044:
		/* APB2 peripheral clock enable register (RCC_APB1ENR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-21: Reserved */
			cpssp->NAME.spi5en = (val >> 20) & 1;
			/* Bit 19: Reserved */
			cpssp->NAME.tim11en = (val >> 18) & 1;
			cpssp->NAME.tim10en = (val >> 17) & 1;
			cpssp->NAME.tim9en = (val >> 16) & 1;
		}
		if ((bs >> 1) & 1) {
			/* Bit 15: Reserved */
			cpssp->NAME.syscfgen = (val >> 14) & 1;
			cpssp->NAME.spi4en = (val >> 13) & 1;
			cpssp->NAME.spi1en = (val >> 12) & 1;
			cpssp->NAME.sdioen = (val >> 11) & 1;
			/* Bit 10-9: Reserved */
			cpssp->NAME.adc1en = (val >> 8) & 1;
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-6: Reserved */
			cpssp->NAME.usart6en = (val >> 5) & 1;
			cpssp->NAME.usart1en = (val >> 4) & 1;
			/* Bit 3-1: Reserved */
			cpssp->NAME.tim1en = (val >> 0) & 1;
		}
		break;
	case 0x84:
		/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
		if ((bs >> 3) & 1) {
			/* Bit 31: Reserved */
			cpssp->NAME.plli2sr = (val >> 28) & 0x7;
			/* Bit 27-14: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Bit 15: Reserved */
			cpssp->NAME.plli2sn &= ~(0x7f << 2);
			cpssp->NAME.plli2sn |= ((val >> 8) & 0x7f) << 2;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.plli2sn &= ~3;
			cpssp->NAME.plli2sn |= (val >> 6) & 3;
			cpssp->NAME.plli2sm = (val >> 0) & 0x3f;
		}
		break;
	case 0x8c:
		/* Dedicated Clocks Configuration Register (RCC_DCKCFGR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-25: Reserved */
			cpssp->NAME.timpre = (val >> 24) & 1;
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-16: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 15-8: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 7-0: Reserved */
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x3ff;

	switch (addr) {
	case 0x000:
		/* Clock control register (RCC_CR) */
		*valp = 0;

		/* Bit 31-28: Reserved */
		*valp |= cpssp->NAME.plli2son << 27; /* plli2srdy */
		*valp |= cpssp->NAME.plli2son << 26;
		*valp |= cpssp->NAME.pllon << 25; /* pllrdy */
		*valp |= cpssp->NAME.pllon << 24;

		/* Bit 23-20: Reserved */
		*valp |= cpssp->NAME.csson << 19;
		*valp |= cpssp->NAME.hsebyp << 18;
		*valp |= cpssp->NAME.hseon << 17; /* hserdy */
		*valp |= cpssp->NAME.hseon << 16;

		*valp |= cpssp->NAME.hsical << 8;

		*valp |= cpssp->NAME.hsitrim << 3;
		/* Bit 2: Reserved */
		*valp |= cpssp->NAME.hsion << 1; /* hsirdy */
		*valp |= cpssp->NAME.hsion << 0;
		break;
	case 0x004:
		/* PLL configuration register (RCC_PLLCFGR) */
		*valp = 0;

		/* Bit 31-28: Reserved */
		*valp |= cpssp->NAME.pllq << 24;

		/* Bit 23: Reserved */
		*valp |= cpssp->NAME.pllsrc << 22;
		/* Bit 21-18: Reserved */
		*valp |= cpssp->NAME.pllp << 16;

		/* Bit 15: Reserved */
		*valp |= cpssp->NAME.plln << 6;

		*valp |= cpssp->NAME.pllm << 0;
		break;
	case 0x008:
		/* Clock configuration register (RCC_CFGR) */
		*valp = 0;

		*valp |= cpssp->NAME.mco2 << 30;
		*valp |= cpssp->NAME.mco2pre << 27;
		*valp |= cpssp->NAME.mco1pre << 24;

		*valp |= cpssp->NAME.i2sscr << 23;
		*valp |= cpssp->NAME.mco1 << 21;
		*valp |= cpssp->NAME.rtcpre << 16;

		*valp |= cpssp->NAME.ppre2 << 13;
		*valp |= cpssp->NAME.ppre1 << 10;
		/* Bit 9-8: Reserved */

		*valp |= cpssp->NAME.hpre << 4;
		*valp |= cpssp->NAME.sw << 2; /* sws */
		*valp |= cpssp->NAME.sw << 0;
		break;
	case 0x00c:
		/* Clock interrupt register (RCC_CIR) */
		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23: Write-only */
		/* Bit 22: Reserved */
		/* Bit 21-16: Write-only */

		/* Bit 15-14: Reserved */
		*valp |= cpssp->NAME.plli2srdyie << 13;
		*valp |= cpssp->NAME.pllrdyie << 12;
		*valp |= cpssp->NAME.hserdyie << 11;
		*valp |= cpssp->NAME.hsirdyie << 10;
		*valp |= cpssp->NAME.lserdyie << 9;
		*valp |= cpssp->NAME.lsirdyie << 8;

		*valp |= cpssp->NAME.cssf << 7;
		/* Bit 6: Reserved */
		*valp |= cpssp->NAME.plli2srdyf << 5;
		*valp |= cpssp->NAME.pllrdyf << 4;
		*valp |= cpssp->NAME.hserdyf << 3;
		*valp |= cpssp->NAME.hsirdyf << 2;
		*valp |= cpssp->NAME.lserdyf << 1;
		*valp |= cpssp->NAME.lsirdyf << 0;
		break;
	case 0x020:
		/* APB1 peripheral reset register (RCC_APB1RSTR) */
		*valp = cpssp->NAME.rst1;
		break;
	case 0x024:
		/* APB2 peripheral reset register (RCC_APB1RSTR) */
		*valp = cpssp->NAME.rst2;
		break;
	case 0x030:
		/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23: Reserved */
		*valp |= cpssp->NAME.dma2en << 22;
		*valp |= cpssp->NAME.dma1en << 21;
		/* Bit 20-16: Reserved */

		/* Bit 15-13: Reserved */
		*valp |= cpssp->NAME.crcen << 12;
		/* Bit 11-8: Reserved */

		*valp |= cpssp->NAME.gpiohen << 7;
		/* Bit 15-13: Reserved */
		*valp |= cpssp->NAME.gpioeen << 4;
		*valp |= cpssp->NAME.gpioden << 3;
		*valp |= cpssp->NAME.gpiocen << 2;
		*valp |= cpssp->NAME.gpioben << 1;
		*valp |= cpssp->NAME.gpioaen << 0;
		break;
	case 0x040:
		/* APB1 peripheral clock enable register (RCC_APB1ENR) */
		*valp = 0;

		/* Bit 31-29: Reserved */
		*valp |= cpssp->NAME.pwren << 28;
		/* Bit 27-24: Reserved */

		*valp |= cpssp->NAME.i2c3en << 23;
		*valp |= cpssp->NAME.i2c2en << 22;
		*valp |= cpssp->NAME.i2c1en << 21;
		/* Bit 20-18: Reserved */
		*valp |= cpssp->NAME.usart2en << 17;
		/* Bit 16: Reserved */

		*valp |= cpssp->NAME.spi3en << 15;
		*valp |= cpssp->NAME.spi2en << 14;
		/* Bit 13-12: Reserved */
		*valp |= cpssp->NAME.wwdgen << 11;
		/* Bit 10-8: Reserved */

		/* Bit 7-4: Reserved */
		*valp |= cpssp->NAME.tim5en << 3;
		*valp |= cpssp->NAME.tim4en << 2;
		*valp |= cpssp->NAME.tim3en << 1;
		*valp |= cpssp->NAME.tim2en << 0;
		break;
	case 0x044:
		/* APB2 peripheral clock enable register (RCC_APB1ENR) */
		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23-21: Reserved */
		*valp |= cpssp->NAME.spi5en << 20;
		/* Bit 19: Reserved */
		*valp |= cpssp->NAME.tim11en << 18;
		*valp |= cpssp->NAME.tim10en << 17;
		*valp |= cpssp->NAME.tim9en << 16;

		/* Bit 15: Reserved */
		*valp |= cpssp->NAME.syscfgen << 14;
		*valp |= cpssp->NAME.spi4en << 13;
		*valp |= cpssp->NAME.spi1en << 12;
		*valp |= cpssp->NAME.sdioen << 11;
		/* Bit 10-9: Reserved */
		*valp |= cpssp->NAME.adc1en << 8;

		/* Bit 7-6: Reserved */
		*valp |= cpssp->NAME.usart6en << 5;
		*valp |= cpssp->NAME.usart1en << 4;
		/* Bit 3-1: Reserved */
		*valp |= cpssp->NAME.tim1en << 0;
		break;
	case 0x84:
		/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
		*valp = 0;

		/* Bit 31: Reserved */
		*valp |= cpssp->NAME.plli2sr << 28;
		/* Bit 27-24: Reserved */

		/* Bit 23-16: Reserved */

		/* Bit 15: Reserved */
		*valp |= cpssp->NAME.plli2sn << 6;

		*valp |= cpssp->NAME.plli2sm << 0;
		break;
	case 0x8c:
		/* Dedicated Clocks Configuration Register (RCC_DCKCFGR) */
		*valp = 0;

		/* Bit 31-25: Reserved */
		*valp |= cpssp->NAME.timpre << 24;
		/* Bit 23-0: Reserved */
		break;
	default:
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Clock control register (RCC_CR) */
	cpssp->NAME.plli2son = 0;
	cpssp->NAME.pllon = 0;
	cpssp->NAME.hsebyp = 0;
	cpssp->NAME.hseon = 0;
	cpssp->NAME.hsical = 0x00;
	cpssp->NAME.hsitrim = 0x10;
	cpssp->NAME.hsion = 1;

	/* PLL configuration register (RCC_PLLCFGR) */
	cpssp->NAME.pllq = 0x4;
	cpssp->NAME.pllsrc = 0;
	cpssp->NAME.pllp = 0x0;
	cpssp->NAME.plln = 0x30 << 2;
	cpssp->NAME.pllm = 0x10;

	/* Clock configuration register (RCC_CFGR) */
	cpssp->NAME.mco2 = 0x0;
	cpssp->NAME.mco2pre = 0x0;
	cpssp->NAME.mco1pre = 0x0;
	cpssp->NAME.i2sscr = 0;
	cpssp->NAME.mco1 = 0x0;
	cpssp->NAME.rtcpre = 0x00;
	cpssp->NAME.ppre2 = 0x0;
	cpssp->NAME.ppre1 = 0x0;
	cpssp->NAME.hpre = 0x0;
	cpssp->NAME.sw = 0x0;

	/* Clock interrupt register (RCC_CIR) */
	cpssp->NAME.plli2srdyie = 0;
	cpssp->NAME.pllrdyie = 0;
	cpssp->NAME.hserdyie = 0;
	cpssp->NAME.hsirdyie = 0;
	cpssp->NAME.lserdyie = 0;
	cpssp->NAME.lsirdyie = 0;
	cpssp->NAME.cssf = 0;
	cpssp->NAME.plli2srdyf = 0;
	cpssp->NAME.pllrdyf = 0;
	cpssp->NAME.hserdyf = 0;
	cpssp->NAME.hsirdyf = 0;
	cpssp->NAME.lserdyf = 0;
	cpssp->NAME.lsirdyf = 0;

	/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
	cpssp->NAME.dma2en = 0;
	cpssp->NAME.dma1en = 0;
	cpssp->NAME.crcen = 0;
	cpssp->NAME.gpiohen = 0;
	cpssp->NAME.gpioeen = 0;
	cpssp->NAME.gpioden = 0;
	cpssp->NAME.gpiocen = 0;
	cpssp->NAME.gpioben = 0;
	cpssp->NAME.gpioaen = 0;

	/* APB1 peripheral clock enable register (RCC_APB1ENR) */
	cpssp->NAME.pwren = 0;
	cpssp->NAME.i2c3en = 0;
	cpssp->NAME.i2c2en = 0;
	cpssp->NAME.i2c1en = 0;
	cpssp->NAME.usart2en = 0;
	cpssp->NAME.spi3en = 0;
	cpssp->NAME.spi2en = 0;
	cpssp->NAME.wwdgen = 0;
	cpssp->NAME.tim5en = 0;
	cpssp->NAME.tim4en = 0;
	cpssp->NAME.tim3en = 0;
	cpssp->NAME.tim2en = 0;

	/* APB2 peripheral clock enable register (RCC_APB1ENR) */
	cpssp->NAME.spi5en = 0;
	cpssp->NAME.tim11en = 0;
	cpssp->NAME.tim10en = 0;
	cpssp->NAME.tim9en = 0;
	cpssp->NAME.syscfgen = 0;
	cpssp->NAME.spi4en = 0;
	cpssp->NAME.spi1en = 0;
	cpssp->NAME.adc1en = 0;
	cpssp->NAME.usart6en = 0;
	cpssp->NAME.usart1en = 0;
	cpssp->NAME.tim1en = 0;

	/* APB1 peripheral reset register (RCC_APB1RSTR) */
	cpssp->NAME.rst1 = 0x00000000;

	/* APB2 peripheral reset register (RCC_APB1RSTR) */
	cpssp->NAME.rst2 = 0x00000000;

	/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
	cpssp->NAME.plli2sr = 2;
	cpssp->NAME.plli2sn = 3 << 6;
	cpssp->NAME.plli2sm = 0;

	/* Dedicated Clocks Configuration Register (RCC_DCKCFGR) */
	cpssp->NAME.timpre = 0;
}

static void
NAME_(systickclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_systickclk += mul;
	while (div <= cpssp->NAME.count_systickclk) {
		NAME_(cortex_system_timer)(cpssp);

		cpssp->NAME.count_systickclk -= div;
	}
}

static void
NAME_(apb1clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_apb1clk += mul;
	while (div <= cpssp->NAME.count_apb1clk) {
		NAME_(apb1_clocks)(cpssp); /* FIXME */

		cpssp->NAME.count_apb1clk -= div;
	}
}

static void
NAME_(apb2clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_apb2clk += mul;
	while (div <= cpssp->NAME.count_apb2clk) {
		NAME_(apb2_clocks)(cpssp); /* FIXME */

		cpssp->NAME.count_apb2clk -= div;
	}
}

static void
NAME_(apb1timclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_apb1clk += mul;
	while (div <= cpssp->NAME.count_apb1timclk) {
		NAME_(apb1_timer_clocks)(cpssp); /* FIXME */

		cpssp->NAME.count_apb1timclk -= div;
	}
}

static void
NAME_(apb2timclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_apb2timclk += mul;
	while (div <= cpssp->NAME.count_apb2timclk) {
		NAME_(apb2_timer_clocks)(cpssp); /* FIXME */

		cpssp->NAME.count_apb2timclk -= div;
	}
}

static void
NAME_(fclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	static const unsigned int shift[8] = {
		0, 0, 0, 0, 1, 2, 3, 4,
	};
	static const unsigned int timshift[2][8] = {
		{ 0, 0, 0, 0, 0, 1, 2, 3 },
		{ 0, 0, 0, 0, 0, 0, 1, 2 },
	};

	NAME_(apb1clk)(cpssp, mul, div << shift[cpssp->NAME.ppre1]);
	NAME_(apb2clk)(cpssp, mul, div << shift[cpssp->NAME.ppre2]);
	NAME_(apb1timclk)(cpssp, mul,
			div << timshift[cpssp->NAME.timpre][cpssp->NAME.ppre1]);
	NAME_(apb2timclk)(cpssp, mul,
			div << timshift[cpssp->NAME.timpre][cpssp->NAME.ppre2]);
#if 0 /* FIXME */
	NAME_(systickclk)(cpssp, mul, div * 8);
#else
	NAME_(systickclk)(cpssp, mul, div);
#endif

	cpssp->NAME.count_fclk += mul;
	while (div <= cpssp->NAME.count_fclk) {
		NAME_(hclk)(cpssp);
		/* NAME_(fclk)(cpssp); FIXME */

		cpssp->NAME.count_fclk -= div;
	}
}

static void
NAME_(sysclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	static const unsigned int shift[16] = {
		0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 6, 7, 8, 9,
	};

	div <<= shift[cpssp->NAME.hpre];

	NAME_(fclk)(cpssp, mul, div);
}

static void
NAME_(pllclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* MCO2 - FIXME */
	if (cpssp->NAME.sw == 0b10) {
		NAME_(sysclk)(cpssp, mul, div);
	}
}

static void
NAME_(pll48clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* FIXME */
}

static void
NAME_(plli2sclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* FIXME */
}

static void
NAME_(pllinclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	div *= cpssp->NAME.pllm;
	mul *= cpssp->NAME.plln;

	NAME_(pllclk)(cpssp, mul, div * (cpssp->NAME.pllp + 1) * 2);
	NAME_(pll48clk)(cpssp, mul, div * cpssp->NAME.pllq);
}

static void
NAME_(plli2sinclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	div *= cpssp->NAME.plli2sm;
	mul *= cpssp->NAME.plli2sn;

	NAME_(plli2sclk)(cpssp, mul, div * cpssp->NAME.plli2sr);
}

static void
NAME_(hse_clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* MCO1 - FIXME */
	/* MCO2 - FIXME */
	/* RTCPRE - FIXME */
	if (cpssp->NAME.sw == 0b01) {
		NAME_(sysclk)(cpssp, mul, div);
	}
	if (cpssp->NAME.pllsrc == 1) {
		NAME_(pllinclk)(cpssp, mul, div);
		NAME_(plli2sinclk)(cpssp, mul, div);
	}
}

static void
NAME_(hsi_clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* MCO1 - FIXME */
	if (cpssp->NAME.sw == 0b00) {
		NAME_(sysclk)(cpssp, mul, div);
	}
	if (cpssp->NAME.pllsrc == 0) {
		NAME_(pllinclk)(cpssp, mul, div);
		NAME_(plli2sinclk)(cpssp, mul, div);
	}
}

static void
NAME_(lse_clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* FIXME */
}

static void
NAME_(lsi_clk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	/* FIXME */
}

static void __attribute__((__noreturn__))
NAME_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	while (cpssp->NAME.process.inst_cnt < cpssp->NAME.process.inst_limit) {
		NAME_(hse_clk)(cpssp, 1, 1); /* FIXME */
		NAME_(hsi_clk)(cpssp, 16/8, 1);
		NAME_(lse_clk)(cpssp, 1, 8*1024/32);
		NAME_(lsi_clk)(cpssp, 1, 8*1024/32);
		cpssp->NAME.process.inst_cnt++;
	}

	sched_to_scheduler();
	goto again;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	cpssp->NAME.process.inst_hz = 8*1024*1024;
	sched_process_init(&cpssp->NAME.process, NAME_(step), cpssp);

	cpssp->NAME.count_systickclk = 0;
	cpssp->NAME.count_fclk = 0;
	cpssp->NAME.count_apb1clk = 0;
	cpssp->NAME.count_apb2clk = 0;
	cpssp->NAME.count_apb1timclk = 0;
	cpssp->NAME.count_apb2timclk = 0;
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
