/*
 * Copyright (C) 2014-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define LOG_COUNT	0

#define SETS	(CONFIG_CACHE2_ASSOC)
#define LINES	(CONFIG_CACHE2_SIZE / CONFIG_CACHE2_LINE_SIZE / SETS)

#ifdef INCLUDE

#include "glue-lru.h"

#endif /* INCLUDE */
#ifdef STATE

struct {
#if LOG_COUNT
	uint64_t miss;
#endif
#ifdef CONFIG_CACHE2_SIZE
	struct {
		struct NAME_(entry) {
			paddr_t paddr;
			int (*rcf)(void *, paddr_t, unsigned int, udata_t *);
			void *rcs;
			char *rhaddr;
			int (*wcf)(void *, paddr_t, unsigned int, udata_t);
			void *wcs;
			char *whaddr;
			int (*xcf)(void *, paddr_t, unsigned int, udata_t *);
			void *xcs;
			char *xhaddr;
		} set[SETS];
		LRU_DECL(SETS, lru);
	} line[LINES];
#endif /* CONFIG_CACHE2_SIZE */
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(mr)(struct cpssp *cpssp, paddr_t paddr, unsigned int bs, udata_t *valp);
/*forward*/ static void
NAME_(mw)(struct cpssp *cpssp, paddr_t paddr, unsigned int bs, udata_t val);
/*forward*/ static void
NAME_(mx)(struct cpssp *cpssp, paddr_t paddr, unsigned int bs, udata_t *valp);
/*forward*/ static void
NAME_(map_r)(struct cpssp *cpssp, paddr_t paddr,
		int (**cfp)(void *, paddr_t, unsigned int, udata_t *), void **csp, char **haddrp);
/*forward*/ static void
NAME_(map_w)(struct cpssp *cpssp, paddr_t paddr,
		int (**cfp)(void *, paddr_t, unsigned int, udata_t), void **csp, char **haddrp);
/*forward*/ static void
NAME_(map_x)(struct cpssp *cpssp, paddr_t paddr,
		int (**cfp)(void *, paddr_t, unsigned int, udata_t *), void **csp, char **haddrp);
/*forward*/ static void
NAME_(unmap)(struct cpssp *cpssp, paddr_t paddr, paddr_t len);
/*forward*/ static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

#ifdef CONFIG_CACHE2_SIZE

static struct NAME_(entry) *
NAME_(entry)(struct cpssp *cpssp, paddr_t paddr)
{
	int line;
	int set;

	line = (paddr / CONFIG_CACHE2_LINE_SIZE) % LINES;
	for (set = 0; ; set++) {
		struct NAME_(entry) *entry;

		if (set == SETS) {
			/* Miss */
#if LOG_COUNT
			cpssp->NAME.miss++;
#endif
			set = lru_oldest(SETS, cpssp->NAME.line[line].lru);
			lru_use(SETS, cpssp->NAME.line[line].lru, set);
			entry = &cpssp->NAME.line[line].set[set];
			entry->paddr = paddr;
			entry->xhaddr = NULL;
			entry->whaddr = NULL;
			entry->rhaddr = NULL;
			entry->xcf = NULL;
			entry->wcf = NULL;
			entry->rcf = NULL;
			return entry;
		}
		entry = &cpssp->NAME.line[line].set[set];
		if (entry->paddr == paddr) {
			/* Hit */
			lru_use(SETS, cpssp->NAME.line[line].lru, set);
			return entry;
		}
	}
}

#endif /* CONFIG_CACHE2_SIZE */

static void
NAME_(map_r)(
	struct cpssp *cpssp,
	paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t *),
	void **csp,
	char **haddrp
)
{
#ifdef CONFIG_CACHE2_SIZE
	struct NAME_(entry) *entry;

	entry = NAME_(entry)(cpssp, paddr);

	if (unlikely(! entry->rcf && ! entry->rhaddr)) {
		NAME_(apic_map_r)(cpssp, paddr, &entry->rcf, &entry->rcs, &entry->rhaddr);
	}

	*cfp = entry->rcf;
	*csp = entry->rcs;
	*haddrp = entry->rhaddr;

#else /* ! CONFIG_CACHE2_SIZE */
	NAME_(apic_map_r)(cpssp, paddr, cfp, csp, haddrp);
#endif /* ! CONFIG_CACHE2_SIZE */
}

static void
NAME_(map_w)(
	struct cpssp *cpssp,
	paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t),
	void **csp,
	char **haddrp
)
{
#ifdef CONFIG_CACHE2_SIZE
	struct NAME_(entry) *entry;

	entry = NAME_(entry)(cpssp, paddr);

	if (unlikely(! entry->wcf && ! entry->whaddr)) {
		NAME_(apic_map_w)(cpssp, paddr, &entry->wcf, &entry->wcs, &entry->whaddr);
	}

	*cfp = entry->wcf;
	*csp = entry->wcs;
	*haddrp = entry->whaddr;

#else /* ! CONFIG_CACHE2_SIZE */
	NAME_(apic_map_w)(cpssp, paddr, cfp, csp, haddrp);
#endif /* ! CONFIG_CACHE2_SIZE */
}

static void
NAME_(map_x)(
	struct cpssp *cpssp,
	paddr_t paddr,
	int (**cfp)(void *, paddr_t, unsigned int, udata_t *),
	void **csp,
	char **haddrp
)
{
#ifdef CONFIG_CACHE2_SIZE
	struct NAME_(entry) *entry;

	entry = NAME_(entry)(cpssp, paddr);

	if (unlikely(! entry->xcf && ! entry->xhaddr)) {
		NAME_(apic_map_x)(cpssp, paddr, &entry->xcf, &entry->xcs, &entry->xhaddr);
	}

	*cfp = entry->xcf;
	*csp = entry->xcs;
	*haddrp = entry->xhaddr;

#else /* ! CONFIG_CACHE2_SIZE */
	NAME_(apic_map_x)(cpssp, paddr, cfp, csp, haddrp);
#endif /* ! CONFIG_CACHE2_SIZE */
}

static void
NAME_(mr)(struct cpssp *cpssp, paddr_t paddr, unsigned int bs, udata_t *valp)
{
	NAME_(apic_mr)(cpssp, paddr, bs, valp);
}

static void
NAME_(mw)(struct cpssp *cpssp, paddr_t paddr, unsigned int bs, udata_t val)
{
	NAME_(apic_mw)(cpssp, paddr, bs, val);
}

static void
NAME_(mx)(struct cpssp *cpssp, paddr_t paddr, unsigned int bs, udata_t *valp)
{
	NAME_(apic_mx)(cpssp, paddr, bs, valp);
}

static void
NAME_(unmap)(struct cpssp *cpssp, paddr_t paddr, paddr_t len)
{
#ifdef CONFIG_CACHE2_SIZE
	int line;
	int set;
	struct NAME_(entry) *entry;

	len--;
	for (line = 0; line < LINES; line++) {
		for (set = 0; set < SETS; set++) {
			entry = &cpssp->NAME.line[line].set[set];
			if (paddr <= entry->paddr
			 && entry->paddr < paddr + len) {
				entry->xcf = NULL;
				entry->xhaddr = NULL;
				entry->wcf = NULL;
				entry->whaddr = NULL;
				entry->rcf = NULL;
				entry->rhaddr = NULL;
			}
		}
	}
#endif /* CONFIG_CACHE2_SIZE */

	NAME_(mmu_unmap)(cpssp, paddr, len);
}

static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int n_val)
{
#ifdef CONFIG_CACHE2_SIZE
	int line;
	int set;

	for (line = 0; line < LINES; line++) {
		for (set = 0; set < SETS; set++) {
			cpssp->NAME.line[line].set[set].paddr = (paddr_t) -1; /* Invalid */
		}
		lru_reset(SETS, cpssp->NAME.line[line].lru);
	}
#endif /* CONFIG_CACHE2_SIZE */
}

static void
NAME_(create)(struct cpssp *cpssp)
{
#if LOG_COUNT
	cpssp->NAME.miss = 0;
#endif
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
#if LOG_COUNT
	fprintf(stderr, "Miss: %llu\n", cpssp->NAME.miss);
#endif
}

#endif /* BEHAVIOR */

#undef LINES
#undef SETS

#undef LOG_COUNT
