//                                               -*- C++ -*-
/**
 *  @file  FunctionalChaosAlgorithm.hxx
 *  @brief The class building chaos expansions
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 17:44:02 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.hxx 818 2008-05-21 15:44:02Z dutka $
 */
#ifndef OPENTURNS_FUNCTIONALCHAOSALGORITHM_HXX
#define OPENTURNS_FUNCTIONALCHAOSALGORITHM_HXX

#include "MetaModelResult.hxx"
#include "MetaModelAlgorithm.hxx"
#include "Indices.hxx"
#include "NumericalPoint.hxx"
#include "NumericalMathFunction.hxx"
#include "Distribution.hxx"
#include "AdaptiveStrategy.hxx"
#include "ProjectionStrategy.hxx"
#include "FunctionalChaosResult.hxx"

BEGIN_NAMESPACE_OPENTURNS



/**
 * @class FunctionalChaosAlgorithm
 *
 * The class building chaos expansions
 */

class FunctionalChaosAlgorithm
  : public MetaModelAlgorithm
{
  CLASSNAME;

public:


  /** Constructor */
  FunctionalChaosAlgorithm(const NumericalMathFunction & model,
                           const Distribution & distribution,
                           const AdaptiveStrategy & adaptiveStrategy,
                           const ProjectionStrategy & projectionStrategy);

  /** Constructor */
  FunctionalChaosAlgorithm(const NumericalSample & inputSample,
                           const NumericalSample & outputSample,
                           const Distribution & distribution,
                           const AdaptiveStrategy & adaptiveStrategy,
                           const ProjectionStrategy & projectionStrategy);

  /** Constructor */
  FunctionalChaosAlgorithm(const NumericalSample & inputSample,
                           const NumericalPoint & weights,
                           const NumericalSample & outputSample,
                           const Distribution & distribution,
                           const AdaptiveStrategy & adaptiveStrategy,
                           const ProjectionStrategy & projectionStrategy);

  /** Constructor */
  FunctionalChaosAlgorithm(const NumericalMathFunction & model,
                           const Distribution & distribution,
                           const AdaptiveStrategy & adaptiveStrategy);

  /** Constructor */
  FunctionalChaosAlgorithm(const NumericalSample & inputSample,
                           const NumericalSample & outputSample,
                           const Distribution & distribution,
                           const AdaptiveStrategy & adaptiveStrategy);

  /** Constructor */
  FunctionalChaosAlgorithm(const NumericalSample & inputSample,
                           const NumericalPoint & weights,
                           const NumericalSample & outputSample,
                           const Distribution & distribution,
                           const AdaptiveStrategy & adaptiveStrategy);

  /** Virtual constructor */
  virtual FunctionalChaosAlgorithm * clone() const;

  /** String converter */
  virtual String __repr__() const;

  /** Maximum residual accessors */
  void setMaximumResidual(NumericalScalar residual);
  NumericalScalar getMaximumResidual() const;

  /** Projection strategy accessor */
  void setProjectionStrategy(const ProjectionStrategy & projectionStrategy);
  ProjectionStrategy getProjectionStrategy() const;

  /** Computes the functional chaos */
  void run();

  /** Get the functional chaos result */
  FunctionalChaosResult getResult() const;

  /** Method save() stores the object through the StorageManager */
  virtual void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  virtual void load(Advocate & adv);


protected:

  friend class Factory<FunctionalChaosAlgorithm>;

  /** Default constructor */
  FunctionalChaosAlgorithm();

private:

  /** Marginal computation */
  void runMarginal(const UnsignedLong marginalIndex,
                   Indices & indices,
                   NumericalPoint & coefficients,
                   NumericalScalar & residual,
                   NumericalScalar & relativeError);

  /** The isoprobabilistic transformation maps the distribution into the orthogonal measure */
  NumericalMathFunction transformation_;

  /** The inverse isoprobabilistic transformation */
  NumericalMathFunction inverseTransformation_;

  /** The composed model */
  NumericalMathFunction composedModel_;

  /** The adaptive strategy */
  AdaptiveStrategy adaptiveStrategy_;

  /** The projection strategy */
  ProjectionStrategy projectionStrategy_;

  /** Maximum residual */
  NumericalScalar maximumResidual_;

  /** Result of the projection */
  FunctionalChaosResult result_;

} ; /* class FunctionalChaosAlgorithm */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_FUNCTIONALCHAOSALGORITHM_HXX */
