/*
**  MimeUtility.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**          Alexander Malmberg <alexander@malmberg.org>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/MimeUtility.h>

#import <Pantomime/Constants.h>
#import <Pantomime/ISO8859_1.h>
#import <Pantomime/ISO8859_2.h>
#import <Pantomime/ISO8859_3.h>
#import <Pantomime/ISO8859_4.h>
#import <Pantomime/ISO8859_5.h>
#import <Pantomime/ISO8859_6.h>
#import <Pantomime/ISO8859_7.h>
#import <Pantomime/ISO8859_8.h>
#import <Pantomime/ISO8859_9.h>
#import <Pantomime/ISO8859_10.h>
#import <Pantomime/ISO8859_11.h>
#import <Pantomime/ISO8859_13.h>
#import <Pantomime/ISO8859_14.h>
#import <Pantomime/ISO8859_15.h>
#import <Pantomime/UTF8.h>
#import <Pantomime/KOI8_R.h>
#import <Pantomime/KOI8_U.h>
#import <Pantomime/WINDOWS_1251.h>
#import <Pantomime/WINDOWS_1252.h>


#import <Pantomime/Message.h>
#import <Pantomime/MimeBodyPart.h>
#import <Pantomime/MimeMultipart.h>
#import <Pantomime/NSStringExtensions.h>
#import <Pantomime/NSDataExtensions.h>
#import <Pantomime/Part.h>
#import <Pantomime/GSMD5.h>

#import <netdb.h>
#import <unistd.h>
#import <string.h>
#import <time.h>

#define UUDECODE(c)  (((c) - ' ') & 077)

int getValue(char c);
void nb64ChunkFor3Characters(char *buf, const char *inBuf, int numChars);
void uudecodeline(char *line, NSMutableData *data);

static const char *hexDigit = "0123456789ABCDEF";
static char basis_64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

static NSMutableDictionary *charsets = nil;


@implementation MimeUtility


//
//
//
+ (void) initialize
{
  if (! charsets )
    {
      NSLog(@"Initializing the Charset cache.");
      charsets = [[NSMutableDictionary alloc] init];
    }
}


//
// This method is used to return the string value of a
// specified encoding. If the enconding isn't found,
// it simply returns the default encoding: 7bit.
//
+ (NSString *) stringValueOfTransferEncoding: (int) theEncoding
{
  switch (theEncoding) {

  case 0:
    break;
    
  case 1: return @"quoted-printable";

  case 2: return @"base64";
    
  case 3: return @"8bit";

  case 4: return @"binary";

  default:
    break;
  }
  
  return @"7bit";
}



//
// This method is used to obtain a charset from the name
// of this charset. It caches this charset for future
// usage when it's found.
//
+ (Charset *) charsetForName: (NSString *) theName
{
  Charset *theCharset;

  theCharset = [charsets objectForKey: [theName lowercaseString]];

  if (! theCharset )
    {
      Charset *aCharset;

      if ( [[theName lowercaseString] isEqualToString: @"iso-8859-2"] )
	{
	  aCharset = (Charset *)[[ISO8859_2 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-3"] )
	{
	  aCharset = (Charset *)[[ISO8859_3 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-4"] )
	{
	  aCharset = (Charset *)[[ISO8859_4 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-5"] )
	{
	  aCharset = (Charset *)[[ISO8859_5 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-6"] )
	{
	  aCharset = (Charset *)[[ISO8859_6 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-7"] )
	{
	  aCharset = (Charset *)[[ISO8859_7 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-8"] )
	{
	  aCharset = (Charset *)[[ISO8859_8 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-9"] )
	{
	  aCharset = (Charset *)[[ISO8859_9 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-10"] )
	{
	  aCharset = (Charset *)[[ISO8859_10 alloc] init];
	  AUTORELEASE(aCharset);
 	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-11"] )
	{
	  aCharset = (Charset *)[[ISO8859_11 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-13"] )
	{
	  aCharset = (Charset *)[[ISO8859_13 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-14"] )
	{
	  aCharset = (Charset *)[[ISO8859_14 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"iso-8859-15"] )
	{
	  aCharset = (Charset *)[[ISO8859_15 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"koi8-r"] )
	{
	  aCharset = (Charset *)[[KOI8_R alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"koi8-u"] )
	{
	  aCharset = (Charset *)[[KOI8_U alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"windows-1251"] )
	{
	  aCharset = (Charset *)[[WINDOWS_1251 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else if ( [[theName lowercaseString] isEqualToString: @"windows-1252"] )
	{
	  aCharset = (Charset *)[[WINDOWS_1252 alloc] init];
	  AUTORELEASE(aCharset);
	}
      else
	{
	  aCharset = (Charset *)[[ISO8859_1 alloc] init];
	  AUTORELEASE(aCharset);
	}
      
      [charsets setObject: aCharset
		forKey: [theName lowercaseString]];
      
      return aCharset;
    }

  return theCharset;
}



+ (int) stringEncodingForCharset: (NSData *) charset
{
  
  // We define some aliases for the string encoding.
  // FIXME, this won't work under OSX right now.
  static struct { NSString *name; int encoding; } encodings[] = {
    {@"ascii"      ,NSASCIIStringEncoding},
    {@"us-ascii"   ,NSASCIIStringEncoding},
    {@"utf-8"      ,NSUTF8StringEncoding},
    {@"iso-8859-1" ,NSISOLatin1StringEncoding},
    {@"iso-8859-2" ,NSISOLatin2StringEncoding},
#ifdef MACOSX
    {@"iso-8859-3" ,NSISOLatin1StringEncoding},
    {@"iso-8859-4" ,NSISOLatin1StringEncoding},
    {@"iso-8859-5" ,NSISOLatin1StringEncoding},
    {@"iso-8859-6" ,NSISOLatin1StringEncoding},
    {@"iso-8859-7" ,NSISOLatin1StringEncoding},
    {@"iso-8859-8" ,NSISOLatin1StringEncoding},
    {@"iso-8859-9" ,NSISOLatin1StringEncoding},
    {@"iso-8859-10",NSISOLatin1StringEncoding},
    // {@"iso-8859-11",???},
    // {@"iso-8859-12",???},
    {@"iso-8859-13",NSISOLatin1StringEncoding},
    {@"iso-8859-14",NSISOLatin1StringEncoding},
    {@"iso-8859-15",NSISOLatin1StringEncoding},
    {@"koi8-r"     ,NSISOLatin1StringEncoding},
    // {@"koi8-u"     ,???},      
#else
    {@"iso-8859-3" ,NSISOLatin3StringEncoding},
    {@"iso-8859-4" ,NSISOLatin4StringEncoding},
    {@"iso-8859-5" ,NSISOCyrillicStringEncoding},
    {@"iso-8859-6" ,NSISOArabicStringEncoding},
    {@"iso-8859-7" ,NSISOGreekStringEncoding},
    {@"iso-8859-8" ,NSISOHebrewStringEncoding},
    {@"iso-8859-9" ,NSISOLatin5StringEncoding},
    {@"iso-8859-10",NSISOLatin6StringEncoding},
    // {@"iso-8859-11",???},
    // {@"iso-8859-12",???},
    {@"iso-8859-13",NSISOLatin7StringEncoding},
    {@"iso-8859-14",NSISOLatin8StringEncoding},
    {@"iso-8859-15",NSISOLatin9StringEncoding},
    {@"koi8-r"     ,NSKOI8RStringEncoding},
    // {@"koi8-u"     ,???},
#endif
    {@"windows-1250",NSWindowsCP1250StringEncoding},
    {@"windows-1251",NSWindowsCP1251StringEncoding},
    {@"windows-1252",NSWindowsCP1252StringEncoding},
    {@"windows-1253",NSWindowsCP1253StringEncoding},
    {@"windows-1254",NSWindowsCP1254StringEncoding},
  };
  
  NSString *name;
  int i;

  name = [[NSString stringWithCString: [charset bytes] length: [charset length]] lowercaseString];
  
  for ( i = 0; i < sizeof(encodings)/sizeof(encodings[0]); i++)
    {
      if ( [name isEqualToString: encodings[i].name] )
	{
	  return encodings[i].encoding;
	}
    }

  return -1;
}


+ (NSString *) stringWithData: (NSData *) theData
                      charset: (NSData *) theCharset
{
  int encoding;

  if (theData == nil)
    {
      return nil;
    }

  encoding = [self stringEncodingForCharset: theCharset];
  
  if (encoding == -1)
    {
      return nil;
    }
  
  return AUTORELEASE( [[NSString alloc] initWithData: theData
					encoding: encoding] );
}


//
// See RFC2047.
// It supports:
// 
// Abcd =?ISO-8859-1?Q?=E8fgh?=
// =?iso-8859-1?Q?ab=E7de?= =?iso-8859-1?Q?_?= =?iso-8859-1?Q?oo=F4oo?=
// Abd =?ISO-8859-1?Q?=E8?= fghijklmn
//
+ (NSString *) decodeHeader: (NSData *) theData
{
  NSMutableString *aMutableString;
  NSData *charset, *encoded_text;

  NSString *aString;
  
  int i, length, start, i_charset, i_encoding, end; 
  const unsigned char *bytes;
 
  BOOL ignore_span;
  char encoding;

  // We first verify for a nil value
  if (theData == nil)
    {
      return @"";
    }
  
  length = [theData length];
  bytes = [theData bytes];
  
  aMutableString = [[NSMutableString alloc] initWithCapacity: length];
  
  start = i = 0;
  ignore_span = NO;
  
  while (i < (length - 1))
    {
      if ( bytes[i] != '=' || bytes[i+1] != '?')
	{
	  if ( bytes[i] > 32 )
	    {
	      ignore_span = NO;
	    }
	  
	  i++;
	  continue;
	}
      
      if (i != start && !ignore_span)
	{
	  aString = [[NSString alloc] initWithCString: bytes+start
				      length: i-start];
	  [aMutableString appendString: aString];
	  DESTROY(aString);
	}
      
      start = i;
      
      // skip the =? and one character (or it isn't valid)
      i += 3; 
      while ( i < length && bytes[i] != '?') 
	{
	  i++;
	}
      
      if ( i == length) 
	{
	  break;
	}
      
      i_charset = i;
      
      /* encoding is a single character */
      if (i+2 >= length)
	{
	  break;
	}
      
      encoding = bytes[i+1];
      
      if (bytes[i+2] != '?')
	{
	  break;
	}
      
      i += 3;
      i_encoding = i;
      
      while ( i <length && bytes[i] != '?')
	{
	  i++;
	}
      
      if (i+1 >= length)
	{
	  break;
	}
      
      if (bytes[i+1] != '=')
	{
	  break;
	}

      end = i;
      i += 2;
      
      charset = [theData subdataWithRange: NSMakeRange(start+2,i_charset-start-2)];
      encoded_text = [theData subdataWithRange: NSMakeRange(i_encoding,end-i_encoding)];
      
      if ( encoding == 'q' || encoding == 'Q')
	{
	  aString= [self stringWithData: [self decodeQuotedPrintable: encoded_text  inHeader: YES]
			 charset: charset];
	}
      else if ( encoding == 'b' || encoding== 'B' )
	{
	  aString = [self stringWithData: [self decodeBase64: encoded_text]
			  charset: charset];
	}
      else
	{
	  continue;
	}

      if (!aString)
	{
	  continue;
	}
      
      [aMutableString appendString: aString];
      aString = nil;
      start = i;
      ignore_span = YES;
    }
  
  i = length;
  
  if (i != start && !ignore_span)
    {
      aString = [[NSString alloc] initWithCString: bytes+start  length: i-start];
      [aMutableString appendString: aString];
      DESTROY(aString);
    }
  
  return AUTORELEASE(aMutableString);
}


//
// This method is used to decode a string containing characters encoded
// using the Quoted Printable method with the charset "theCharset".
//
+ (NSData *) decodeQuotedPrintable: (NSData *) theData  inHeader: (BOOL)header
{
  NSMutableData *result;
  int i,len;
  const unsigned char *bytes,*b;
  unsigned char ch;

  len = [theData length];
  bytes = [theData bytes];

  result = [[NSMutableData alloc] initWithCapacity: len];
  
  b=bytes;

  for (i = 0; i < len; i++,b++)
    {
      if (b[0]=='=' && i+1<len && b[1]=='\n')
	{
	  b++,i++;
	  continue;
	}
      else if (*b=='=' && i+2<len)
	{
	  b++,i++;
	  if (*b>='A' && *b<='F')
	    {
	      ch=16*(*b-'A'+10);
	    }
	  else if (*b>='a' && *b<='f')
	    {
	      ch=16*(*b-'a'+10);
	    }
	  else if (*b>='0' && *b<='9')
	    {
	      ch=16*(*b-'0');
	    }

	  b++,i++;

	  if (*b>='A' && *b<='F')
	    {
	      ch+=*b-'A'+10;
	    }
	  else if (*b>='a' && *b<='f')
	    {
	      ch+=*b-'a'+10;
	    }
	  else if (*b>='0' && *b<='9')
	    {
	      ch+=*b-'0';
	    }
	  
	  [result appendBytes: &ch length: 1];
	}
      else if (header && *b=='_')
	{
	  ch=0x20;
	  [result appendBytes: &ch length: 1];
	}
      else
	{
	  [result appendBytes: b length: 1];
	}
    }

  return AUTORELEASE(result);
}


//
// This method is used to decode a string that has been encoded
// using the base64 method and returns and array of bytes (NSData)
// corresponding to the decoded data.
//
// FIXME: this should ignore characters in the stream that aren't in
//        the base64 alphabet (as per the spec). would remove need for
//        ...removeLinefeeds... too
//
+ (NSData *) decodeBase64: (NSData *) theData
{
  int i, j, length, rawIndex, block, pad = 0;
  char *raw;

  const unsigned char *bytes;
  int data_len;

  NSData *result;

  if (!theData)
  	return nil;

  bytes = [theData bytes];
  data_len = [theData length];

  for (i = data_len - 1; bytes[i] == '='; i--)
    {
      pad++;
    }
  
  length = data_len * 6 / 8 - pad;
  
  raw = (char*)malloc(length);
  rawIndex = 0;

  for (i = 0; i < data_len; i += 4)
    {
      block = (getValue(bytes[i]) << 18) +
	(getValue(bytes[i + 1]) << 12) +
	(getValue(bytes[i + 2]) << 6) +
	(getValue(bytes[i + 3]));

      for (j = 0; j < 3 && rawIndex + j < length; j++)
	{
	  raw[rawIndex + j] = (char)((block >> (8 * (2 - j))) & 0xff);
	}

      rawIndex += 3;
    }

  result = [[NSData alloc] initWithBytesNoCopy:raw length:length];

  return AUTORELEASE(result);
}


//
//
//
+ (NSData *) unfoldLinesFromData: (NSData *) theData
{
  NSMutableData *aMutableData;
  int i, length;
  
  const unsigned char *bytes, *b;
  
  if ( !theData )
    {
      return nil;
    }
  
  length = [theData length];
  b = bytes = [theData bytes];
  
  aMutableData = [[NSMutableData alloc] initWithCapacity: length];
  
  [aMutableData appendBytes: b 
		length: 1];
  
  b++;
  
  for ( i = 1; i < length; i++,b++)
    {
      if ( b[-1]=='\n' && (*b==' ' || *b=='\t') )
	{
	  [aMutableData setLength: ([aMutableData length] - 1)];
	}
      
      [aMutableData appendBytes: b length: 1];
    }
  
  return AUTORELEASE(aMutableData);
}


//
//
//
+ (NSData *) encodeBase64: (NSData *) theData
               lineLength: (int) numChars
{
  NSData *result;
  const char *inBytes = [theData bytes];
  const char *inBytesPtr = inBytes;
  int inLength = [theData length];

  char *outBytes = malloc(sizeof(char)*inLength*2);
  char *outBytesPtr = outBytes;

  int numWordsPerLine = numChars/4;
  int wordCounter = 0;

  // We bzero our buffer so with are sure to not have
  // any garbage in it.
  bzero(outBytes, sizeof(char)*inLength*2);

  while (inLength > 0)
    {
      nb64ChunkFor3Characters(outBytesPtr, inBytesPtr, inLength);
      outBytesPtr += 4;
      inBytesPtr += 3;
      inLength -= 3;

      wordCounter ++;

      if (numChars && wordCounter == numWordsPerLine)
	{
	  wordCounter = 0;
	  *outBytesPtr = '\n';
	  outBytesPtr++;
	}
    }

  result = [[NSData alloc] initWithBytesNoCopy: outBytes  length: outBytesPtr-outBytes];

  return AUTORELEASE(result);
}


//
//
//
+ (NSData *) encodeQuotedPrintable: (NSData *) theData
                        lineLength: (int) numChars
			  inHeader: (BOOL) header
{
  NSMutableData *aMutableData;
  const unsigned char *b;
  int i, length, line;
  char buf[4];
  
  aMutableData = [[NSMutableData alloc] initWithCapacity: [theData length]];
  b = [theData bytes];
  length = [theData length];

  buf[3] = 0;
  buf[0] = '=';
  line = 0;

  for ( i = 0; i < length; i++, b++ )
    {
      if (numChars && line >= numChars)
	{
	  [aMutableData appendBytes: "=\n" length: 2];
	  line = 0;
	}
      // RFC says must encode space and tab right before end of line
      if ( (*b == ' ' || *b == '\t') && i < length - 1 && b[1] == '\n')
	{
	  buf[1] = hexDigit[(*b)>>4];
	  buf[2] = hexDigit[(*b)&15];
	  [aMutableData appendBytes: buf 
			length: 3];
	  line += 3;
	}
      // FIXME: really always pass \n through here?
      else if (!header &&
	       (*b == '\n' || *b == ' ' || *b == '\t'
		|| (*b >= 33 && *b <= 60)
		|| (*b >= 62 && *b <= 126)))
	{
	  [aMutableData appendBytes: b
			length: 1];
	  if (*b == '\n')
	    {
	      line = 0;
	    }
	  else
	    {
	      line++;
	    }
	}
      else if (header && ((*b >= 'a' && *b <= 'z') || (*b >= 'A' && *b <= 'Z')))
	{
	  [aMutableData appendBytes: b
			length: 1];
	  if (*b == '\n')
	    {
	      line = 0;
	    }
	  else
	    {
	      line++;
	    }
	}
      else if (header && *b==' ')
	{
	  [aMutableData appendBytes: "_"
			length: 1];
	}
      else
	{
	  buf[1] = hexDigit[(*b)>>4];
	  buf[2] = hexDigit[(*b)&15];
	  [aMutableData appendBytes: buf
			length: 3];
	  line += 3;
	}
    }
  
  return AUTORELEASE(aMutableData);
}



//
// This method is used to generate a MIME boundary (or any kind of boundary)
//
+ (NSData *) generateBoundary
{
  NSMutableData *aMutableData;
  GSMD5 *md5;
  
  char random_data[8];
  time_t curtime;
  int i, pid;
  
  pid = getpid();
  time(&curtime);
  
  for (i = 0; i < sizeof(random_data); i++)
    {
      random_data[i] = hexDigit[random()&0xff];
    }
  
  md5 = [[GSMD5 alloc] init];
  
  [md5 updateWithString: [NSString stringWithFormat: @"%d.%d", pid, curtime]
       usingEncoding: NSASCIIStringEncoding];
  
  [md5 updateWithData: [NSData dataWithBytes: random_data  length: sizeof(random_data)]];
  [md5 calculateDigest];
  
  aMutableData = [[NSMutableData alloc] init];
  [aMutableData appendBytes: "=_"
		length: 2];
  [aMutableData appendCFormat: @"%@", DataToHexString([md5 digest])];
  
  DESTROY(md5);

  return AUTORELEASE(aMutableData);
}


//
// This method is used to generate a unique ID that Messages (or Mime Body Parts)
// use as their unique ID. (Message-ID, Content-ID, etc.) 
//
+ (NSData *) generateOSID
{
  NSMutableData *aMutableData;
  GSMD5 *md5;

  char random_data[8];
  time_t curtime;
  int i, pid;

  pid = getpid();
  time(&curtime);
  
  for (i = 0; i < sizeof(random_data); i++)
    {
      random_data[i] = hexDigit[random()&0xff];
    }
  
  md5 = [[GSMD5 alloc] init];
  
  [md5 updateWithString: [NSString stringWithFormat: @"%d.%d", pid, curtime]
       usingEncoding: NSASCIIStringEncoding];
  
  [md5 updateWithData: [NSData dataWithBytes: random_data  length: sizeof(random_data)]];
  [md5 calculateDigest];
  
  aMutableData = [[NSMutableData alloc] init];
  [aMutableData appendCFormat: @"%@", DataToHexString([md5 digest])];
  [aMutableData appendCFormat: @"@%@", [[NSHost currentHost] name]];
  
  DESTROY(md5);

  return AUTORELEASE(aMutableData);
}





//
// This method is used to encode a text string using quoted-printable
// or base64. 
//
//
+ (NSData *) encodeHeader: (NSString *) theText
{
  
  // Initial verification
  if (!theText || [theText length] == 0)
    {
      return [NSData data];
    } 

  // If it's not an ASCII string, we encode it!
  if (! [MimeUtility isASCIIString: theText] )
    {

      NSString *aCharset;

      aCharset = [MimeUtility charsetForString: theText];
      
      return [MimeUtility encodeHeader: theText
			  usingCharset: aCharset
			  encoding: QUOTEDPRINTABLE];
    }
  else
    {
      return [theText dataUsingEncoding: NSASCIIStringEncoding];
    }
}


//
// FIXME: Should we really use NSUTF8StringEncoding in base64?
//
+ (NSData *) encodeHeader: (NSString *) theText
	     usingCharset: (NSString *) theCharset
		 encoding: (int) encoding
{
  NSData *aData;
  
  // Initial verification
  if (!theText || [theText length] == 0)
    {
      return [NSData data];
    }
  
  aData = [theText dataUsingEncoding: [MimeUtility stringEncodingForCharset:
						 [theCharset dataUsingEncoding: NSASCIIStringEncoding]]];
  
  if (encoding == QUOTEDPRINTABLE)
    {
      return [MimeUtility encodeQuotedPrintable: aData
			  lineLength: 0
			  inHeader: YES];
    }
  else if (encoding == BASE64)
    {
      return [MimeUtility encodeBase64: aData
			  lineLength: 0];
    }
  // FIXME: What should we do here, should we just return the 'aData' w/o 
  //        encoding it or should we generate an exception?
  else
    {
      return aData; 
    }
}


//
// The format returned is for example:
//
// =?ISO-8859-1?B?....?=
// 
// This format is known as an encoded-word defined in RFC2047.
// If the word doesn't need to be encoded, it's just returned a
//
// FIXME: We should verify that the length doesn't exceed 75 chars.
//
+ (NSData *) encodeWordUsingBase64: (NSString *) theWord
		      prefixLength: (int) thePrefixLength
{
  // Initial verification
  if (!theWord || [theWord length] == 0)
    {
      return [NSData data];
    }
  else if ( [MimeUtility isASCIIString: theWord] )
    {
      return [theWord dataUsingEncoding: NSASCIIStringEncoding];
    }
  else
    {
      NSMutableData *aMutableData;
      NSString *aCharset;
      
      aMutableData = [[NSMutableString alloc] init];
      aCharset = [MimeUtility charsetForString: theWord];
      
      [aMutableData appendCFormat: @"=?%@?b?", aCharset];
      [aMutableData appendData: [MimeUtility encodeHeader: theWord
						 usingCharset: aCharset
						 encoding: BASE64]];
      [aMutableData appendCString: "?="];

      return AUTORELEASE(aMutableData);
    }

  // Never reached.
  return nil;
}


//
// The format returned is for example:
//
// =?ISO-8859-1?Q?=E8fgh?=
//
// This format is known as an encoded-word defined in RFC2047.
// If the word doesn't need to be encoded, it's just returned as it.
//
// The string returned will NOT be more than 75 characters long
// for each folded part of the original string.
//
+ (NSData *) encodeWordUsingQuotedPrintable: (NSString *) theWord
			       prefixLength: (int) thePrefixLength
{  
  NSMutableString *aMutableString;
  NSMutableArray *aMutableArray;
  NSScanner *aScanner;
  int previousLocation, currentLocation;

  BOOL mustUseEncoding;
  NSString *aCharset;

  // Initial verification
  if ( !theWord || 
       [theWord length] == 0 )
    {
      return [NSData data];
    } 

  // If it's not an ASCII string, we must use the encoding!
  mustUseEncoding = (! [MimeUtility isASCIIString: theWord]);

  aCharset = nil;

  if ( mustUseEncoding )
    {
      aCharset = [MimeUtility charsetForString: theWord];
    }
  
  aMutableString = [[NSMutableString alloc] init];
  
  aMutableArray = [[NSMutableArray alloc] init];
  AUTORELEASE(aMutableArray);
  
  // We initialize our scanner with the content of our word
  aScanner = [[NSScanner alloc] initWithString: theWord];
  
  currentLocation = previousLocation = 0;
  
  while ( [aScanner scanUpToCharactersFromSet: [NSCharacterSet whitespaceCharacterSet]
		    intoString: NULL] )
    {
      NSString *aString;
      int length;
      
      currentLocation = [aScanner scanLocation]; 
	  
      // aString contains the substring WITH the spaces present BEFORE the word and AFTER the last aString.
      // 
      // For example, the result of "this is a test" will be
      //
      // aString = "this"
      // aString = " is"
      // aString = " a"
      // aString = " test"
      //
      aString = [theWord substringWithRange: NSMakeRange(previousLocation, currentLocation - previousLocation)];

      if ( mustUseEncoding  )
	{
	  // Our 'initial' lenght contains =?iso-8859-x?q? and ?=
	  length = 18;
	  length += [[MimeUtility encodeHeader: [NSString stringWithFormat: @"%@%@", aMutableString, aString]
				  usingCharset: aCharset
				  encoding: QUOTEDPRINTABLE] length];
	}
      else
	{
	  length = [aMutableString length] + [aString length];
	}

      // If we are on the first line, we must consider the prefix length.
      // For example, the prefix length might be the length of the string "Subject: "
      if ( [aMutableArray count] == 0 )
	{
	  length += thePrefixLength;
	}

      if ( length > 75 )
	{
	  [aMutableArray addObject: aMutableString];

	  RELEASE(aMutableString);
	  aMutableString = [[NSMutableString alloc] init];
	}
      
      [aMutableString appendString: aString];
      previousLocation = currentLocation;
    }
  
  // We add our last string to the array.
  [aMutableArray addObject: aMutableString];
  RELEASE(aMutableString);
  
  RELEASE(aScanner);
      
    {
      int i;
      NSMutableData *data;

      data = [[NSMutableData alloc] init];
      
      for (i = 0; i < [aMutableArray count]; i++)
	{
	  // We must append a space (' ') before each folded line
	  if ( i > 0 )
	    {
	      [data appendCString: " "];
	    }
	  
	    if ( mustUseEncoding )
	      {
		[data appendCFormat: @"=?%@?q?", aCharset];
		[data appendData: [MimeUtility encodeHeader: [aMutableArray objectAtIndex: i]
					       usingCharset: aCharset
					       encoding: QUOTEDPRINTABLE] ];
		[data appendCString: "?="];
	      }
	    else
	      {
		[data appendData: [[aMutableArray objectAtIndex: 0] dataUsingEncoding: NSASCIIStringEncoding]];
	      }
	  
	      // We if it is our last string, we must NOT append the \n
	  if ( !(i == ([aMutableArray count] - 1)) )
	    {
	      [data appendCString: "\n"];
	    }
	}
      
      return AUTORELEASE(data);
    }
}
  


//
// This method is used to guess which charset is used in the string.
// 
+ (NSString *) charsetForString: (NSString *) theString
{
  NSMutableArray *aMutableArray;
  Charset *aCharset;
  unsigned int i, j;
  NSString *str;

  NSAutoreleasePool *pool;

  aMutableArray = [[NSMutableArray alloc] init];
  pool = nil;

  [aMutableArray addObject: [self charsetForName: @"iso-8859-1"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-2"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-3"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-4"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-5"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-6"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-7"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-8"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-9"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-10"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-11"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-13"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-14"]];
  [aMutableArray addObject: [self charsetForName: @"iso-8859-15"]];
  [aMutableArray addObject: [self charsetForName: @"koi8-r"]];
  [aMutableArray addObject: [self charsetForName: @"koi8-u"]];
  [aMutableArray addObject: [self charsetForName: @"windows-1251"]];
  [aMutableArray addObject: [self charsetForName: @"windows-1252"]];

  for ( i = 0; i < [theString length]; i++ )
    {
      if ( (i % 100) == 0 )
	{
	  TEST_RELEASE(pool);
	  pool = [[NSAutoreleasePool alloc] init];
	} 

      for ( j = 0; j < [aMutableArray count] ; j++ )
        {
          if ( ! [[aMutableArray objectAtIndex: j] characterIsInCharset: [theString substringWithRange:
                                                                                    NSMakeRange(i, 1)]] )
            {
              // Character is not in the charset
              [aMutableArray removeObjectAtIndex: j];
              j--;
            }
        }

      // FIXME: can't break even if there is only one left. First we have to check
      //        whether that encoding will actually work for the entire string. If it
      //	doesn't we'll need to fall back to utf-8 (or something else that can encode
      //        _everything_.
      // 
      // Intelligent string splitting would help, of course
      //
      if ( [aMutableArray count] < 1 )
        {
          // We have zero or one charset
          break;
        }
    }

  if ( [aMutableArray count] )
    {
      aCharset = [aMutableArray objectAtIndex: 0];
      [aMutableArray removeAllObjects];
      str=[aCharset name];
    }
  else
    {
      // We have no charset, we return a default charset
      str = @"utf-8";
    }

  RELEASE(aMutableArray);
  TEST_RELEASE(pool);
  
  return str;
}


#if 0
//
//
//
+ (NSString *) foldStringUsingSoftLineBreaks: (NSString *) theString
				      length: (int) theLength
{
  NSMutableString *aMutableString;
  NSString *aString, *aLine;
  NSArray *allLines;
  
  NSAutoreleasePool *pool;
  int i, start, substringLength;

  // We first verify if we really need to fold
  if ( [theString length] <= theLength) 
    {
      return AUTORELEASE(RETAIN(theString));
    }

  // Yes, we do.
  aMutableString = [[NSMutableString alloc] init];
  pool = nil;

  allLines = [theString componentsSeparatedByString: @"\n"];

  for (i = 0; i < [allLines count]; i++)
    {
      if ( (i % 25) == 0 )
	{
	  TEST_RELEASE(pool);
	  pool = [[NSAutoreleasePool alloc] init];
	} 

      start = 0;
      substringLength = (theLength - 1);
      aLine = [allLines objectAtIndex: i];
	  
      while (YES)
	{
	  // We first verify if our line is smaller than our max length - 1
	  if ( [aLine length] <= (theLength - 1))
	    {
	      [aMutableString appendString: aLine];
	      goto done;
	    }
	  
	  aString = [aLine substringWithRange: NSMakeRange(start, substringLength)];

	  // We MUST verify if we're not folding in the middle of an encoded character.
	  // If yes, we fold BEFORE the character.
	  // 
	  // foofoofoo=E9 -> foofoofoo=E=\n9   IS NOT ALLOWED
	  // foofoofoo=E9 -> foofoofoo==\nE9   IS NOT ALLOWED
	  //
	  // If we have: foofoofoo=
	  if ( [aString length] > 1 &&
	       [aString characterAtIndex: (substringLength - 1)] == '=' )
	    {
	      substringLength = substringLength - 1;
	      aString = [aLine substringWithRange: NSMakeRange(start, substringLength)];
	    }
	  // If we have: foofoofoo=E
	  else if ( [aString length] > 2 &&
		    [aString characterAtIndex: (substringLength - 2)] == '=' )
	    {
	      substringLength = substringLength - 2;
	      aString = [aLine substringWithRange: NSMakeRange(start, substringLength)];
	    }
	  
	  // We append our string
	  [aMutableString appendFormat: @"%@=\n", aString];
	  
	  // We verify if we need to break our loop
	  if ( (start + substringLength) >= [aLine length] )
	    {
	      goto done;
	    }
	  
	  // We adjust our start index to the sum of the substring we read so far
	  start += [aString length];
	
	  // We adjust our substring length
	  if ( ([aLine length] - start) <= (theLength - 1))
	    {
	      substringLength = ([aLine length] - start);
	    }
	  else
	    {
	      substringLength = (theLength - 1);
	    }
	}
     
      // We are done with this line, append our \n and get the next one.
    done:
      [aMutableString appendString: @"\n"];
    
    } // for
  
  TEST_RELEASE(pool);

  // We trim our last "\n"
  [aMutableString deleteCharactersInRange: NSMakeRange([aMutableString length] - 1, 1)];
  
  return AUTORELEASE(aMutableString);
}


#endif




//
//
//
+ (BOOL) isASCIIString: (NSString *) theString
{
  NSAutoreleasePool *pool;
  BOOL result;
  unichar c;
  int i;
  
  pool = [[NSAutoreleasePool alloc] init];
  result = YES;

  // We search for a non-ascii character
  for (i = 0; i < [theString length]; i++)
    {
      c = [theString characterAtIndex: i];
      
      if ( c > 0x007E )
	{
	  result = NO;
	  break;
	}
    }

  RELEASE(pool);

  return result;
}


//
// This methods is used to break a word that is too long
// into multiple words separed by \n.
//
+ (NSString*) _breakWord: (NSString *) theWord
	      usingLimit: (int) theLimit
{
  NSMutableString *aMutableString;
  NSString *aString;
  int location, total_length;

  if ( [theWord length] <= theLimit )
    {
      return AUTORELEASE(RETAIN(theWord));
    } 

  aMutableString = [[NSMutableString alloc] initWithCapacity: [theWord length]];
  AUTORELEASE(aMutableString);

  location = total_length = 0;
  

  while (total_length < [theWord length])
    {
      // We need to verify if our limit will extends our bounds.
      // If so, we readjust it to get the last part of the string.
      if ( (location + theLimit) > [theWord length] )
	{
	  theLimit = [theWord length] - location;
	}
      
      aString = [theWord substringWithRange: NSMakeRange(location, theLimit)];

      [aMutableString appendFormat: @"%@\n", aString];
      total_length += [aString length];

      location += theLimit;
    }

  
  // We return our string by taking care of removing the last \n
  return ([aMutableString substringToIndex: [aMutableString length] - 1]);
}


//
//
//
+ (NSString *) wrapPlainTextString: (NSString *) theString
                usingWrappingLimit: (int) theLimit
{
  NSMutableString *aMutableString;
  NSScanner *aScannerSoftLineBreak;
  NSString *aLine, *aWord, *aPadding;
  NSRange aRange;
  int previousLineLocation, currentLineLocation;
  int previousWordLocation, currentWordLocation;
  int currentLineLength;
  int length;

  // We first verify if the string is valid
  if (! theString || [theString length] == 0)
    {
      return @"";
    }

  // We then verify if the limit is valid
  if (theLimit == 0 || theLimit > 998)
    {
      theLimit = 998;
    }
  
  // We initialize our local variables
  aMutableString = [[NSMutableString alloc] initWithCapacity: [theString length]];
  aScannerSoftLineBreak = [[NSScanner alloc] initWithString: theString];
  currentLineLocation = previousLineLocation = 0;
  
  while ( [aScannerSoftLineBreak scanUpToCharactersFromSet: [NSCharacterSet characterSetWithCharactersInString: @"\n"]
				 intoString: NULL] )
    {
      currentLineLocation = [aScannerSoftLineBreak scanLocation];
      aLine = [theString substringWithRange: NSMakeRange(previousLineLocation, currentLineLocation - previousLineLocation)];

      //NSLog(@"line = \"%@\"", aLine);

      // We verify if the line starts with newline(s)
      aRange = [aLine rangeOfCharacterFromSet: [[NSCharacterSet characterSetWithCharactersInString: @"\n"] invertedSet]];
      if ( aRange.location > 0 )
	{
	  [aMutableString appendString: [aLine substringToIndex: aRange.location]];
	  aLine = [aLine substringFromIndex: aRange.location];
	}

      // Space-stuff line if necessary
      if ( ([aLine length] >= 5 && [[aLine substringToIndex: 5] compare: @"From "] == NSOrderedSame) ||
	   ([aLine length] >= 1 && [aLine characterAtIndex: 0] == ' ') )
  	{
  	  aLine = [@" " stringByAppendingString: aLine];
  	}
      
      // We remove any ending whitespace
      while ( [aLine characterAtIndex: [aLine length] - 1] == ' ' )
	{
	  aLine = [aLine substringToIndex: [aLine length] - 1];
	}
      
      // We verify if we need to wrap the line
      if ( [aLine length] <= theLimit )
	{
	  [aMutableString appendString: aLine];
	}
      // Yes we do
      else
	{
	  NSScanner *aScannerWhitespace;
	  NSAutoreleasePool *pool;

	  aScannerWhitespace = [[NSScanner alloc] initWithString: aLine];
	  currentWordLocation = previousWordLocation = 0;
	  currentLineLength = 0;
	  
	  // We create our local autorelease pool
	  pool = [[NSAutoreleasePool alloc] init];

	  while ( [aScannerWhitespace scanUpToCharactersFromSet: [NSCharacterSet whitespaceCharacterSet]
				      intoString: NULL] )
            {
	      currentWordLocation = [aScannerWhitespace scanLocation];
	      aWord = [aLine substringWithRange: NSMakeRange(previousWordLocation, currentWordLocation - previousWordLocation)];
	      
	      //NSLog(@"word = \"%@\"\t(current line length = %d)", aWord, currentLineLength);

	      // We verify if we can fit the word and the leading whitespace(s) on the line
	      if ( currentLineLength + [aWord length] <= theLimit )
		{
		  [aMutableString appendString: aWord];
		  currentLineLength += [aWord length];
		}
	      // We can't
	      else 
		{
		  // We find the first non-whitespace character
		  aRange = [aWord rangeOfCharacterFromSet: [[NSCharacterSet whitespaceCharacterSet] invertedSet]];
		  if ( aRange.location > 0 )
		    {
		      // There are leading whitespace(s)
		      aPadding = [aWord substringToIndex: aRange.location];
		      aWord = [aWord substringFromIndex: aRange.location];
		      
		      //NSLog(@"word doesn't fit: \"%@\" + \"%@\"", aPadding, aWord);

		      // We add all possible whitespace(s) at end of current line
		      while ( [aPadding length] > 0 )
			{
			  // We must add at least one whitespace for the soft-line break
			  length = theLimit - currentLineLength + 1;
			  
			  // We verify if we can fit all whitespace(s) on the line
			  if ( length >= [aPadding length] )
			    {
			      [aMutableString appendString: aPadding];
			      currentLineLength += [aPadding length];
			      break;
			    }
			  // We can't
			  else
			    {
			      // Add maximum number of whitespace(s) and create new line
			      [aMutableString appendFormat: @"%@\n", [aPadding substringToIndex: length]];
			      aPadding = [aPadding substringFromIndex: length];
			      currentLineLength = 0;
			    }
			}
		    }
		      
		  //NSLog(@"current line length = %d", currentLineLength);
		  
		  // We verify if we can fit the word on the line
		  if (currentLineLength + [aWord length] <= theLimit)
		    {
		      [aMutableString appendString: aWord];
		      currentLineLength += [aWord length];
		    }
		  // We can't
		  else
		    {
 		      if ([aWord compare:  @"From"] == NSOrderedSame)
 			{
 			  // Space-stuff line
 			  aWord = [@" " stringByAppendingString: aWord];
 			}
		      
		      currentLineLength = [aWord length];
		      
		      // We verify if the word exceeds the wrapping limit
		      if ( [aWord length] > theLimit )
			{
			  aWord = [MimeUtility _breakWord: aWord usingLimit: 998];
			  // We verify if the word has been splitted
			  aRange = [aWord rangeOfString: @"\n" options: NSBackwardsSearch];
			  if (aRange.length)
			    {
			      //NSLog(@"word is too long: last CR is at %d", aRange.location);
			      currentLineLength = 998 - aRange.location;
			    }
			  // We verify if the last part of the possibly-splitted word is still larger than the limit
			  if (currentLineLength > theLimit)
			    {
			      // Add soft line break
			      [aMutableString appendString: @" \n"];
			      currentLineLength = 0;
			    }
			}

		      [aMutableString appendFormat: @"\n%@", aWord];
		    }
		}

	      previousWordLocation = currentWordLocation;
	    }
	  
	  //NSLog(@"line (wrapped) length = %d", currentLineLength);

	  RELEASE(pool);
		  
	  RELEASE(aScannerWhitespace);
	}

      previousLineLocation = currentLineLocation;
    }
  
  //NSLog(@"wrapPlainTextString\n\"%@\"\n usingWrappingLimit %d\nreturns\n\"%@\"", theString, theLimit, aMutableString);
  
  RELEASE(aScannerSoftLineBreak);

  // We return the wrapped string
  return AUTORELEASE(aMutableString);
}


//
//
//
+ (NSString *) quotePlainTextString: (NSString *) theString
			 quoteLevel: (int) theLevel
		      wrappingLimit: (int) theLimit
{
  NSMutableString *aMutableString, *aQuotePrefix;
  NSScanner *aScannerSoftLineBreak;
  NSString *aString, *aLine;
  NSRange aRange;
  int previousLineLocation, currentLineLocation;
  BOOL isQuoted;
  int i;

  NSAutoreleasePool *pool;

  // We verify if the wrapping limit is smaller then the quote level
  if (theLevel > theLimit) 
    {
      return @"";
    }
  
  // We initialize our local variables
  // We wrap the text block
  aMutableString = [[NSMutableString alloc] initWithCapacity: [theString length]];
  aQuotePrefix = [[NSMutableString alloc] initWithCapacity: theLevel];

  
  aString = [MimeUtility wrapPlainTextString: theString 
			 usingWrappingLimit: (theLimit - theLevel)];

  aScannerSoftLineBreak = [[NSScanner alloc] initWithString: aString];

  previousLineLocation = currentLineLocation = 0;

  // We prepare the line prefix
  for (i = 0; i < theLevel; i++)
    {
      [aQuotePrefix appendString: @">"];
    }
  
  // We create our local autorelease pool
  pool = [[NSAutoreleasePool alloc] init];
  
  // We add the line prefix to each wrapped line
  while ( [aScannerSoftLineBreak scanUpToCharactersFromSet: [NSCharacterSet characterSetWithCharactersInString: @"\n"]
				 intoString: NULL] )
    {
      currentLineLocation = [aScannerSoftLineBreak scanLocation];
      aLine = [aString substringWithRange: NSMakeRange(previousLineLocation, currentLineLocation - previousLineLocation)];
      
      // We verify if the line starts with newline(s)
      aRange = [aLine rangeOfCharacterFromSet: [[NSCharacterSet characterSetWithCharactersInString: @"\n"] invertedSet]];
      if ( aRange.location > 0 )
	{
	  [aMutableString appendString: [aLine substringToIndex: aRange.location]];
	  aLine = [aLine substringFromIndex: aRange.location];
	}
      
      isQuoted = ([aLine length] > 0 && [aLine characterAtIndex: 0] == '>' );
      
      [aMutableString appendString: aQuotePrefix];
      if (!isQuoted)
	{
	  [aMutableString appendString: @" "];
	}
      [aMutableString appendString: aLine];

      previousLineLocation = currentLineLocation;
    }

  //NSLog(@"quotePlainTextString \n\"%@\"\n quoteLevel %d\n wrappingLimit %d\nreturns\n\"%@\"", theString, theLevel, theLimit, aMutableString);

  RELEASE(pool);

  RELEASE(aScannerSoftLineBreak);
  RELEASE(aQuotePrefix);

  return AUTORELEASE(aMutableString);
}



//
// FIXME: Is this method loosing text?
//
+ (NSString *) unwrapPlainTextString: (NSString *) theString
	     usingQuoteWrappingLimit: (int) theQuoteLimit
{
  NSMutableString *aMutableString, *aQuotedLine;
  NSScanner *aScannerSoftLineBreak;
  NSString *aLine, *aPadding;
  NSRange aRange;
  int previousLineLocation, currentLineLocation;
  int previousQuoteLevel, currentQuoteLevel;
  BOOL newParagraph;
  
  NSAutoreleasePool *pool;
  
  // We first verify if the string is valid
  if (!theString || [theString length] == 0)
    {
      return @"";
    }

  // We initialize our local variables
  aMutableString = [[NSMutableString alloc] initWithCapacity: [theString length]];
  aQuotedLine = [[NSMutableString alloc] initWithCapacity: [theString length]];

  aScannerSoftLineBreak = [[NSScanner alloc] initWithString: theString];

  currentLineLocation = previousLineLocation = 0;
  previousQuoteLevel = currentQuoteLevel = 0;
  newParagraph = YES;

  // We create our local autorelease pool
  pool = [[NSAutoreleasePool alloc] init];
  
  while ( [aScannerSoftLineBreak scanUpToCharactersFromSet: [NSCharacterSet characterSetWithCharactersInString: @"\n"]
				 intoString: NULL] )
    {
      currentLineLocation = [aScannerSoftLineBreak scanLocation];
      currentQuoteLevel = 0;
      aLine = [theString substringWithRange: NSMakeRange(previousLineLocation, currentLineLocation - previousLineLocation)];
      
//      NSLog(@"unwrapPlainTextString IN\n\"%@\"  temp='%@'\n", aLine,aQuotedLine);
      
      // We verify if the previous paragraph must be quoted
      if ( newParagraph && previousQuoteLevel > 0 )
	{
	  [aMutableString appendString: [MimeUtility quotePlainTextString: aQuotedLine 
						     quoteLevel: previousQuoteLevel 
						     wrappingLimit: theQuoteLimit]];
	  [aQuotedLine deleteCharactersInRange: NSMakeRange(0, [aQuotedLine length])];
	}

      // We verify if the line starts with newline(s)
      aRange = [aLine rangeOfCharacterFromSet: [[NSCharacterSet characterSetWithCharactersInString: @"\n"] invertedSet]];
      if ( aRange.location > 0 )
	{
//	  NSLog(@"\t(LF count = %d)", aRange.location);
	  
	  // We verify if it's a new paragraph
	  if ( newParagraph )
	    {
	      aPadding = [aLine substringToIndex: aRange.location];
	      [aMutableString appendString: aPadding];
	    }
	  // We verify if there are more than one leading new line
	  else if (aRange.location > 1)
	    {
	      newParagraph = YES;
	      aPadding = [aLine substringToIndex: aRange.location - 1];
	      [aMutableString appendString: aPadding];
	      aRange.location++;
	    }
	  
	  aLine = [aLine substringFromIndex: aRange.location];
	}

      // We analyse the quote level
      while ( currentQuoteLevel < [aLine length] && [aLine characterAtIndex: currentQuoteLevel] == '>' )
	{
	  currentQuoteLevel++;
	}

//      NSLog(@"\t(quote level = %d)", currentQuoteLevel);

      // We verify if there's any inconsistency between the quote level and the soft-line breaks
      if ( currentQuoteLevel != previousQuoteLevel && !newParagraph )
	{

	  // Wrong soft-line break; must add hard-line break
	  [aMutableString appendString: @"\n"];
	  newParagraph = YES;
	}

      // Remove leading quote characters
      if ( [aLine length] == currentQuoteLevel )
	{
	  aLine = @"\n";
	}
      else
	{
	  aLine = [aLine substringFromIndex: currentQuoteLevel];
	}

      // We verify if the line has been space-stuffed
      if ( [aLine length] > 0 && [aLine characterAtIndex: 0] == ' ' )
	{
	  // Remove leading space
	  aLine = [aLine substringFromIndex: 1];
	}
            
      // We verify if the current line is quoted
      if ( currentQuoteLevel > 0 )
	{
	  [aQuotedLine appendString: aLine];
	}
      else
	{
	  [aMutableString appendString: aLine];
	}
      
      // We verify if the line ends with a soft-line break
      if ( [aLine length] > 0 && [aLine characterAtIndex: [aLine length] - 1] == ' ' )
	{
	  newParagraph = NO;
	}
      else
	{
	  newParagraph = YES;
	}
      
      previousQuoteLevel = currentQuoteLevel;
      previousLineLocation = currentLineLocation;

      //      NSLog(@"unwrapPlainTextString OUT\n\"%@\"", aMutableString);
      
    } // while (...)
  
  RELEASE(pool);
  
  //  NSLog(@"unwrapPlainTextString \n\"\%@\"\n usingQuoteWrappingLimit %d\nreturns\n\"%@\"", theString, theQuoteLimit, aMutableString);

  RELEASE(aScannerSoftLineBreak);
  RELEASE(aQuotedLine);

  return AUTORELEASE(aMutableString);
}


//
//
//
+ (Message *) compositeMessageContentFromRawSource: (NSData *) theData
{
  Message *aMessage;
  
  aMessage = [[Message alloc] initWithData: theData];
  
  return AUTORELEASE(aMessage);
}


//
// FIXME: whitespace after boundary markers
// 
+ (MimeMultipart *) compositeMultipartContentFromRawSource: (NSData *) theData
					     usingBoundary: (NSData *) theBoundary
{
  MimeMultipart *aMimeMultipart;
  
  NSMutableData *aMutableData;
  NSArray *allMimeBodyParts;
  NSRange aRange;
  int i;
  
  // We first create our MimeMultipart object that will hold our MimeBodyPart objects
  aMimeMultipart = [[MimeMultipart alloc] init];

  aMutableData=[[NSMutableData alloc] init];
  [aMutableData appendBytes: "--"  length: 2];
  [aMutableData appendData: theBoundary];
  
  // We first skip everything before the first boundary
  aRange = [theData rangeOfData: aMutableData];
   
  // Was if ( aRange.length > 0 ) ...
  if ( aRange.length && aRange.location )
    {
      theData = [theData subdataFromIndex: (aRange.location + aRange.length)];
    }
  
  [aMutableData setLength: 0];
  [aMutableData appendBytes: "\n--"  length: 3];
  [aMutableData appendData: theBoundary];
  
  // Add terminating 0 so we can use it as a cstring below
  [aMutableData appendBytes: "\0" length: 1];

  // We split this mime body part into multiple parts since we have various representations
  // of the actual body part.
  allMimeBodyParts = [theData componentsSeparatedByCString: [aMutableData bytes] ];

  DESTROY(aMutableData);

  for (i = 0; i < [allMimeBodyParts count]; i++)
    {
      MimeBodyPart *aMimeBodyPart;
      NSData *aData;

      // We get the string corresponding to our alternative body part
      aData = [allMimeBodyParts objectAtIndex: i];
      
      if (aData && [aData length] > 0)
	{
	  // this is the last part. Ignore everything past the end marker
	  if ( [aData hasCPrefix: "--\n"] ||
	       ( [aData length] == 2 && [aData hasCPrefix: "--"] ) )
	    {
	      break;
	    }
	  
	  // We then skip the first character since it's a \n (the one at the end of the boundary)
	  if ( [aData hasCPrefix: "\n"] )
	    {
	      aData = [aData subdataFromIndex: 1];
	    }
	  
	  aMimeBodyPart = [[MimeBodyPart alloc] initWithData: aData];
	  [aMimeMultipart addBodyPart: aMimeBodyPart];
	  RELEASE(aMimeBodyPart);
	}
    }

  return AUTORELEASE(aMimeMultipart);
}


//
// We handle in this method discrete types likes text/*, application/*, image/*, etc..
//
// It might return a NSString or a NSData object (if the encoding is BASE64).
// 
+ (id) discreteContentFromRawSource: (NSData *) theData
       usingContentTransferEncoding: (int) theContentTransferEncoding
			    charset: (NSString *) theCharset
			       part: (Part *) thePart
{
  NSString *aString;

  // We get the right charset for our message
  if (!theCharset || [theCharset caseInsensitiveCompare: @"us-ascii"] == NSOrderedSame)
    {
      theCharset = @"iso-8859-1";
    }

  if ( theContentTransferEncoding == QUOTEDPRINTABLE )
    {
      NSData *aData;
      
      // We decode our content from QP using our charset
      aData = [MimeUtility decodeQuotedPrintable: theData  inHeader: NO];
      
      aString = [MimeUtility stringWithData: aData
			  charset: [theCharset dataUsingEncoding: NSASCIIStringEncoding]];
      if ( aString )
	{
	  return aString;
	}
      
      // Decoding failed, we return the raw data
      return aData;
    }
  else if ( theContentTransferEncoding == BASE64 && [thePart isMimeType: @"text" : @"*"] )
    {	  
      NSData *aData;
      
      aData = [theData dataByRemovingLineFeedCharacters];
      aData = [MimeUtility decodeBase64: aData];

      aString = [MimeUtility stringWithData: aData
			  charset: [theCharset dataUsingEncoding: NSASCIIStringEncoding]];
      if ( aString )
	{
	  return aString;
	}
      
      // Decoding failed, we return the raw data
      return aData;
    }
  else if ( theContentTransferEncoding == BASE64 )
    {	  
      NSData *aData;
      
      aData = [theData dataByRemovingLineFeedCharacters];
      aData = [MimeUtility decodeBase64: aData];
      
      return aData;
    }

  aString= [MimeUtility stringWithData: theData
			charset: [theCharset dataUsingEncoding: NSASCIIStringEncoding]];

  // Decoding failed, we return the raw data
  if ( !aString )
    {
      return theData;
    }
  
  // We have a 'standard encoding' but the format is "flowed"
  if ( [thePart format] == FORMAT_FLOWED )
    {
      // We decode the flowed text
      aString =  [MimeUtility unwrapPlainTextString: aString usingQuoteWrappingLimit: 80];
      
      return aString;
    }
  
  // Our format isn't flowed and we use NONE (7BIT), 8BIT or BINARY as our encoding.
  return aString;
}


+ (void) setContentFromRawSource: (NSData *) theData
                          inPart: (Part *) thePart
{
  NSAutoreleasePool *pool;

  RETAIN(theData);
  RETAIN(thePart);
  
  // We create a temporary autorelease pool since this method can be
  // memory consuming on our default autorelease pool.
  pool = [[NSAutoreleasePool alloc] init];

  // 
  // Discrete types (text/application/audio/image/video).
  //
  if ( [thePart isMimeType: @"text" : @"*"] ||
       [thePart isMimeType: @"application" : @"*"] ||
       [thePart isMimeType: @"audio" : @"*"] ||
       [thePart isMimeType: @"image" : @"*"] ||
       [thePart isMimeType: @"video" : @"*"]  ||
       [thePart isMimeType: @"message" : @"delivery-status"] )
    {
      [thePart setContent: [MimeUtility discreteContentFromRawSource: theData
					usingContentTransferEncoding: [thePart contentTransferEncoding]
					charset: [thePart charset]
					part: thePart] ];
    }
  //
  // Composite types (message/multipart).
  //
  else if ( [thePart isMimeType: @"message" : @"rfc822"] )
    {
      [thePart setContent: [MimeUtility compositeMessageContentFromRawSource: theData] ];
    }
  else if ( [thePart isMimeType: @"multipart" : @"*"] )
    {
      [thePart setContent: [MimeUtility compositeMultipartContentFromRawSource: theData
					usingBoundary: [thePart boundary]] ];
    }
  //
  // Extension type -> not supported for now.
  //
  // FIXME: RFCs say to treat this like application/octet-stream */
  else
    {
      NSLog( @"UNSUPPORTED CONTENT-TYPE: PLEASE REPORT THIS BUG!" );
      NSLog( @"The Content-Type was: %@", [thePart contentType] );
      [thePart setContent: @"UNSUPPORTED CONTENT-TYPE: PLEASE REPORT THIS BUG!"];
    }
  
  RELEASE(pool);

  RELEASE(theData);
  RELEASE(thePart);
}


//
//
//
+ (NSFileWrapper *) fileWrapperFromUUEncodedString: (NSString *) theString
{
  NSFileWrapper *aFileWrapper;

  NSMutableData *aMutableData;
  NSArray *allLines;
  
  NSNumber *filePermissions;
  NSString *aString, *filename;
  
  int i;

  aMutableData = [NSMutableData dataWithLength: [theString length]];

  allLines = [theString componentsSeparatedByString: @"\n"];

  // We decode our filename and our mode
  aString = [allLines objectAtIndex: 0];

  filePermissions = [NSNumber numberWithInt: [[[aString componentsSeparatedByString: @" "] objectAtIndex: 1] intValue]];
  filename = [[aString componentsSeparatedByString: @" "] objectAtIndex: 2];
  
  // We now get the data representing our uuencoding string
  for (i = 1; i < ([allLines count] - 1); i++)
    {
      NSString *aLine;

      aLine = [allLines objectAtIndex: i];

      uudecodeline((char *)[aLine cString], aMutableData);
    }

  // We finally initialize our file wrapper will all our informations
  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aMutableData];
  
  [aFileWrapper setPreferredFilename: filename];
  [aFileWrapper setFileAttributes: [NSDictionary dictionaryWithObject: filePermissions
						 forKey: NSFilePosixPermissions]];

  return aFileWrapper;
}


//
// FIXME, we currently ignore theRange
//
+ (NSRange) rangeOfUUEncodedStringFromString: (NSString *) theString
                                       range: (NSRange) theRange
{
  NSRange r1, r2;

  r1 = [theString rangeOfString: @"begin "];

  if ( r1.length == 0 )
    {
      return NSMakeRange(NSNotFound, 0);
    }

  r2 = [theString rangeOfString: @"\nend"
		  options: 0
		  range: NSMakeRange(r1.location, [theString length] - r1.location)];
  
  if ( r2.length == 0 )
    {
      return NSMakeRange(NSNotFound, 0);
    }
  
  return NSMakeRange(r1.location, (r2.location + r2.length) - r1.location);
}


@end


//
// C functions
//

//
//
//
int getValue(char c) {
  if (c >= 'A' && c <= 'Z') return (c - 'A');
  if (c >= 'a' && c <= 'z') return (c - 'a' + 26);
  if (c >= '0' && c <= '9') return (c - '0' + 52);
  if (c == '+') return 62;
  if (c == '/') return 63;
  if (c == '=') return 0;
  return -1;
}

//
//
//
void nb64ChunkFor3Characters(char *buf, const char *inBuf, int numChars) {
  if (numChars >= 3)
    {
      buf[0] = basis_64[inBuf[0]>>2 & 0x3F];
      buf[1] = basis_64[(((inBuf[0] & 0x3)<< 4) | ((inBuf[1] & 0xF0) >> 4)) & 0x3F];
      buf[2] = basis_64[(((inBuf[1] & 0xF) << 2) | ((inBuf[2] & 0xC0) >>6)) & 0x3F];
      buf[3] = basis_64[inBuf[2] & 0x3F];
    }
  else if(numChars == 2)
    {
      buf[0] = basis_64[inBuf[0]>>2 & 0x3F];
      buf[1] = basis_64[(((inBuf[0] & 0x3)<< 4) | ((inBuf[1] & 0xF0) >> 4)) & 0x3F];
      buf[2] = basis_64[(((inBuf[1] & 0xF) << 2) | ((0 & 0xC0) >>6)) & 0x3F];
      buf[3] = '=';
    }
  else
    {
      buf[0] = basis_64[inBuf[0]>>2 & 0x3F];
      buf[1] = basis_64[(((inBuf[0] & 0x3)<< 4) | ((0 & 0xF0) >> 4)) & 0x3F];
      buf[2] = '=';
      buf[3] = '=';
    }
}

//
//
//
void uudecodeline(char *line, NSMutableData *data)
{
  int c, len;
  
  len = UUDECODE(*line++);
  
  while (len)
    {
      c = UUDECODE(*line) << 2 | UUDECODE(line[1]) >> 4;
      
      [data appendBytes: &c
	    length: 1];
      
      if (--len)
	{
	  c = UUDECODE(line[1]) << 4 | UUDECODE(line[2]) >> 2;
	  
	  [data appendBytes: &c
		length: 1];
	  
	  if (--len)
	    {
	      c = UUDECODE(line[2]) << 6 | UUDECODE(line[3]);
	      [data appendBytes: &c
		    length: 1];
	      len--;
	    }
	}
      line += 4;
    }
  
  return;
}
