/* $Id: grid.h,v 1.19 2000/04/04 23:58:15 knepley Exp $ */

/*
    Routines for working with grids
*/

#ifndef __GRID_H
#define __GRID_H

#include "mesh.h"
#include "discretization.h"

/*
  A Grid encapsulates information about fields defined over a mesh. Thus it must
  contain information about the discretization procedure and ordering of variables.
  We also maintain all the information about the problem, such as operators involved, etc.
*/
typedef struct _Grid* Grid;

#define GRID_TRIANGULAR_1D "tri1d"
#define GRID_TRIANGULAR_2D "tri2d"
typedef char *GridType;

#define GRID_SER_GENERIC "generic"
typedef char *GridSerializeType;

/* Allows the user to select the type of boundary values, mostly for time-dependent problems */
typedef enum {BC_VALUES, BC_VALUES_OLD, BC_VALUES_DIFF} BCValuesType;
/* Types of constraint information */
typedef enum {CONSTRAINT_ROW_INDEX, CONSTRAINT_COL_INDEX, CONSTRAINT_NEW_INDEX, CONSTRAINT_ROW, CONSTRAINT_COL,
              CONSTRAINT_ROW_TRANS, CONSTRAINT_COL_TRANS} ConstraintFlag;

/* Logging support */
extern int GRID_COOKIE;
extern int ELEMENT_VEC_COOKIE;
extern int ELEMENT_MAT_COOKIE;
extern int VAR_ORDER_COOKIE;
extern int CLASS_MAP_COOKIE;
extern int GRID_Reform, GRID_SetUp;

EXTERN int GridInitializePackage(char *);

EXTERN int GridCreate(Mesh, Grid *);
extern int GridSetUp(Grid);
extern int GridSetupGhostScatter(Grid);
extern int GridSetupBoundary(Grid);
extern int GridSerialize(MPI_Comm, Grid *, PetscViewer, PetscTruth);
extern int GridDestroy(Grid);
extern int GridView(Grid, PetscViewer);
extern int GridViewFromOptions(Grid, char *);

extern int GridSetFromOptions(Grid);
extern int GridPrintHelp(Grid);
extern int GridSetOptionsPrefix(Grid, char *);
extern int GridAppendOptionsPrefix(Grid, char *);
extern int GridGetOptionsPrefix(Grid, char **);
extern int GridSetViewField(Grid, int, int);

extern int GridGetMesh(Grid, Mesh *);
extern int GridGetNodeClass(Grid, int, int *);
extern int GridGetNearestNode(Grid, int, double, double, double, int *);
extern int GridGetNearestBdNode(Grid, int, double, double, double, int *);
extern int GridGetBoundarySize(Grid, int, int, int *);
extern int GridGetBoundaryStart(Grid, int, int, PetscTruth, int *, int *);
extern int GridGetBoundaryNext(Grid, int, int, PetscTruth, int *, int *);
extern int GridReformMesh(Grid, PetscTruth, PointFunction, PetscTruth, Mesh *);
extern int GridRefineMesh(Grid, PointFunction, Grid *);

extern int GridGetDiscretization(Grid, int, Discretization *);

extern int GridInterpolateElementVec(Grid, int, ElementVec, Grid, int, ElementVec);

extern int GridGhostExchange(MPI_Comm, int, int *, int *, PetscDataType, int *, InsertMode, ScatterMode, void *, void *);

#include "gridBC.h"
#include "gridDB.h"

/* Dynamic creation and loading functions */
extern PetscFList GridList;
extern PetscTruth GridRegisterAllCalled;
extern int GridSetType(Grid, GridType);
extern int GridGetType(Grid, GridType *);
extern int GridRegister(const char[],const char[],const char[],int(*)(Grid));
extern int GridRegisterAll(const char []);
extern int GridRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define GridRegisterDynamic(a,b,c,d) GridRegister(a,b,c,0)
#else
#define GridRegisterDynamic(a,b,c,d) GridRegister(a,b,c,d)
#endif

extern PetscFList GridSerializeList;
extern PetscTruth GridSerializeRegisterAllCalled;
extern int GridSetSerializeType(Grid, GridSerializeType);
extern int GridGetSerializeType(Grid, GridSerializeType *);
extern int GridSerializeRegister(const char [], const char [], const char [], int (*)(MPI_Comm, Grid *, PetscViewer, PetscTruth));
extern int GridSerializeRegisterAll(const char []);
extern int GridSerializeRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define GridSerializeRegisterDynamic(a,b,c,d) GridSerializeRegister(a,b,c,0)
#else
#define GridSerializeRegisterDynamic(a,b,c,d) GridSerializeRegister(a,b,c,d)
#endif

/*-------------------------------------------- Class Structure ------------------------------------------------------*/
/*
  FieldClassMap encapsulates the class structure, the assigments of classes to nodes, and cosntraints implemented
  through classes.
*/
typedef struct _FieldClassMap *FieldClassMap;
#define CLASS_MAP_TRIANGULAR_2D  "tri2d"
typedef char *FieldClassMapType;
#define CLASS_MAP_SER_TRIANGULAR_2D_BINARY "tri_2d_binary"
typedef char *FieldClassMapSerializeType;

extern int FieldClassMapCreateTriangular2D(Grid, int, int *, FieldClassMap *);

extern int FieldClassMapCreateSubset(FieldClassMap, int, int *, FieldClassMap *);
extern int FieldClassMapConstrain(FieldClassMap, Grid, PetscTruth, PetscTruth, FieldClassMap *);
extern int FieldClassMapReduce(FieldClassMap, Grid, FieldClassMap *);
extern int FieldClassMapDuplicate(FieldClassMap, FieldClassMap *);

extern int FieldClassMapDestroy(FieldClassMap);
extern int FieldClassMapView(FieldClassMap, PetscViewer);

extern PetscFList FieldClassMapList;
extern int FieldClassMapRegisterAll(const char []);
extern int FieldClassMapRegisterDestroy(void);
extern int FieldClassMapRegisterAllCalled;
extern int FieldClassMapCreate(MPI_Comm, FieldClassMap *);
extern int FieldClassMapSetType(FieldClassMap, FieldClassMapType);
extern int FieldClassMapGetType(FieldClassMap, FieldClassMapType *);
extern int FieldClassMapRegister_Private(const char[],const char[],const char[],int(*)(FieldClassMap));
extern int FieldClassMapSerializeRegister_Private(const char [], const char [], const char [],
                                                  int (*)(MPI_Comm, FieldClassMap *, PetscViewer, PetscTruth));
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define FieldClassMapRegister(a,b,c,d) FieldClassMapRegister_Private(a,b,c,0)
#define FieldClassMapSerializeRegister(a,b,c,d) FieldClassMapSerializeRegister_Private(a,b,c,0)
#else
#define FieldClassMapRegister(a,b,c,d) FieldClassMapRegister_Private(a,b,c,d)
#define FieldClassMapSerializeRegister(a,b,c,d) FieldClassMapSerializeRegister_Private(a,b,c,d)
#endif

extern PetscFList FieldClassMapSerializeList;
extern int FieldClassMapSerializeRegisterAll(const char []);
extern int FieldClassMapSerializeRegisterDestroy(void);
extern int FieldClassMapSerializeRegisterAllCalled;
extern int FieldClassMapSerialize(MPI_Comm, FieldClassMap *, PetscViewer, PetscTruth);
extern int FieldClassMapSetSerializeType(FieldClassMap, FieldClassMapSerializeType);

extern int GridGetClassMap(Grid, FieldClassMap *);

extern int FieldClassMapIsConstrained(FieldClassMap, PetscTruth *);
extern int FieldClassMapIsDefined(FieldClassMap, int, int, PetscTruth *);
extern int FieldClassMapGetNumFields(FieldClassMap, int *);
extern int FieldClassMapGetField(FieldClassMap, int, int *);
extern int FieldClassMapGetNodeClass(FieldClassMap, int, int *);

/*------------------------------------------- Variable Ordering -----------------------------------------------------*/
/*
  VarOrdering is an auxiliary object which encapsulates a global variable ordering for a local node-based discretization.
  LocalVarOrdering is an auxiliary object which encapsulates a local variable ordering for a node-based discretization.
*/
typedef struct _VarOrdering      *VarOrdering;
typedef struct _LocalVarOrdering *LocalVarOrdering;

extern int VarOrderingCreate(Grid, VarOrdering *);
extern int VarOrderingCreateConstrained(Grid, VarOrdering *);
extern int VarOrderingCreateGeneral(Grid, int, int *, VarOrdering *);
extern int VarOrderingCreateSubset(VarOrdering, int, int *, PetscTruth, VarOrdering *);
extern int VarOrderingConstrain(Grid, VarOrdering, VarOrdering *);
extern int VarOrderingCreateReduce(Grid, VarOrdering *);
extern int VarOrderingDestroy(VarOrdering);
extern int VarOrderingSerialize(FieldClassMap, VarOrdering *, PetscViewer, PetscTruth);
extern int VarOrderingDuplicate(VarOrdering, VarOrdering *);
extern int VarOrderingDuplicateIndices(VarOrdering, VarOrdering);

extern int VarOrderingGetClassMap(VarOrdering, FieldClassMap *);
extern int VarOrderingGetNumTotalFields(VarOrdering, int *);
extern int VarOrderingGetSize(VarOrdering, int *);
extern int VarOrderingGetLocalSize(VarOrdering, int *);
extern int VarOrderingGetFirst(VarOrdering, int **);

extern int VarOrderingCompatible(VarOrdering, VarOrdering, PetscTruth *);

extern int LocalVarOrderingCreate(Grid, int , int *, LocalVarOrdering *);
extern int LocalVarOrderingDestroy(LocalVarOrdering);
extern int LocalVarOrderingSerialize(Grid, LocalVarOrdering *, PetscViewer, PetscTruth);
extern int LocalVarOrderingDuplicate(LocalVarOrdering, LocalVarOrdering *);
extern int LocalVarOrderingDuplicateIndices(LocalVarOrdering, LocalVarOrdering);

extern int LocalVarOrderingGetSize(LocalVarOrdering, int *);
extern int LocalVarOrderingGetField(LocalVarOrdering, int, int *);
extern int LocalVarOrderingGetFieldIndex(LocalVarOrdering, int, int *);
extern int LocalVarOrderingGetFieldStart(LocalVarOrdering, int, int *);

extern int GridGetOrder(Grid, VarOrdering *);
extern int GridGetLocalOrder(Grid, LocalVarOrdering *);
extern int GridCalcElementVecIndices(Grid, int, ElementVec);
extern int GridCalcElementMatIndices(Grid, int, ElementMat);
extern int GridCalcGeneralElementVecIndices(Grid, int, VarOrdering, VarOrdering, PetscTruth, ElementVec);
extern int GridCalcGeneralElementMatIndices(Grid, int, VarOrdering, VarOrdering, PetscTruth, ElementMat);
extern int GridCalcLocalElementVecIndices(Grid, int, ElementVec);
extern int GridCalcLocalElementMatIndices(Grid, int, ElementMat);
extern int GridCalcBoundaryElementVecIndices(Grid, int, int, int, VarOrdering, PetscTruth, ElementVec);
extern int GridCalcBoundaryElementMatIndices(Grid, int, int, int, VarOrdering, VarOrdering, PetscTruth, ElementMat);
extern int GridCalcInterpolationElementVecIndices(Grid, Mesh, int, VarOrdering, ElementVec);
extern int GridCalcReductionElementMatIndices(Grid, int, VarOrdering, VarOrdering, ElementMat);
extern int GridProjectElementVec(Grid, Mesh, int, VarOrdering, PetscTruth, ElementVec);
extern int GridProjectInterpolationElementVec(Grid, Mesh, int, VarOrdering, PetscTruth, ElementVec);

#endif /* __GRID_H */
