#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: tri2d_Triangle.c,v 1.6 2000/10/17 13:48:57 knepley Exp $";
#endif

#include "src/mesh/impls/triangular/2d/2dimpl.h"         /*I "mesh.h" I*/
#include "tri2d_Triangle.h"

#ifdef PETSC_HAVE_TRIANGLE

#define ANSI_DECLARATORS
#include "triangle.h"

extern int MeshSetupBoundary_Triangular_2D(Mesh);
extern int MeshDebug_Triangular_2D(Mesh, PetscTruth);
extern int MeshSetupSupport_Triangular_2D(Mesh);
extern int MeshAssemble_Private(Mesh, double **, int **, int **, int **);

/*-------------------------------------------- Standard Interface ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshInitInput_Triangle"
/* MeshInitInput_Triangle
  This function initializes the input structure for Triangle.

  Collective on Mesh

  Output Parameter:
. inputCtx - Structure for communicating with Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitOutput_Triangle()
*/
int MeshInitInput_Triangle(struct triangulateio *inputCtx)
{
  PetscFunctionBegin;
  /* Setup the input structure */
  inputCtx->numberofpoints          = 0;
  inputCtx->numberofpointattributes = 0;
  inputCtx->pointlist               = PETSC_NULL;
  inputCtx->pointattributelist      = PETSC_NULL;
  inputCtx->pointmarkerlist         = PETSC_NULL;
  inputCtx->numberofsegments        = 0;
  inputCtx->segmentlist             = PETSC_NULL;
  inputCtx->segmentmarkerlist       = PETSC_NULL;
  inputCtx->numberofholes           = 0;
  inputCtx->holelist                = PETSC_NULL;
  inputCtx->numberofregions         = 0;
  inputCtx->regionlist              = PETSC_NULL;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshInitRefineInput_Triangle"
/* MeshInitRefineInput_Triangle
  This function fills the input structure for Triangle with mesh information before refinement.

  Collective on Mesh

  Input Parameters:
+ mesh    - The mesh being refined
- area     - A function which gives an area constraint when evaluated inside an element

  Output Parameter:
. inputCtx - Structure for communicating with Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitInput_Triangle(), MeshInitOutput_Triangle()
*/
int MeshInitRefineInput_Triangle(Mesh mesh, PointFunction area, struct triangulateio *inputCtx)
{
  Mesh_Triangular         *tri        = (Mesh_Triangular *) mesh->data;
  Partition                p          = mesh->part;
  Partition_Triangular_2D *q          = (Partition_Triangular_2D *) p->data;
  int                      numNodes   = q->numNodes;
  int                      numFaces   = p->numElements;
  int                      numCorners = mesh->numCorners;
  int                      rank       = p->rank;
  double                  *nodes;
  int                     *markers, *faces, *edges;
  int                     *segments, *segmarkers;
  double                   x, y;                    /* Centroid of the current element */
  int                      elem, bd, edge;
  int                      ierr;

  PetscFunctionBegin;
  ierr = MeshAssemble_Private(mesh, &nodes, &markers, &faces, &edges);                                    CHKERRQ(ierr);

  if (rank == 0) {
    ierr = PetscMalloc(mesh->numBdEdges*2 * sizeof(int), &segments);                                       CHKERRQ(ierr);
    ierr = PetscMalloc(mesh->numBdEdges   * sizeof(int), &segmarkers);                                     CHKERRQ(ierr);

    /* Needed to maintain boundary markers on edges */
    for(bd = 0; bd < mesh->numBd; bd++) {
      for(edge = tri->bdEdgeBegin[bd]; edge < tri->bdEdgeBegin[bd+1]; edge++) {
        /* Make all node numbers global */
        segments[edge*2]   = edges[tri->bdEdges[edge]*2];
        segments[edge*2+1] = edges[tri->bdEdges[edge]*2+1];
        segmarkers[edge]   = tri->bdMarkers[bd];
      }
    }
    ierr = PetscFree(edges);                                                                              CHKERRQ(ierr);

    if (q->nodeOrdering != PETSC_NULL) {
      /* We must globally renumber and permute so that midnodes come after vertices */
      ierr = AOPetscToApplication(q->nodeOrdering, numFaces*numCorners, faces);                           CHKERRQ(ierr);
      ierr = AOPetscToApplicationPermuteReal(q->nodeOrdering, 2, nodes);                                  CHKERRQ(ierr);
      ierr = AOPetscToApplicationPermuteInt(q->nodeOrdering,    1, markers);                              CHKERRQ(ierr);
      /* We can globally renumber only the segments, not the whole edge array */
      ierr = AOPetscToApplication(q->nodeOrdering, mesh->numBdEdges*2, segments);                          CHKERRQ(ierr);
    }
    if (mesh->nodeOrdering != PETSC_NULL) {
      ierr = AOPetscToApplication(mesh->nodeOrdering, numFaces*numCorners, faces);                        CHKERRQ(ierr);
      ierr = AOPetscToApplicationPermuteReal(mesh->nodeOrdering, 2, nodes);                               CHKERRQ(ierr);
      ierr = AOPetscToApplicationPermuteInt(mesh->nodeOrdering,    1, markers);                           CHKERRQ(ierr);
      ierr = AOPetscToApplication(mesh->nodeOrdering, mesh->numBdEdges*2, segments);                       CHKERRQ(ierr);
    }

    /* Setup the input structure */
    inputCtx->numberofpoints             = numNodes;
    inputCtx->numberofpointattributes    = 0;
    inputCtx->pointlist                  = nodes;
    inputCtx->pointattributelist         = PETSC_NULL;
    inputCtx->pointmarkerlist            = markers;
    inputCtx->numberoftriangles          = numFaces;
    inputCtx->numberofcorners            = numCorners;
    inputCtx->trianglelist               = faces;
    inputCtx->numberoftriangleattributes = 0;
    inputCtx->triangleattributelist      = PETSC_NULL;
    inputCtx->trianglearealist           = PETSC_NULL;
    inputCtx->numberofsegments           = mesh->numBdEdges;
    inputCtx->segmentlist                = segments;
    inputCtx->segmentmarkerlist          = segmarkers;
    inputCtx->numberofholes              = 0;
    inputCtx->holelist                   = PETSC_NULL;
    inputCtx->numberofregions            = 0;
    inputCtx->regionlist                 = PETSC_NULL;

    /* Create list of area constraints */
    ierr = PetscMalloc(numFaces * sizeof(double), &inputCtx->trianglearealist);                           CHKERRQ(ierr);
    for(elem = 0; elem < numFaces; elem++) {
      PetscReal maxArea;

      /* Calculate the centroid of the element */
      x = MeshPeriodicX(mesh, (nodes[faces[elem*numCorners]*2]   + nodes[faces[elem*numCorners+1]*2]   +
                        nodes[faces[elem*numCorners+2]*2])  /3.0);
      y = MeshPeriodicY(mesh, (nodes[faces[elem*numCorners]*2+1] + nodes[faces[elem*numCorners+1]*2+1] +
                        nodes[faces[elem*numCorners+2]*2+1])/3.0);
      ierr = (*area)(1, 1, &x, &y, PETSC_NULL, &maxArea, mesh->usr);                                      CHKERRQ(ierr);
      inputCtx->trianglearealist[elem] = maxArea;
    }
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscTrValid(__LINE__, __FUNCT__, __FILE__, __SDIR__);                                           CHKERRQ(ierr);
#endif

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshFinalizeRefineInput_Triangle"
/* MeshFinalizeRefineInput_Triangle
  This function cleans up the input structure for Triangle after refinement.

  Collective on Mesh

  Input Parameters:
. mesh     - The mesh being refined

  Output Parameters:
. inputCtx - Structure for communicating with Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitRefineInput_Triangle(), MeshInitInput_Triangle()
*/
int MeshFinalizeRefineInput_Triangle(Mesh mesh, struct triangulateio *inputCtx)
{
  int ierr;

  PetscFunctionBegin;
  if (mesh->part->rank == 0) {
    ierr = PetscFree(inputCtx->pointlist);                                                               CHKERRQ(ierr);
    ierr = PetscFree(inputCtx->pointmarkerlist);                                                         CHKERRQ(ierr);
    ierr = PetscFree(inputCtx->trianglelist);                                                            CHKERRQ(ierr);
    ierr = PetscFree(inputCtx->segmentlist);                                                             CHKERRQ(ierr);
    ierr = PetscFree(inputCtx->segmentmarkerlist);                                                       CHKERRQ(ierr);
    ierr = PetscFree(inputCtx->trianglearealist);                                                        CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshInitOutput_Triangle"
/* MeshInitOutput_Triangle
  This function initializes the output structure for Triangle.

  Collective on Mesh

  Output Parameter:
. outputCtx - Structure for communicating with Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitInput_Triangle()
*/
int MeshInitOutput_Triangle(struct triangulateio *outputCtx)
{
  PetscFunctionBegin;
  /* Setup the output structure */
  outputCtx->pointlist             = PETSC_NULL;
  outputCtx->pointattributelist    = PETSC_NULL;
  outputCtx->pointmarkerlist       = PETSC_NULL;
  outputCtx->trianglelist          = PETSC_NULL;
  outputCtx->triangleattributelist = PETSC_NULL;
  outputCtx->neighborlist          = PETSC_NULL;
  outputCtx->segmentlist           = PETSC_NULL;
  outputCtx->segmentmarkerlist     = PETSC_NULL;
  outputCtx->edgelist              = PETSC_NULL;
  outputCtx->edgemarkerlist        = PETSC_NULL;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshDistribute_Triangle"
/* MeshDistribute_Triangle
  This function distributes the output from Triangle identically among processors.

  Collective on Mesh

  Output Parameters:
+ mesh - The mesh
- out  - Structure for communicating with Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitInput_Triangle
*/
int MeshDistribute_Triangle(Mesh mesh, struct triangulateio *out)
{
  int rank;
  int ierr;

  PetscFunctionBegin;
  /* Communicate values */
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  ierr = MPI_Bcast(&out->numberofpoints,    1, MPI_INT, 0, mesh->comm);                                   CHKERRQ(ierr);
  ierr = MPI_Bcast(&out->numberofedges,     1, MPI_INT, 0, mesh->comm);                                   CHKERRQ(ierr);
  ierr = MPI_Bcast(&out->numberoftriangles, 1, MPI_INT, 0, mesh->comm);                                   CHKERRQ(ierr);
  ierr = MPI_Bcast(&out->numberofcorners,   1, MPI_INT, 0, mesh->comm);                                   CHKERRQ(ierr);
  if (rank != 0) {
    out->numberofholes   = 0;
    out->holelist        = PETSC_NULL;
    ierr = PetscMalloc(out->numberofpoints*2                       * sizeof(double), &out->pointlist);    CHKERRQ(ierr);
    ierr = PetscMalloc(out->numberofpoints                         * sizeof(int),    &out->pointmarkerlist); CHKERRQ(ierr);
    ierr = PetscMalloc(out->numberofedges*2                        * sizeof(int),    &out->edgelist);     CHKERRQ(ierr);
    ierr = PetscMalloc(out->numberofedges                          * sizeof(int),    &out->edgemarkerlist); CHKERRQ(ierr);
    ierr = PetscMalloc(out->numberoftriangles*3                    * sizeof(int),    &out->neighborlist); CHKERRQ(ierr);
    ierr = PetscMalloc(out->numberoftriangles*out->numberofcorners * sizeof(int),    &out->trianglelist); CHKERRQ(ierr);
  }
  ierr = MPI_Bcast(out->pointlist,       out->numberofpoints*2,    MPI_DOUBLE, 0, mesh->comm);            CHKERRQ(ierr);
  ierr = MPI_Bcast(out->pointmarkerlist, out->numberofpoints,      MPI_INT,    0, mesh->comm);            CHKERRQ(ierr);
  ierr = MPI_Bcast(out->edgelist,        out->numberofedges*2,     MPI_INT,    0, mesh->comm);            CHKERRQ(ierr);
  ierr = MPI_Bcast(out->edgemarkerlist,  out->numberofedges,       MPI_INT,    0, mesh->comm);            CHKERRQ(ierr);
  ierr = MPI_Bcast(out->neighborlist,    out->numberoftriangles*3, MPI_INT,    0, mesh->comm);            CHKERRQ(ierr);
  ierr = MPI_Bcast(out->trianglelist,    out->numberoftriangles*out->numberofcorners, MPI_INT, 0, mesh->comm); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshCreate_Triangle"
/* MeshCreate_Triangle
  This function creates a 2D unstructured mesh using Triangle.

  Collective on Mesh

  Input Parameter:
. numCorners  - The number of nodes in an element

  Input Parameters from bdCtx:
+ numBD       - The number of closed boundaries in the geometry, or different markers
. numVertices - The number of boundary points
. vertices    - The (x,y) coordinates of the boundary points
. markers     - The boundary markers for nodes, 0 indicates an interior point, each boundary must have a different marker
. numSegments - The number of boundary segments
. segments    - The endpoints of boundary segments or PETSC_NULL
. segMarkers  - The boundary markers for each segment
. numHoles    - The number of holes
- holes       - The (x,y) coordinates of holes or PETSC_NULL

  Output Parameter:
. mesh        - The new mesh created by Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitInput_Triangle
*/
int MeshCreate_Triangle(MeshBoundary2D *bdCtx, int numCorners, Mesh mesh)
{
  Mesh_Triangular     *tri = (Mesh_Triangular *) mesh->data;
  struct triangulateio in;            /* Input  for Triangle mesh generator */
  struct triangulateio out;           /* Output for Triangle mesh generator */
  char                 cmd_line[256]; /* Command line for Triangle */
  int                  rank;
  PetscTruth           opt;
  int                  ierr;

  PetscFunctionBegin;
  /* Initialize communication structures for Triangle */
  ierr = MeshInitInput_Triangle(&in);                                                                    CHKERRQ(ierr);
  ierr = MeshInitOutput_Triangle(&out);                                                                  CHKERRQ(ierr);

  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                               CHKERRQ(ierr);
  if (rank == 0) {
    /* Setup boundaries and holes */
    in.numberofpoints      = bdCtx->numVertices;
    if (bdCtx->numVertices > 0) {
      PetscValidScalarPointer(bdCtx->vertices);
      PetscValidIntPointer(bdCtx->markers);
      in.pointlist         = bdCtx->vertices;
      in.pointmarkerlist   = bdCtx->markers;
    }
    in.numberofsegments    = bdCtx->numSegments;
    if (bdCtx->numSegments > 0) {
      PetscValidIntPointer(bdCtx->segments);
      PetscValidIntPointer(bdCtx->segMarkers);
      in.segmentlist       = bdCtx->segments;
      in.segmentmarkerlist = bdCtx->segMarkers;
    }
    in.numberofholes       = bdCtx->numHoles;
    if (bdCtx->numHoles    > 0) {
      /* We keep these */
      PetscValidScalarPointer(bdCtx->holes);
      ierr = PetscMalloc(in.numberofholes*2 * sizeof(double), &in.holelist);                              CHKERRQ(ierr);
      ierr = PetscMemcpy(in.holelist, bdCtx->holes, in.numberofholes*2 * sizeof(double));                 CHKERRQ(ierr);
    }

    /* Generate the inital coarse mesh and check whether we should create midnodes */
    if (numCorners == 3) {
      ierr = PetscStrcpy(cmd_line, "pqcenzQ");                                                            CHKERRQ(ierr);
    } else if (numCorners == 6) {
      ierr = PetscStrcpy(cmd_line, "pqcenzo2Q");                                                          CHKERRQ(ierr);
    } else {
      SETERRQ1(PETSC_ERR_SUP, "Number of local nodes %d not supported", numCorners);
    }
    ierr = PetscOptionsGetString(PETSC_NULL, "-triangle_cmd", cmd_line, 255, &opt);                       CHKERRQ(ierr);
    triangulate(cmd_line, &in, &out, PETSC_NULL);

    /* Get rid of extra information */
    ierr = PetscFree(out.segmentlist);                                                                    CHKERRQ(ierr);
    ierr = PetscFree(out.segmentmarkerlist);                                                              CHKERRQ(ierr);
  }

  /* Communicate values */
  ierr = MeshDistribute_Triangle(mesh, &out);                                                             CHKERRQ(ierr);

  /* Store the information */
  mesh->numBd       = bdCtx->numBd;
  mesh->numNodes    = out.numberofpoints;
  mesh->numEdges    = out.numberofedges;
  mesh->numVertices = (mesh->numNodes <= mesh->numEdges ? mesh->numNodes : mesh->numNodes - mesh->numEdges);
  mesh->numFaces    = out.numberoftriangles;
  mesh->numCorners  = out.numberofcorners;
  mesh->numHoles    = out.numberofholes;
  tri->nodes       = out.pointlist;
  tri->markers     = out.pointmarkerlist;
  tri->edges       = out.edgelist;
  tri->edgemarkers = out.edgemarkerlist;
  tri->faces       = out.trianglelist;
  tri->neighbors   = out.neighborlist;
  mesh->holes       = out.holelist;
  PetscLogObjectMemory(mesh, (mesh->numNodes*2 + mesh->numNodes) * sizeof(double));
  PetscLogObjectMemory(mesh, (mesh->numEdges*2+mesh->numEdges+mesh->numFaces*mesh->numCorners+mesh->numFaces*3) * sizeof(int));

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshRefine_Triangle"
/*
  MeshRefine_Triangle - This function refines a two dimensional unstructured mesh
  with Triangle using area constraints.

  Collective on Mesh

  Input Parameters:
+ oldMesh - The mesh begin refined
- area    - A function which gives an area constraint when evaluated inside an element

  Output Parameter:
. mesh    - The refined mesh

  Level: developer

.keywords: mesh, Triangle, refine
.seealso: MeshRefine(), MeshCoarsen(), MeshReform()
*/
int MeshRefine_Triangle(Mesh oldMesh, PointFunction area, Mesh mesh)
{
  int                  rank = oldMesh->part->rank;
  Mesh_Triangular     *tri;
  struct triangulateio in;            /* Input  for Triangle mesh generator */
  struct triangulateio out;           /* Output for Triangle mesh generator */
  char                 cmd_line[256]; /* Command line for Triangle */
  MPI_Comm             comm;
  PetscTruth           opt;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscNew(Mesh_Triangular, &tri);                                                                 CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, sizeof(Mesh_Triangular));
  ierr = PetscMemcpy(mesh->ops, oldMesh->ops, sizeof(struct _MeshOps));                                   CHKERRQ(ierr);
  mesh->data             = (void *) tri;
  ierr = PetscStrallocpy(MESH_TRIANGULAR_2D,            &mesh->type_name);                                CHKERRQ(ierr);
  ierr = PetscStrallocpy(MESH_SER_TRIANGULAR_2D_BINARY, &mesh->serialize_name);                           CHKERRQ(ierr);
  mesh->dim              = 2;
  mesh->isPeriodic       = oldMesh->isPeriodic;
  mesh->isPeriodicDim[0] = oldMesh->isPeriodicDim[0];
  mesh->isPeriodicDim[1] = oldMesh->isPeriodicDim[1];
  mesh->isPeriodicDim[2] = oldMesh->isPeriodicDim[2];
  mesh->nodeOrdering     = PETSC_NULL;
  mesh->partitioned      = 0;
  mesh->highlightElement = -1;
  mesh->usr              = oldMesh->usr;

  /* Copy function list */
  if (mesh->qlist != PETSC_NULL) {
    ierr = PetscFListDestroy(&mesh->qlist);                                                               CHKERRQ(ierr);
  }
  ierr = PetscFListDuplicate(oldMesh->qlist, &mesh->qlist);                                               CHKERRQ(ierr);

  /* Initialize communication structures for Triangle */
  ierr = MeshInitRefineInput_Triangle(oldMesh, area, &in);                                                CHKERRQ(ierr);
  ierr = MeshInitOutput_Triangle(&out);                                                                   CHKERRQ(ierr);

  /* Generate a refined mesh */
  if (rank == 0) {
    /* Generate the refined mesh and check whether we should create midnodes */
    if (oldMesh->numCorners == 3) {
      ierr = PetscStrcpy(cmd_line, "qcenzQ");                                                             CHKERRQ(ierr);
    } else if (oldMesh->numCorners == 6) {
      ierr = PetscStrcpy(cmd_line, "qcenzo2Q");                                                           CHKERRQ(ierr);
    } else {
      SETERRQ1(PETSC_ERR_SUP, "Number of local nodes %d not supported", oldMesh->numCorners);
    }
    ierr = PetscOptionsGetString("ref_", "-triangle_cmd", cmd_line, 251, &opt);                           CHKERRQ(ierr);
    ierr = PetscStrcat(cmd_line, "pra");                                                                  CHKERRQ(ierr);
    triangulate(cmd_line, &in, &out, PETSC_NULL);

    /* Get rid of extra information */
    ierr = PetscFree(out.segmentlist);                                                                    CHKERRQ(ierr);
    ierr = PetscFree(out.segmentmarkerlist);                                                              CHKERRQ(ierr);
  }
  ierr = MeshFinalizeRefineInput_Triangle(oldMesh, &in);                                                  CHKERRQ(ierr);

  /* Communicate values */
  ierr = MeshDistribute_Triangle(oldMesh, &out);                                                          CHKERRQ(ierr);

  /* Store the information */
  mesh->numBd       = oldMesh->numBd;
  mesh->numNodes    = out.numberofpoints;
  mesh->numEdges    = out.numberofedges;
  mesh->numVertices = (mesh->numNodes <= mesh->numEdges ? mesh->numNodes : mesh->numNodes - mesh->numEdges);
  mesh->numFaces    = out.numberoftriangles;
  mesh->numCorners  = out.numberofcorners;
  tri->nodes       = out.pointlist;
  tri->markers     = out.pointmarkerlist;
  tri->edges       = out.edgelist;
  tri->edgemarkers = out.edgemarkerlist;
  tri->faces       = out.trianglelist;
  tri->neighbors   = out.neighborlist;
  PetscLogObjectMemory(mesh, (mesh->numNodes*2 + mesh->numNodes) * sizeof(double));
  PetscLogObjectMemory(mesh, (mesh->numEdges*2 + mesh->numEdges + mesh->numFaces*mesh->numCorners + mesh->numFaces*3) * sizeof(int));
  tri->nodesOld    = PETSC_NULL;

  /* Copy holes from previous mesh */
  mesh->numHoles    = oldMesh->numHoles;
  mesh->holes       = PETSC_NULL;
  if (mesh->numHoles > 0) {
    ierr = PetscMalloc(mesh->numHoles*2 * sizeof(double), &mesh->holes);                                    CHKERRQ(ierr);
    ierr = PetscMemcpy(mesh->holes, oldMesh->holes, mesh->numHoles*2 * sizeof(double));                      CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, oldMesh->numHoles*2 * sizeof(double));
  }

  /* Calculate maximum degree of vertices */
  ierr = MeshSetupSupport_Triangular_2D(mesh);                                                            CHKERRQ(ierr);

  /* Construct derived and boundary information */
  ierr = MeshSetupBoundary_Triangular_2D(mesh);                                                           CHKERRQ(ierr);

#ifdef PETSC_USE_BOPT_g
  /* Check mesh integrity */
  ierr = MeshDebug_Triangular_2D(mesh, PETSC_FALSE);                                                      CHKERRQ(ierr);
#endif

  /* Reorder nodes before distributing mesh */
  ierr = PetscOptionsHasName(oldMesh->prefix, "-mesh_reorder", &opt);                                     CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    /* MUST FIX: Since we have duplicated the whole mesh, we may impersonate a serial mesh */
    comm       = mesh->comm;
    mesh->comm = PETSC_COMM_SELF;
    ierr       = MeshGetOrdering(mesh, MESH_ORDER_TRIANGULAR_2D_RCM, &mesh->nodeOrdering);                CHKERRQ(ierr);
    mesh->comm = comm;

    /* Permute arrays implicitly numbered by node numbers */
    ierr = AOApplicationToPetscPermuteReal(mesh->nodeOrdering, 2, tri->nodes);                            CHKERRQ(ierr);
    ierr = AOApplicationToPetscPermuteInt(mesh->nodeOrdering, 1, tri->markers);                           CHKERRQ(ierr);
    ierr = AOApplicationToPetscPermuteInt(mesh->nodeOrdering, 1, tri->degrees);                           CHKERRQ(ierr);
    /* Renumber arrays dependent on the canonical node numbering */
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numEdges*2,                   tri->edges);      CHKERRQ(ierr);
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numFaces*oldMesh->numCorners, tri->faces);      CHKERRQ(ierr);
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numBdNodes,                   tri->bdNodes);    CHKERRQ(ierr);
  }

  /* Initialize partition */
  ierr = MeshPartition(mesh);                                                                             CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

/*-------------------------------------------- Periodic Interface ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshInitInput_Periodic"
/* MeshInitInput_Periodic
  This function initializes the input structure for Triangle in the periodic case.

  Collective on Mesh

  Output Parameters:
+ mesh     - The mesh
. inputCtx - The structure for communicating with Triangle
- oldHoles - The previous holes, which are saved for MeshFinalizeOutput_Periodic

  Level: developer

.keywords mesh, periodic, Triangle
.seealso MeshFinalizeOutput_Periodic(), MeshInitInput_Triangle()
*/
int MeshInitInput_Periodic(Mesh mesh, struct triangulateio *inputCtx, double **oldHoles)
{
  int     numPoints = inputCtx->numberofpoints;
  double *points    = inputCtx->pointlist;
  int     numHoles  = inputCtx->numberofholes;
  double *holes     = inputCtx->holelist;
  double  r_inner   = mesh->sizeX / (2*PETSC_PI);
  double *newHoles;
  double  r, theta;
  int     rank;
  int     p, h;
  int     ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  if (rank == 0) {
    /* Map the domain on to an annulus -
       We will preserve the periodic length on the inner surface, so that

         2 \pi r_inner     = mesh->sizeX

       and we preserve the bounding rectangle, so that

         r_outer - r_inner = mesh->sizeY

       where we reverse X and Y if the domain is periodic in Y. Also, we
       let \theta map along the x-axis in the reverse direction to preserve
       the orientation of objects. The equations are therefore

         r      = r_inner + (y - mesh->startY)
         \theta = (2 \pi (mesh->sizeX - (x - mesh->startX)) / mesh->sizeX)
  */
    for(p = 0; p < numPoints; p++) {
      r             = r_inner + (points[p*2+1] - mesh->startY);
      theta         = 2*PETSC_PI * ((mesh->sizeX - (points[p*2] - mesh->startX)) / mesh->sizeX);
      points[p*2]   = r*cos(theta);
      points[p*2+1] = r*sin(theta);
    }
    /* Add the hole in the center of the annulus */
    inputCtx->numberofholes = numHoles+1;
    ierr = PetscMalloc((numHoles+1)*2 * sizeof(double), &newHoles);                                       CHKERRQ(ierr);
    for(h = 0; h < numHoles; h++) {
      r               = r_inner + (holes[h*2+1] - mesh->startY);
      theta           = 2*PETSC_PI * ((mesh->sizeX - (holes[h*2] - mesh->startX)) / mesh->sizeX);
      newHoles[h*2]   = r*cos(theta);
      newHoles[h*2+1] = r*sin(theta);
    }
    newHoles[numHoles*2]   = 0.0;
    newHoles[numHoles*2+1] = 0.0;
    inputCtx->holelist     = newHoles;
    if (oldHoles != PETSC_NULL) {
      *oldHoles            = holes;
    }
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshFinalizeOutput_Periodic"
/* MeshFinalizeOutput_Periodic
  This function initializes the input structure for Triangle in the periodic case.

  Collective on Mesh

  Output Parameters:
. mesh      - The mesh
. outputCtx - The structure for communicating with Triangle
. oldHoles  - The previous holes, which were saved in MeshInitInput_Periodic

  Level: developer

.keywords mesh, periodic, Triangle
.seealso MeshInitInput_Periodic()
*/
int MeshFinalizeOutput_Periodic(Mesh mesh, struct triangulateio *outputCtx, double **oldHoles)
{
  int     numPoints = outputCtx->numberofpoints;
  double *points    = outputCtx->pointlist;
  int    *markers   = outputCtx->pointmarkerlist;
  int     numHoles  = outputCtx->numberofholes;
  double  r_inner   = mesh->sizeX / (2*PETSC_PI);
  double  r_outer   = r_inner + mesh->sizeY;
  double  r, theta;
  int     p;
  int     ierr;

  PetscFunctionBegin;
  /* Map the domain from an annulus to a rectangle -
       The inverse map is given by

         x = mesh->sizeX ((2 \pi - \theta) / 2 \pi) + mesh->startX
         y = r - r_inner + mesh->startY
  */
  for(p = 0; p < numPoints; p++) {
    /* If points were inserted on he boundary, we must push them to the correct radius */
    if (markers[p] == 2) {        /* TOP_BD    */
      r           = r_outer;
    } else if (markers[p] == 3) { /* BOTTOM_BD */
      r           = r_inner;
    } else {
      r           = hypot(points[p*2], points[p*2+1]);
    }
    theta         = atan2(points[p*2+1], points[p*2]);
    if (theta < 0)
      theta       = 2*PETSC_PI + theta;
    points[p*2]   = ((2*PETSC_PI - theta) / (2*PETSC_PI))*mesh->sizeX + mesh->startX;
    points[p*2+1] = r - r_inner + mesh->startY;
  }
  /* Reset the holes */
  ierr = PetscFree(outputCtx->holelist);                                                                 CHKERRQ(ierr);
  outputCtx->numberofholes = numHoles-1;
  if (oldHoles != PETSC_NULL)
    outputCtx->holelist    = *oldHoles;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshCreate_Periodic"
/* MeshCreate_Periodic
  This function creates a 2D periodic unstructured mesh using Triangle.

  Collective on Mesh

  Input Parameter:
. numCorners  - The number of nodes in an element

  Input Parameters from bdCtx:
+ numBD       - The number of closed boundaries in the geometry, or different markers
. numVertices - The umber of boundary points
. vertices    - The (x,y) coordinates of the boundary points
. markers     - The boundary markers for nodes, 0 indicates an interior point, each boundary must have a different marker
. numSegments - The number of boundary segments
. segments    - The endpoints of boundary segments or PETSC_NULL
. segMarkers  - The boundary markers for each segment
. numHoles    - The number of holes
- holes       - The (x,y) coordinates of holes or PETSC_NULL

  Output Parameter:
. mesh        - The new mesh created by Triangle

  Level: developer

.keywords mesh, Triangle
.seealso MeshInitInput_Periodic(), MeshFinalizeOutput_Periodic()
*/
int MeshCreate_Periodic(MeshBoundary2D *bdCtx, int numCorners, Mesh mesh)
{
  Mesh_Triangular     *tri = (Mesh_Triangular *) mesh->data;
  struct triangulateio in;            /* Input  for Triangle mesh generator */
  struct triangulateio out;           /* Output for Triangle mesh generator */
  char                 cmd_line[256]; /* Command line for Triangle */
  double              *oldHoles;
  int                  rank;
  PetscTruth           opt;
  int                  ierr;

  PetscFunctionBegin;
  /* Initialize communication structures for Triangle */
  ierr = MeshInitInput_Triangle(&in);                                                                    CHKERRQ(ierr);
  ierr = MeshInitOutput_Triangle(&out);                                                                  CHKERRQ(ierr);

  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                               CHKERRQ(ierr);
  if (rank == 0) {
    /* Setup boundaries and holes */
    in.numberofpoints      = bdCtx->numVertices;
    if (bdCtx->numVertices > 0) {
      PetscValidScalarPointer(bdCtx->vertices);
      PetscValidIntPointer(bdCtx->markers);
      in.pointlist         = bdCtx->vertices;
      in.pointmarkerlist   = bdCtx->markers;
    }
    in.numberofsegments    = bdCtx->numSegments;
    if (bdCtx->numSegments > 0) {
      PetscValidIntPointer(bdCtx->segments);
      PetscValidIntPointer(bdCtx->segMarkers);
      in.segmentlist       = bdCtx->segments;
      in.segmentmarkerlist = bdCtx->segMarkers;
    }
    in.numberofholes       = bdCtx->numHoles;
    if (bdCtx->numHoles    > 0) {
      /* We keep these */
      PetscValidScalarPointer(bdCtx->holes);
      ierr = PetscMalloc(in.numberofholes*2 * sizeof(double), &in.holelist);                              CHKERRQ(ierr);
      ierr = PetscMemcpy(in.holelist, bdCtx->holes, in.numberofholes*2 * sizeof(double));                 CHKERRQ(ierr);
    }

    /* Generate boundary compatible with Triangle */
    ierr = MeshInitInput_Periodic(mesh, &in, &oldHoles);                                                  CHKERRQ(ierr);

    /* Generate the inital coarse mesh and check whether we should create midnodes */
    if (numCorners == 3) {
      ierr = PetscStrcpy(cmd_line, "pqcenzQ");                                                            CHKERRQ(ierr);
    } else if (numCorners == 6) {
      ierr = PetscStrcpy(cmd_line, "pqcenzo2Q");                                                          CHKERRQ(ierr);
    } else {
      SETERRQ1(PETSC_ERR_SUP, "Number of local nodes %d not supported", numCorners);
    }
    ierr = PetscOptionsGetString(PETSC_NULL, "-triangle_cmd", cmd_line, 255, &opt);                       CHKERRQ(ierr);
    triangulate(cmd_line, &in, &out, PETSC_NULL);

    /* Remap output of Triangle to the periodic domain */
    ierr = MeshFinalizeOutput_Periodic(mesh, &out, &oldHoles);                                            CHKERRQ(ierr);

    /* Get rid of extra information */
    ierr = PetscFree(out.segmentlist);                                                                    CHKERRQ(ierr);
    ierr = PetscFree(out.segmentmarkerlist);                                                              CHKERRQ(ierr);
  }

  /* Communicate values */
  ierr = MeshDistribute_Triangle(mesh, &out);                                                             CHKERRQ(ierr);

  /* Store the information */
  mesh->numBd       = bdCtx->numBd;
  mesh->numNodes    = out.numberofpoints;
  mesh->numEdges    = out.numberofedges;
  mesh->numVertices = (mesh->numNodes <= mesh->numEdges ? mesh->numNodes : mesh->numNodes - mesh->numEdges);
  mesh->numFaces    = out.numberoftriangles;
  mesh->numCorners  = out.numberofcorners;
  mesh->numHoles    = out.numberofholes;
  tri->nodes       = out.pointlist;
  tri->markers     = out.pointmarkerlist;
  tri->edges       = out.edgelist;
  tri->edgemarkers = out.edgemarkerlist;
  tri->faces       = out.trianglelist;
  tri->neighbors   = out.neighborlist;
  mesh->holes       = out.holelist;
  PetscLogObjectMemory(mesh, (mesh->numNodes*2 + mesh->numNodes) * sizeof(double));
  PetscLogObjectMemory(mesh, (mesh->numEdges*2+mesh->numEdges+mesh->numFaces*mesh->numCorners+mesh->numFaces*3) * sizeof(int));

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshRefine_Periodic"
/*
  MeshRefine_Periodic - This function refines a two dimensional periodic unstructured mesh
  with Triangle using area constraints.

  Collective on Mesh

  Input Parameters:
+ oldMesh - The mesh begin refined
- area    - A function which gives an area constraint when evaluated inside an element

  Output Parameter:
. mesh    - The refined mesh

  Level: developer

.keywords: mesh, Triangle, refine, periodic
.seealso: MeshRefine(), MeshCoarsen(), MeshReform()
*/
int MeshRefine_Periodic(Mesh oldMesh, PointFunction area, Mesh mesh)
{
  int                  rank = oldMesh->part->rank;
  Mesh_Triangular     *tri;
  struct triangulateio in;            /* Input  for Triangle mesh generator */
  struct triangulateio out;           /* Output for Triangle mesh generator */
  char                 cmd_line[256]; /* Command line for Triangle */
  MPI_Comm             comm;
  PetscTruth           opt;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscNew(Mesh_Triangular, &tri);                                                                 CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, sizeof(Mesh_Triangular));
  ierr = PetscMemcpy(mesh->ops, oldMesh->ops, sizeof(struct _MeshOps));                                   CHKERRQ(ierr);
  mesh->data             = (void *) tri;
  ierr = PetscStrallocpy(MESH_TRIANGULAR_2D,            &mesh->type_name);                                CHKERRQ(ierr);
  ierr = PetscStrallocpy(MESH_SER_TRIANGULAR_2D_BINARY, &mesh->serialize_name);                           CHKERRQ(ierr);
  mesh->dim              = 2;
  mesh->isPeriodic       = oldMesh->isPeriodic;
  mesh->isPeriodicDim[0] = oldMesh->isPeriodicDim[0];
  mesh->isPeriodicDim[1] = oldMesh->isPeriodicDim[1];
  mesh->isPeriodicDim[2] = oldMesh->isPeriodicDim[2];
  mesh->nodeOrdering     = PETSC_NULL;
  mesh->partitioned      = 0;
  mesh->highlightElement = -1;
  mesh->usr              = oldMesh->usr;

  /* Copy function list */
  if (mesh->qlist != PETSC_NULL) {
    ierr = PetscFListDestroy(&mesh->qlist);                                                               CHKERRQ(ierr);
  }
  ierr = PetscFListDuplicate(oldMesh->qlist, &mesh->qlist);                                               CHKERRQ(ierr);

  /* Initialize communication structures for Triangle */
  ierr = MeshInitRefineInput_Triangle(oldMesh, area, &in);                                                CHKERRQ(ierr);
  ierr = MeshInitInput_Periodic(oldMesh, &in, PETSC_NULL);                                                CHKERRQ(ierr);
  ierr = MeshInitOutput_Triangle(&out);                                                                   CHKERRQ(ierr);

  /* Copy extent from previous mesh: don't make separate boxes for now */
  mesh->startX    = oldMesh->startX;
  mesh->endX      = oldMesh->endX;
  mesh->startY    = oldMesh->startY;
  mesh->endY      = oldMesh->endY;
  mesh->locStartX = mesh->startX;
  mesh->locEndX   = mesh->endX;
  mesh->locStartY = mesh->startY;
  mesh->locEndY   = mesh->endY;
  mesh->sizeX     = mesh->endX    - mesh->startX;
  mesh->sizeY     = mesh->endY    - mesh->startY;
  mesh->locSizeX  = mesh->locEndX - mesh->locStartX;
  mesh->locSizeY  = mesh->locEndY - mesh->locStartY;

  /* Generate a refined mesh */
  if (rank == 0) {
    /* Generate the refined mesh and check whether we should create midnodes */
    if (oldMesh->numCorners == 3) {
      ierr = PetscStrcpy(cmd_line, "qcenzQ");                                                             CHKERRQ(ierr);
    } else if (oldMesh->numCorners == 6) {
      ierr = PetscStrcpy(cmd_line, "qcenzo2Q");                                                           CHKERRQ(ierr);
    } else {
      SETERRQ1(PETSC_ERR_SUP, "Number of local nodes %d not supported", oldMesh->numCorners);
    }
    ierr = PetscOptionsGetString("ref_", "-triangle_cmd", cmd_line, 251, &opt);                           CHKERRQ(ierr);
    ierr = PetscStrcat(cmd_line, "pra");                                                                  CHKERRQ(ierr);
    triangulate(cmd_line, &in, &out, PETSC_NULL);

    /* Remap output of Triangle to the periodic domain */
    ierr = MeshFinalizeOutput_Periodic(mesh, &out, PETSC_NULL);                                           CHKERRQ(ierr);

    /* Get rid of extra information */
    ierr = PetscFree(out.segmentlist);                                                                    CHKERRQ(ierr);
    ierr = PetscFree(out.segmentmarkerlist);                                                              CHKERRQ(ierr);
  }
  ierr = MeshFinalizeRefineInput_Triangle(oldMesh, &in);                                                  CHKERRQ(ierr);

  /* Communicate values */
  ierr = MeshDistribute_Triangle(oldMesh, &out);                                                          CHKERRQ(ierr);

  /* Store the information */
  mesh->numBd       = oldMesh->numBd;
  mesh->numNodes    = out.numberofpoints;
  mesh->numEdges    = out.numberofedges;
  mesh->numVertices = (mesh->numNodes <= mesh->numEdges ? mesh->numNodes : mesh->numNodes - mesh->numEdges);
  mesh->numFaces    = out.numberoftriangles;
  mesh->numCorners  = out.numberofcorners;
  tri->nodes       = out.pointlist;
  tri->markers     = out.pointmarkerlist;
  tri->edges       = out.edgelist;
  tri->edgemarkers = out.edgemarkerlist;
  tri->faces       = out.trianglelist;
  tri->neighbors   = out.neighborlist;
  PetscLogObjectMemory(mesh, (mesh->numNodes*2 + mesh->numNodes) * sizeof(double));
  PetscLogObjectMemory(mesh, (mesh->numEdges*2 + mesh->numEdges + mesh->numFaces*mesh->numCorners + mesh->numFaces*3) * sizeof(int));
  tri->nodesOld    = PETSC_NULL;

  /* Copy holes from previous mesh */
  mesh->numHoles    = oldMesh->numHoles;
  mesh->holes       = PETSC_NULL;
  if (mesh->numHoles > 0) {
    ierr = PetscMalloc(mesh->numHoles*2 * sizeof(double), &mesh->holes);                                    CHKERRQ(ierr);
    ierr = PetscMemcpy(mesh->holes, oldMesh->holes, mesh->numHoles*2 * sizeof(double));                      CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, oldMesh->numHoles*2 * sizeof(double));
  }

  /* Calculate maximum degree of vertices */
  ierr = MeshSetupSupport_Triangular_2D(mesh);                                                            CHKERRQ(ierr);

  /* Construct derived and boundary information */
  ierr = MeshSetupBoundary_Triangular_2D(mesh);                                                           CHKERRQ(ierr);

#ifdef PETSC_USE_BOPT_g
  /* Check mesh integrity */
  ierr = MeshDebug_Triangular_2D(mesh, PETSC_FALSE);                                                      CHKERRQ(ierr);
#endif

  /* Reorder nodes before distributing mesh */
  ierr = PetscOptionsHasName(oldMesh->prefix, "-mesh_reorder", &opt);                                     CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    /* MUST FIX: Since we have duplicated the whole mesh, we may impersonate a serial mesh */
    comm       = mesh->comm;
    mesh->comm = PETSC_COMM_SELF;
    ierr       = MeshGetOrdering(mesh, MESH_ORDER_TRIANGULAR_2D_RCM, &mesh->nodeOrdering);                CHKERRQ(ierr);
    mesh->comm = comm;

    /* Permute arrays implicitly numbered by node numbers */
    ierr = AOApplicationToPetscPermuteReal(mesh->nodeOrdering, 2, tri->nodes);                            CHKERRQ(ierr);
    ierr = AOApplicationToPetscPermuteInt(mesh->nodeOrdering, 1, tri->markers);                           CHKERRQ(ierr);
    ierr = AOApplicationToPetscPermuteInt(mesh->nodeOrdering, 1, tri->degrees);                           CHKERRQ(ierr);
    /* Renumber arrays dependent on the canonical node numbering */
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numEdges*2,                   tri->edges);      CHKERRQ(ierr);
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numFaces*oldMesh->numCorners, tri->faces);      CHKERRQ(ierr);
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numBdNodes,                   tri->bdNodes);    CHKERRQ(ierr);
  }

  /* Initialize partition */
  ierr = MeshPartition(mesh);                                                                             CHKERRQ(ierr);

  /* Reset the midnodes */
  ierr = MeshResetNodes(mesh, PETSC_TRUE);                                                                CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#endif /* PETSC_HAVE_TRIANGLE */
