/** 
 * -- Terminal Driver for K12xx-2 Emulation System @ WIN32 Offline
 *
 *  Copyright (C) Tektronix, Inc. 2001 - 2001. All rights reserved.
 *
 *  @author Tektronix CTE             @(#)  %derived_by: guidod %
 *  @version %version:  %
 *    (%date_modified: %)
 *
 *  @description
 *              This is a hybrid Terminal Driver - it uses the
 *              Event Queues of the K12xx-2 Emulation System
 *              but it spawns its own Console Threads for input
 *              and outputs directly. The WIN32 Console Thread will
 *              repackage the events as if being from the K12xx
 *              GUI Terminal application.
 *              The Drivers uses the structure of term-k12 in
 *              the first part but the output-routines have been
 *              ripped out (they were based on tqsend messages).
 *              The second part of this files uses renamed versions
 *              of the term-wincon driver that get called for
 *              actual handling of the inpupt/output functions.
 */
/*@{*/
#if defined(__version_control__) && defined(__GNUC__)
static char* id __attribute__((unused)) = 
"@(#) $Id: %full_filespec: term-k12-win.c % $ ";
#endif

/* #define _P4_SOURCE 1 */
#include <pfe/incl-sub.h>
#include <pfe/term-sub.h>
#include <pfe/term-k12.h>

/* include <wincon.h> */
#include <windows.h>

#include <ctype.h>
#include <string.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>

#include <pfe/def-pth.h>
#include <pfe/incl-sup.h>
#include <pfe/logging.h>
#include <time.h>

/* #include <winerror.h> */

/* two defines stolen from gemrpc.h which is not included here */
#ifndef FORTH_MAX_MESSAGE_SIZE
#define FORTH_MAX_MESSAGE_SIZE     0x200
#endif
#ifndef FORTH_NBR_MESSAGES
#define FORTH_NBR_MESSAGES     4
#endif

#define OUTBUF_SIZE (FORTH_MAX_MESSAGE_SIZE)
#define OUTBUF_RING (4*80*25) /* four screenful */
static long outring_size = 0; /* four screenful */

#define DATA_SAP K12_FORTH_COMMAND_SAP 
         /* has to be in consensus w/ main-k12.c */

#define K12PRIV(P) P4_K12_PRIV(P)

/* note that we can not include term-k12.h which heavily uses 
 * defines from vxworks - which we don't have ready on win32 
 * ... therefore, define our own version of k12_priv
 */
typedef struct k12_wincon
{
    struct k12_priv k12;

    PFE_THR_TYPE thread;
    p4_threadP   pfe;
    char         c[2];

    /* names like in psdk/winbase/conchar_156b.htm */
    HANDLE hStdout;
    HANDLE hStdin;
    WORD   wColor;
    WORD   wOldColorAttrs;
    DWORD  fdwOldMode;
    CHAR   AsciiChar;
    int    isNoConsole;
} k12_wincon_term;

/* #define P4_K12_PRIV(P) ((struct k12_wincon*)(P->priv)) */
#define pfeTerm ((k12_wincon_term*)(PFE.priv))

#define FENCE p4_FENCE

#ifndef WINCON_NOECHO               /* USER-CONFIG */
#define WINCON_NOECHO 1
#endif

#ifndef WINCON_KEYDOWN              /* USER-CONFIG */
#define WINCON_KEYDOWN 1
#endif


/* forward */ 
static int  c_interrupt_key (char ch);
static int  c_prepare_terminal (void);
static void c_cleanup_terminal (void);
static void c_interactive_terminal (void);
static void c_system_terminal (void);
static void c_query_winsize (void);

static int c_keypressed (void);	
static int c_getkey (void);

static void c_putc_noflush (char c);
static void c_put_flush (void);
static void c_putc (char c);
static void c_puts (const char *s);
static void c_gotoxy (int x, int y);
static void c_wherexy (int *x, int *y);
static void c_tput (int);

/* ************************************************ */

char const *
term_k12_rawkey_string [P4_NUM_KEYS]  = /* what function keys send */
{
    /* Terminal Application sends VT100 key sequences (hopefully) */
    "\033OP",   "\033OQ",   "\033OR",   "\033OS", 
    "\033[15~", "\033[17~", "\033[18~", "\033[19~", 
    "\033[20~", "\033[21~", "\033[22~", "\033[23~", 
    "\033[24~", "\033[25~", "\033[26~", "\033[27~", 
    "\033[28~", "\033[29~", "\033[30~", "\033[31~", 
    
    "\033OD",  "\033OC",    "\033OA",   "\033OB", /* le re up do */

    "\033[1~",  "\033[4~", "\033[6~",   "\033[5~", /* kh kH kN kP */
    "\b",       "\033[3~",  NULL,       "\033[2~", /* kb kD kM kI */
    NULL,       NULL,       NULL,       NULL, 
};


static int
k12_prepare_terminal (void)
{
    k12_priv* k12p = K12PRIV(p4TH);
    
    /* setbuf (stdout, NULL); */

    /* create RX_TASK */
    
    
    k12p->rx_dataSAP = DATA_SAP;
    return 1;
}

static void
k12_cleanup_terminal (void)
{
    k12_priv* k12p = K12PRIV(p4TH);

    /* delete RX_TASK */
    
    if (k12p->tx_logfile)
    {
        close (k12p->tx_logfile);
        k12p->tx_logfile = 0;
    }
}

void p4_set_fkey_name (int num, char* name);

/* handle incoming events until a terminal' key-event is received */
static int 
p4_k12_wait_for_stdin ()
{
    register k12_priv* k12p = K12PRIV(p4TH);
    k12p->answering = FALSE;
    k12p->frm_putback = 0;

    if (k12p->state == K12_EMU_NOT_LOADED)
        k12p->state = K12_EMU_IDLE;
    
#   define _sizeof_subhead_t (sizeof(k12_emu_msg_subhead_t))  
    while (1)
    {
        if (k12p->frm_putback) { k12p->frm_putback = 0; }
        else{
            k12EmuLowEventGet (k12p->emu, 
              &k12p->frm_input, 
              &k12p->frm_data, 
              &k12p->frm_datalen, 
              &k12p->frm_option);
        }
        if (k12p->frm_input == k12p->rx_dataSAP)
        {  
            register k12_emu_msg_subhead_t* msg = (void*)k12p->frm_data;
            switch (msg->type) 
            {
             case K12_EMU_DATA_REQ: 
                 k12p->rx_data = k12p->frm_data +_sizeof_subhead_t;
                 k12p->rx_datalen = k12p->frm_datalen -_sizeof_subhead_t;
                 k12p->rx_dataIN = 0;
                   
                 if (! k12p->rx_datalen || ! k12p->rx_data[0])
                 {
                     /* the k12-terminal may send a flush request 
                        as being an empty key buffer, so flush now.
                     */
                     if (k12p->bufidx)       /* | quick path: p4_put_flush */
                         p4_put_flush ();    /* | has the same if-check */

                     /* and the c_keypressed routine may send an empty
                        key buffer to detect a queue without a key-event
                        - is will also set ->para2
                     */
                     if (msg->para2)
                         return 0;
                 }else{
                     /* there are new chars to be handled (will go to TIB) */
                     return 1;
                 }
                 break;
             case K12_EMU_XDAT_REQ: /* command/answer facility */
                 k12p->rx_data = k12p->frm_data +_sizeof_subhead_t;
                 k12p->rx_datalen = k12p->frm_datalen -_sizeof_subhead_t;
                 k12p->rx_dataIN = 0;
                 
                 k12p->answering = TRUE;
                 if (!k12p->rx_datalen || !k12p->rx_data[0])
                 {
                     /* empty input */
                     k12p->rx_data = "\n";
                     k12p->rx_datalen = 1;
                     k12p->rx_dataIN = 0;
                 }
                 /* -> start command/answer facility */
                 strcpy (k12p->answerbuf, "\\ ");
                 k12p->answeridx = strlen (k12p->answerbuf);
                 return 1;
	    case K12_EMU_CONN_REQ: /* attach to new terminal */
                 k12p->rx_data = 0; 
                 k12p->rx_datalen = k12p->frm_datalen;
                 k12p->rx_dataIN = 0; 
                 k12p->tx_qid = msg->handle;
                 break;
	    case K12_EMU_DISC_REQ:  /* detach that terminal */
                 k12p->rx_data = 0; 
                 k12p->rx_datalen = k12p->frm_datalen;
                 k12p->rx_dataIN = 0; 
                 k12p->tx_qid = msg->handle;         /* should be null */
                 break;
             default: 
                 if (k12p->eventHook 
                   && 0x4e00 < msg->type && msg->type < 0x5000)
                 { (*k12p->eventHook)(
                                      k12p->frm_input, 
                                      k12p->frm_data, 
                                      k12p->frm_datalen);
                 }
            }
        }else{
            if (k12p->eventHook) 
            {
                if ( (*k12p->eventHook)(
                                        k12p->frm_input, 
                                        k12p->frm_data, 
                                        k12p->frm_datalen))
                {
                    continue;
                }
            }
        }
    }
#   undef _sizeof_subhead_t
}

/* #define NOCH ((int)0xdeadc4a3) */
#define NOCH 0

static int
c_keypressed (void)
{
    status_t e;
    int k12_sap;
    k12_emu_msg_subhead_t* buf;
    time_t sec;
    k12_priv* k12p = K12PRIV(p4TH);

    /* part A: check if there is a putback/onhold char */
    if (k12p->nxch != NOCH)
        return 1;

    /* since k12xx eventsys can not scan into the input queue
       for some kind of type (like an X11 eventqueue would do)
       we will just go to handle all request. If the event-queue
       had some events, but no key-events, the getevent routine
       would go blocking. Therefore, we feed an empty KEY-event
       to the end of the event-queue so it will definitly return 
       here, either with the actual key-buffer or with that empty
       key-buffer (length zero) that we will generate now.
    */

    /* part B: check if the last key buffer is not yet empty */
    if (k12p->rx_data && k12p->rx_dataIN < k12p->rx_datalen)
    {
        return 1;
    }
        
    /* part C: get new event-buffer, and add it to our event queue */

    if (! FENCE)
        k12_sap = K12_FORTH_COMMAND_SAP;
    else
        k12_sap = k12p->rx_dataSAP;
    
    e=k12EmuLowBufferGet (k12p->emu, sizeof(*buf), (char**) &buf);
    if (e) { 
        P4_fail ("cannot make term nonblocking "
          "because timeout buffer could not be allocated");
        return 0;
    }
    memset (buf, 0, sizeof(*buf));
    buf->type = K12_EMU_DATA_REQ;
    buf->para2 = (u32_t) buf;
    
    e=k12EmuLowEventPut (k12p->emu, k12_sap, 
      (char*) buf, sizeof(*buf), K12_EMU_NOOPT);
    if (e) {
        P4_fail ("cannot make term nonblocking "
          "because timeout buffer could not be allocated");
        return 0;
    }

    sec = time (0); /*let the loop terminate even if our event was lost*/
    do
    {
        /* part D: go to handle requests on the event-queue */
        if (PFE.wait_for_stdin)
            (*PFE.wait_for_stdin) ();
        else
        p4_k12_wait_for_stdin ();
    
        /* part E: check again if there is a non-empty key-buffer now */
        if (k12p->rx_data && k12p->rx_dataIN < k12p->rx_datalen)
        {
            return 1;
        }

        /* no new keys in event-buffer, but a key flushrequest received.
           Now check that this flushrequest is the one issued by this
           call to c_keypressed. Then return a "no new keys" flag == false;
        */
        if (((k12_emu_msg_subhead_t*)k12p->frm_data)->para2 == (u32_t) buf)
            return 0;

        /* There was an old flush request, go again to the getevent loop */
    } while (time (0) - sec <= 1);

    /* ouch, no key event received in about one..two seconds, and we did
       not see our flushrequest in that time either. Better return now. 
    */
    return 0;
}

static int
c_getkey (void)
{
    k12_priv* k12p = K12PRIV(p4TH);
    
    if (k12p->nxch != NOCH) {
        int ch = k12p->nxch;
        
        k12p->nxch = NOCH;
        return ch;
    }
    
    /* loop until a non-empty key-buffer has been received */
    while (1) 
    {
        /* check if there is a non-empty key-buffer */
        if (k12p->rx_data && k12p->rx_dataIN < k12p->rx_datalen)
        {
            register int ch = k12p->rx_data[k12p->rx_dataIN++];
            return ch;
        }else{
            /* handle requests on the event-queue and try again */
            if (PFE.wait_for_stdin)
                (*PFE.wait_for_stdin) ();
            else
                p4_k12_wait_for_stdin ();
        }
    }
}

/************************************************************************/
/* Input from screen */

static int c_interrupt_key (char ch)		{ return 0; }

static void c_query_winsize (void)		
{
    /* psdk/winbase/conchar_34dr.htm */
    CONSOLE_SCREEN_BUFFER_INFO screenInfo;
    if (! pfeTerm || pfeTerm->hStdout == INVALID_HANDLE_VALUE)
	return;
    
    if (GetConsoleScreenBufferInfo (pfeTerm->hStdout, &screenInfo))
    {
	PFE.cols = screenInfo.dwSize.X;
	PFE.rows = screenInfo.dwSize.Y;
	pfeTerm->wColor = screenInfo.wAttributes;
    }else{
	/* using defaults */
	PFE.cols = PFE.set->cols;
	PFE.rows = PFE.set->rows;
	pfeTerm->wColor = 0x7; /* white on black */
        if (pfeTerm->hStdout != INVALID_HANDLE_VALUE)
        {
            if (! pfeTerm->isNoConsole)
                P4_info ("going into NoConsole mode");
            pfeTerm->isNoConsole = 1;
        }
    }
}

static int
win_prepare_terminal (void)
{
    P4_enter ("now");
    if (!pfeTerm)
    {
	pfeTerm = calloc (1, sizeof(*pfeTerm));
	if (!pfeTerm) return 0;
	pfeTerm->hStdout = INVALID_HANDLE_VALUE;
	pfeTerm->hStdin = INVALID_HANDLE_VALUE;
        pfeTerm->isNoConsole = 0;
    }

    if (pfeTerm->hStdout == INVALID_HANDLE_VALUE)
    {
	/* should we check for --bye ? is this an implicit check     *checkme*
	 * for a kind of terminal, i.e. isatty on stdio/stdin?       *checkme* 
	 */
	pfeTerm->hStdout = GetStdHandle(STD_OUTPUT_HANDLE);
	pfeTerm->hStdin  = GetStdHandle(STD_INPUT_HANDLE);
	if (pfeTerm->hStdout == INVALID_HANDLE_VALUE ||
	    pfeTerm->hStdin  == INVALID_HANDLE_VALUE)
	{
	    P4_warn1 ("no default console window found: %ld", 
		      GetLastError());
	    pfeTerm->hStdout = pfeTerm->hStdin = INVALID_HANDLE_VALUE;
	}
    }
    
    if (pfeTerm->hStdout == INVALID_HANDLE_VALUE)
    {
	/* psdk/winbase/conchar_93n6.htm */
	pfeTerm->hStdout = CreateConsoleScreenBuffer(
	    GENERIC_READ|GENERIC_WRITE, /* access */
	    0, /* buffer share mode */
	    0, /* lpSecurityAttributes */
	    CONSOLE_TEXTMODE_BUFFER, /* buffer type (the only one possible) */
	    0 /* reserved */
	    );
	if (pfeTerm->hStdout == INVALID_HANDLE_VALUE)
	{
	    P4_fail1 ("could not open console window: %ld", 
		      GetLastError());
	    return 0;
	}
	/* psdk/winbase/conchar_9hrm.htm */
	if (! SetConsoleActiveScreenBuffer(pfeTerm->hStdout))
	{
	    P4_fail1 ("could not activate console window: %ld", 
		      GetLastError());
	    return 0;
	}
	pfeTerm->hStdin = pfeTerm->hStdout;
    }

    if (! GetConsoleMode (pfeTerm->hStdin, &pfeTerm->fdwOldMode))
    {
	P4_warn1 ("can not retrieve console window mode, guess default %ld",
		 GetLastError());
	pfeTerm->fdwOldMode = 
	    ENABLE_LINE_INPUT | 
	    ENABLE_ECHO_INPUT |
	    ENABLE_PROCESSED_INPUT ;
    }

    c_query_winsize ();
    pfeTerm->wOldColorAttrs = pfeTerm->wColor;

    P4_leave1 ("hStdout=%p", pfeTerm->hStdout);
    return 1;
}

static void
win_cleanup_terminal (void)
{
    if (! pfeTerm ) return;

    SetConsoleMode (pfeTerm->hStdin, pfeTerm->fdwOldMode);
    SetConsoleTextAttribute (pfeTerm->hStdout, pfeTerm->wOldColorAttrs);

    free (pfeTerm); pfeTerm = 0;
    return;
}

static void c_interactive_terminal (void)	
{
# if WINCON_NOECHO
    P4_enter ("now");

    if (! pfeTerm ) return;
    SetConsoleMode (pfeTerm->hStdin, 
		    pfeTerm->fdwOldMode & 
		    ~(ENABLE_LINE_INPUT | ENABLE_ECHO_INPUT));
    P4_leave ("now");
# endif
}
static void c_system_terminal (void)		
{
# if WINCON_NOECHO
    P4_enter ("now");
    if (! pfeTerm ) return;
    SetConsoleMode (pfeTerm->hStdin, pfeTerm->fdwOldMode);
    P4_leave ("now");
# endif
}

static const char*
KeyEventProc (k12_wincon_term* term, KEY_EVENT_RECORD* ev)
{
#define __KEY(X) (term_k12_rawkey_string[(X) - P4_KEY_k1])
    int k;

    if (! term ) 
	return 0;

# if WINCON_KEYDOWN
    if (! ev->bKeyDown)
	return 0;
# else
    if (ev->bKeyDown)
	return 0;
# endif

    if (ev->uChar.AsciiChar)
    {
	term->AsciiChar = 0;
	term->c[0] = ev->uChar.AsciiChar;
	term->c[1] = 0;
	return term->c;
    }

#  ifndef MOD_SHIFT
#  define MOD_SHIFT 0
#  endif
    k = ev->dwControlKeyState & MOD_SHIFT ? (P4_KEY_k1 - P4_KEY_F1) : 0;

    switch (ev->wVirtualKeyCode)
    {
    case VK_F1:  return __KEY (P4_KEY_F1 + k);
    case VK_F2:  return __KEY (P4_KEY_F2 + k);
    case VK_F3:  return __KEY (P4_KEY_F3 + k);
    case VK_F4:  return __KEY (P4_KEY_F4 + k);
    case VK_F5:  return __KEY (P4_KEY_F5 + k);
    case VK_F6:  return __KEY (P4_KEY_F6 + k);
    case VK_F7:  return __KEY (P4_KEY_F7 + k);
    case VK_F8:  return __KEY (P4_KEY_F8 + k);
    case VK_F9:  return __KEY (P4_KEY_F9 + k);
    case VK_F10: return __KEY (P4_KEY_FA + k);

    case VK_LEFT:  return __KEY (P4_KEY_kl);
    case VK_UP:    return __KEY (P4_KEY_ku);
    case VK_RIGHT: return __KEY (P4_KEY_kr);
    case VK_DOWN:  return __KEY (P4_KEY_kd);

    case VK_HOME:  return __KEY (P4_KEY_kh);
    case VK_END:   return __KEY (P4_KEY_kH);
    case VK_NEXT:  return __KEY (P4_KEY_kN);
    case VK_PRIOR: return __KEY (P4_KEY_kP);

    case VK_BACK:   return "\b";
    case VK_DELETE: return __KEY (P4_KEY_kD);
    case VK_INSERT: return __KEY (P4_KEY_kI);
#  ifdef VK_EREOF
    case VK_EREOF:  return __KEY (P4_KEY_kE);
#  endif
    case VK_CLEAR:  
    case VK_CANCEL: return __KEY (P4_KEY_kC);
    case VK_TAB:    return "\t";
    case VK_RETURN: return "\n";
    default:
	return 0;
    }
#undef __KEY
}

static char*
c_getvkey (k12_wincon_term* term)
{
    /* psdk/winbase/conchar_2nw3.htm */
    INPUT_RECORD irInBuf[1];
    DWORD cNumRead;

    if (term->AsciiChar)
    {
	term->c[0] = term->AsciiChar;
	term->c[1] = 0;
	term->AsciiChar = 0;
	return term->c;
    }

    while (1)
    {
	if (! ReadConsoleInput(
	    term->hStdin,
	    irInBuf,
	    1,
	    &cNumRead))
	{
	    P4_warn ("ReadConsoleInput Failed");
	    return 0;
	}

	if (! cNumRead) continue;

	switch (irInBuf[0].EventType)
	{
	case KEY_EVENT:
	    { 
		return KeyEventProc (term, &irInBuf[0].Event.KeyEvent);
	    }
	case WINDOW_BUFFER_SIZE_EVENT:
	    term->pfe->cols = irInBuf[0].Event.WindowBufferSizeEvent.dwSize.X;
	    term->pfe->rows = irInBuf[0].Event.WindowBufferSizeEvent.dwSize.Y;
	    break;
	case MOUSE_EVENT:
	    /* ignore */
	case FOCUS_EVENT:
		/* ignore */
	default:
	    /* ignore */
	    break;
	}
    }
}

static void*
input_thread (k12_wincon_term* term)
{
# define _sizeof_subhead_t (sizeof(k12_emu_msg_subhead_t))
    char* s;
    int l;
    s8_t* b;

    while (1) 
    {
	s = c_getvkey (term);
	if (! s) { /* SLEEP; */ continue; }
	l = strlen (s);
	k12EmuLowBufferGet (term->k12.emu, l + _sizeof_subhead_t, &b);
	((k12_emu_msg_subhead_t*)b)->type = K12_EMU_DATA_REQ;

	memcpy (b + _sizeof_subhead_t, s, l);
	k12EmuLowEventPut (term->k12.emu, term->k12.rx_dataSAP, 
			   b, l + _sizeof_subhead_t, 0);
    }
}

static int
c_prepare_terminal (void)
{
    k12_prepare_terminal ();
    win_prepare_terminal ();
    PFE_THR_SPAWN (pfeTerm->thread, input_thread, pfeTerm, 8000);
    return 1;
}

static void
c_cleanup_terminal (void)
{
    PFE_THR_KILL (pfeTerm->thread, 0);
    k12_cleanup_terminal ();
    win_cleanup_terminal ();
}

/************************************************************************/
/* Output to screen */

static void
c_putc_noflush (char c)
{
    DWORD ignore;
    if (c != '\n')
    {
        if (pfeTerm->isNoConsole)
            WriteFile (pfeTerm->hStdout, &c, 1, &ignore, 0);
        else 
            WriteConsole (pfeTerm->hStdout, &c, 1, &ignore, 0);
    }else{
        /* psdk/winbase/conchar_156b.htm */
        CONSOLE_SCREEN_BUFFER_INFO screenInfo;
        if (! GetConsoleScreenBufferInfo(pfeTerm->hStdout, &screenInfo)) 
            goto failed;
        screenInfo.dwCursorPosition.X = 0; 

        /* If it is the last line in the screen buffer, 
         * scroll the buffer up. 
         */

        if ((screenInfo.dwSize.Y-1) == screenInfo.dwCursorPosition.Y) 
        { 
            /* ScrollScreenBuffer(pfeTerm->hStdout, 1); */

            SMALL_RECT srRect;
            CHAR_INFO fill;
            COORD coord = { 0 , 0 };
            fill.Attributes = pfeTerm->wColor;
            fill.Char.AsciiChar = ' ';
            memcpy (&srRect, &screenInfo.srWindow, sizeof(srRect));
            srRect.Top++;

            ScrollConsoleScreenBuffer(
                pfeTerm->hStdout,
                &srRect,
                0,
                coord,
                &fill);
        } 

        /* Otherwise, advance the cursor to the next line. */

        else screenInfo.dwCursorPosition.Y += 1; 
 
        if (! SetConsoleCursorPosition(
            pfeTerm->hStdout, 
            screenInfo.dwCursorPosition))
            goto failed;
        return;
    failed:
        if (pfeTerm->isNoConsole)
            WriteFile (pfeTerm->hStdout, &c, 1, &ignore, 0);
        else
            WriteConsole (pfeTerm->hStdout, &c, 1, &ignore, 0);
    }
}

static void
c_put_flush (void)
{
    return;
}

static void
c_putc (char c)
{
    c_putc_noflush (c);
    c_put_flush ();
}

static void
c_puts (const char *s)
{
    /* optimize path */
    DWORD n = 0;
    while (s[n] && isprint(s[n]) && s[n] != '\n')
        n++;

    if (n)
    {
        if (pfeTerm->isNoConsole)
            WriteFile (pfeTerm->hStdout, s, n, &n, 0);
        else
            WriteConsole (pfeTerm->hStdout, s, n, &n, 0);
                
        s += n;
    }

    /* standard path */
    while (*s)
        c_putc_noflush (*s++);
    c_put_flush ();
}

static void
c_gotoxy (int x, int y)
{
    CONSOLE_SCREEN_BUFFER_INFO screenInfo;
    if (! GetConsoleScreenBufferInfo(pfeTerm->hStdout, &screenInfo)) 
        return;

    screenInfo.dwCursorPosition.X = x; 
    screenInfo.dwCursorPosition.Y = y; 

    if (! SetConsoleCursorPosition(
        pfeTerm->hStdout, 
        screenInfo.dwCursorPosition))
        return;
    return;
}

static void
c_wherexy (int *x, int *y)
{
    CONSOLE_SCREEN_BUFFER_INFO screenInfo;
    if (! GetConsoleScreenBufferInfo(pfeTerm->hStdout, &screenInfo)) 
    {
            *x = *y = 0;                        /* uargh! */
            if (! pfeTerm->isNoConsole)
            {
                P4_warn1 ("could not get cursor position: %ld",
                  GetLastError ());
            }
    }else{
        *x = screenInfo.dwCursorPosition.X; 
        *y = screenInfo.dwCursorPosition.Y; 
    }
}

static void                     /* move cursor in x and y */
addxy (int dx, int dy)
{
    CONSOLE_SCREEN_BUFFER_INFO screenInfo;
    if (! GetConsoleScreenBufferInfo(pfeTerm->hStdout, &screenInfo)) 
        return;
    
    screenInfo.dwCursorPosition.X += dx; 
    screenInfo.dwCursorPosition.Y += dy; 
    
    SetConsoleCursorPosition(
        pfeTerm->hStdout, 
        screenInfo.dwCursorPosition);
    
    p4_OUT = screenInfo.dwCursorPosition.X;
}

static void
c_clreol (void)
{
    /*FIXME: does not work */
    DWORD ignore;
    CONSOLE_SCREEN_BUFFER_INFO screenInfo;

    if (! GetConsoleScreenBufferInfo(pfeTerm->hStdout, &screenInfo)) 
        return;
    FillConsoleOutputCharacter (
        pfeTerm->hStdout,
        ' ',
        screenInfo.dwSize.X - screenInfo.dwCursorPosition.X, 
        screenInfo.dwCursorPosition,
        &ignore);
    FillConsoleOutputAttribute (
        pfeTerm->hStdout,
        pfeTerm->wColor,
        screenInfo.dwSize.X - screenInfo.dwCursorPosition.X, 
        screenInfo.dwCursorPosition,
        &ignore);
}

static void
c_clrscr (void)
{
    DWORD ignore;
    CONSOLE_SCREEN_BUFFER_INFO screenInfo;
    if (! GetConsoleScreenBufferInfo(pfeTerm->hStdout, &screenInfo)) 
        return;
    c_gotoxy (0, 0);
    FillConsoleOutputCharacter (
        pfeTerm->hStdout,
        ' ',
        screenInfo.dwSize.X*screenInfo.dwSize.Y, 
        screenInfo.dwCursorPosition,
        &ignore);
    FillConsoleOutputAttribute (
        pfeTerm->hStdout,
        pfeTerm->wColor,
        screenInfo.dwSize.X*screenInfo.dwSize.Y, 
        screenInfo.dwCursorPosition,
        &ignore);
}

static void
c_clrdown (void)
{
    int x, y, i;
    
    c_clreol ();
    c_wherexy (&x, &y);
    for (i = y + 1; i < PFE.rows; i++)
    {
        c_gotoxy (i, 0);
        c_clreol ();
    }
    c_gotoxy (x, y);
}

enum {
    none, bold, faint, italic, blink = 5,
    rapid_blink, reverse_video, concealed
};

static void
setattr (int attr)
{
    switch (attr)
    {
    case none:
        pfeTerm->wColor &= 0x77;
        break;
    case bold:
        pfeTerm->wColor |= FOREGROUND_INTENSITY;
        break;
    case italic:
        pfeTerm->wColor |= BACKGROUND_INTENSITY;
        break;
    default:
        /* ignore */
        return;
    }
    SetConsoleTextAttribute (pfeTerm->hStdout, pfeTerm->wColor);
}

static void
clrattr (int attr)
{
    switch (attr)
    {
    case bold:
        pfeTerm->wColor &=~ FOREGROUND_INTENSITY;
        break;
    case italic:
        pfeTerm->wColor &=~ BACKGROUND_INTENSITY;
        break;
    default:
        /* ignore */
        return;
    }
    SetConsoleTextAttribute (pfeTerm->hStdout, pfeTerm->wColor);
}

static void 
c_tput (int attr)
{
    switch (attr)
    {
    case P4_TERM_GOLEFT:               addxy (-1,  0); break;
    case P4_TERM_GORIGHT:              addxy ( 1,  0); break;
    case P4_TERM_GOUP:                 addxy ( 0, -1); break;
    case P4_TERM_GODOWN:               addxy ( 0,  1); break;
	
    case P4_TERM_HOME:                 c_gotoxy (0, 0); break;
    case P4_TERM_CLRSCR:               c_clrscr (); break;
    case P4_TERM_CLRDOWN:              c_clrdown (); break;
    case P4_TERM_CLREOL:               c_clreol (); break;
    case P4_TERM_BELL:                 MessageBeep (MB_ICONASTERISK); break;
	
    case P4_TERM_NORMAL:               setattr (none); break;
    case P4_TERM_BOLD_ON:              setattr (bold); break;
    case P4_TERM_BOLD_OFF:             clrattr (bold); break;
    case P4_TERM_REVERSE:              setattr (reverse_video); break;
    case P4_TERM_BRIGHT:               setattr (bold); break;
    case P4_TERM_BLINKING:             setattr (blink); break;
    case P4_TERM_UNDERLINE_ON:         setattr (italic); break;
    case P4_TERM_UNDERLINE_OFF:        clrattr (italic); break;
    default: break;
    }
}

void 
p4_set_fkey_name (int num, char* name) 
{ 
};

#ifdef USE_TERMCAP
static char const tckeycode[][3] =
{
    "k1", "k2", "k3", "k4", "k5",	/* keys in same order as enumkeycode */
    "k6", "k7", "k8", "k9", "k0",	/* from term.h */
    "F1", "F2", "F3", "F4", "F5",
    "F6", "F7", "F8", "F9", "FA",
    "kl", "kr", "ku", "kd",	/* result is just what has to be exported */
    "kh", "kH", "kN", "kP",	/* via variable rawkey_string */
    "kb", "kD", "kM", "kI",
    "kA", "kE", "kL", "kC"
};
#endif /* USE_TERMCAP */

#if defined USE_TERMCAP || defined USE_TERMINFO
static char const tcctlcode[][3] =
{
    "cm", "ho",
    "le", "nd", "up", "do",
    "cl", "cd", "ce", "bl",
    "dc", "dl",
    "sf", "sr",
    "so", "se", "us", "ue",
    "md", "mr", "mb", "me",
    "ks", "ke"
};
#endif /* USE_TERMCAP || USE_TERMINFO */

#if __STDC_VERSION__ > 199900L || (__GNUC__*100+__GNUC_MINOR__) > 206
#define INTO(x) .x =
#else
#define INTO(x) 
#endif

#define p4_term_k12_win p4_term_ios
p4_term_struct p4_term_k12_win =
{
    "term-k12-win",
    INTO(control_string) 	0,
    INTO(rawkey_string)  	term_k12_rawkey_string,
    INTO(init) 			c_prepare_terminal, 
    INTO(fini) 			c_cleanup_terminal,
    INTO(tput)			c_tput,
    
    INTO(tty_interrupt_key) 	c_interrupt_key,
    INTO(interactive_terminal)  c_interactive_terminal,
    INTO(system_terminal)   	c_system_terminal,
    INTO(query_winsize)     	c_query_winsize,
    
    INTO(c_keypressed)		c_keypressed,
    INTO(c_getkey)		c_getkey,
    INTO(c_putc_noflush)  	c_putc_noflush,
    INTO(c_put_flush)		c_put_flush,
    INTO(c_putc)		c_putc,
    INTO(c_puts)		c_puts,
    INTO(c_gotoxy)		c_gotoxy,
    INTO(c_wherexy)		c_wherexy
};

/*@}*/

/* 
 * Local variables:
 * c-file-style: "stroustrup"
 * End:
 */
