#define __USE_W32_SOCKETS
#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "sys/wcebase.h"
#include "sys/wceerror.h"
#include "sys/wcefile.h"
#include "sys/wcenetwork.h"
#include "sys/wcetrace.h"
#include "sys/io.h"

static int fdsinitialized = 0;

_fdent_t _fdtab[MAXFDS];

/* Prototypes from local.h that probably shouldn't be here.. */
extern int __sclose(void *);
extern _READ_WRITE_RETURN_TYPE __sread(void *, char *, int);
extern _READ_WRITE_RETURN_TYPE __swrite(void *, char const *, int);
extern fpos_t __sseek(void *, fpos_t, int);

void
_initfds()
{
  HANDLE a, b, c;
  int i;

  if (fdsinitialized)
    return;
  fdsinitialized = 1;

  for (i = 0; i < MAXFDS; i++)
    _fdtab[i].fd = -1;
}

void
_initstdio()
{
  _initstdfd(stdin,  0, (HANDLE)_fileno(_getstdfilex(0)), __SRD);
  _initstdfd(stdout, 1, (HANDLE)_fileno(_getstdfilex(1)), __SWR);
  _initstdfd(stderr, 2, (HANDLE)_fileno(_getstdfilex(2)), __SWR);
}

int
_getnewfd()
{
  int i;

  for (i = 0; i < MAXFDS; i++) {
    if (_fdtab[i].fd == -1)	{
   	  _fdtab[i].flags = 0;
   	  return i;
   	}
  }

  WCETRACE(WCE_IO, "Out of file descriptors!");

  return(-1);
}

void
_setfd(int fd, int type, HANDLE hnd, int flags)
{
  _fdtab[fd].fd = fd;
  _fdtab[fd].type = type;
  _fdtab[fd].hnd = hnd;
  _fdtab[fd].flags = flags;
}

int
_assignfd(int type, HANDLE hnd, int flags)
{
  int fd;

  WCETRACE(WCE_IO, "_assignfd(%x)", hnd);

  if ((fd = _getnewfd()) >= 0)
    _setfd(fd, type, hnd, flags);

  WCETRACE(WCE_IO, "_assignfd returns %d", fd);
  return(fd);
}

void
_initstdfd(FILE *fp, int fd, HANDLE hnd, int flags)
{

  if (fd < 0 || fd > 2 || fp == NULL)
    return;

  _setfd(fd, IO_FILE_TYPE_CONSOLE, hnd, 0);

  WCETRACE(WCE_IO, "_initstdfd: fd %d hnd %x", fd, hnd);

  fp->_file = fd;
  fp->_flags = flags;
  fp->_cookie = (_PTR) fp;
  fp->_read = __sread;
  fp->_write = __swrite;
  fp->_seek = __sseek;
  fp->_close = __sclose;

#ifdef __SCLE
  if (__stextmode(fp->_file))
    fp->_flags |= __SCLE;
#endif
}

int
_open_r(struct _reent *reent, const char *path, int flags, int mode)
{
  wchar_t wpath[MAX_PATH];
  char newpath[MAX_PATH];
  HANDLE hnd;
  DWORD fileaccess;
  DWORD fileshare; 
  DWORD filecreate;
  DWORD fileattrib;
  int fd;

  WCETRACE(WCE_IO, "open(%s, %x, %o)", path, flags, mode);

  _initfds();

  if (strlen(path) >= MAX_PATH) {
    WCETRACE(WCE_IO, "open fails, invalid path\n");
    return(-1);
  }

  strcpy(newpath, path);
  mbstowcs(wpath, newpath, strlen(newpath) + 1);

  fileshare = FILE_SHARE_READ|FILE_SHARE_WRITE;
  fileattrib = FILE_ATTRIBUTE_NORMAL;

  switch (flags & (O_RDONLY | O_WRONLY | O_RDWR)) {
  case O_RDONLY:              /* read access */
    fileaccess = GENERIC_READ;
    break;
  case O_WRONLY:              /* write access */
    fileaccess = GENERIC_WRITE;
    break;
  case O_RDWR:                /* read and write access */
    fileaccess = GENERIC_READ | GENERIC_WRITE;
    break;
  default:                    /* error, bad flags */
    errno = EINVAL;
    return -1;
  }

  switch (flags & (O_CREAT | O_EXCL | O_TRUNC)) {
  case 0:
  case O_EXCL:                /* ignore EXCL w/o CREAT */
    filecreate = OPEN_EXISTING;
    break;
  case O_CREAT:
    filecreate = OPEN_ALWAYS;
    break;
  case O_CREAT | O_EXCL:
  case O_CREAT | O_TRUNC | O_EXCL:
    filecreate = CREATE_NEW;
    break;

  case O_TRUNC:
  case O_TRUNC | O_EXCL:      /* ignore EXCL w/o CREAT */
    filecreate = TRUNCATE_EXISTING;
    break;
  case O_CREAT | O_TRUNC:
    filecreate = CREATE_ALWAYS;
    break;
  default:
    /* this can't happen ... all cases are covered */
    errno = EINVAL;
    return(-1);
  }

  if ((hnd = CreateFileW(wpath, fileaccess, fileshare, NULL, filecreate,
                         fileattrib, NULL)) == INVALID_HANDLE_VALUE) {
    errno = _winerr2errno(GetLastError());
    WCETRACE(WCE_IO, "CreateFile(%s): errno=%d oserr=%d\n", newpath, errno, GetLastError());
    return(-1);
  }

  fd = _assignfd(IO_FILE_TYPE_FILE, hnd, 0);

  if (fd < 0)
    return(-1);

  if (flags & O_APPEND) {
    SetFilePointer(hnd, 0, NULL, FILE_END);
  }

  WCETRACE(WCE_IO, "open returns %d (hnd %x)", fd, hnd);
  return fd;
}

int
_close_r(struct _reent *reent, int fd)
{
  WCETRACE(WCE_IO, "close(%d)", fd);

  FDCHECK(fd);

  if (_fdtab[fd].type == IO_FILE_TYPE_FILE) {
    CloseHandle(_fdtab[fd].hnd);
  } else if (_fdtab[fd].type == IO_FILE_TYPE_SOCKET) {
    M$_closesocket((SOCKET) _fdtab[fd].hnd);
  } else if(_fdtab[fd].type == IO_FILE_TYPE_NULL) {
  }

  _fdtab[fd].fd = -1;

  return 0;
}

int
_read_r(struct _reent *reent, int fd, char *buf, int count)
{
  int nread;
  int error;

  WCETRACE(WCE_IO, "read(fd = %d, count = %d, hnd %x)", fd, count, _fdtab[fd].hnd);

  FDCHECK(fd);

  if (_fdtab[fd].type == IO_FILE_TYPE_FILE || _fdtab[fd].type == IO_FILE_TYPE_CONSOLE) {
    if (ReadFile(_fdtab[fd].hnd, buf, count, (DWORD *)&nread, NULL) == FALSE) {
      WCETRACE(WCE_IO, "ReadFile: %d", GetLastError());
      errno = EIO;
      return(-1);
    }
  } else if (_fdtab[fd].type == IO_FILE_TYPE_SOCKET) {
    if ((nread = M$_recv((SOCKET) _fdtab[fd].hnd, buf, count, 0)) == SOCKET_ERROR) {
      error = WSAGetLastError();
      WCETRACE(WCE_IO, "read: recv failed %d\n", error);
      if (error == WSAEWOULDBLOCK) {
        errno = EAGAIN;
        return(-1);
      }

      errno = _winerr2errno(error);
      return(-1);
    }
  } else if (_fdtab[fd].type == IO_FILE_TYPE_NULL) {
    WCETRACE(WCE_IO, "warning - read called w/IO_FILE_TYPE_NULL");
    nread = 0;
  }

  return(nread);
}

int
_write_r(struct _reent *reent, int fd, char *buf, int count){
  int nwritten = 0;
  int werr;

  WCETRACE(WCE_IO, "write(%d, %d, %x)", fd, count, _fdtab[fd].hnd);

  if (_fdtab[fd].type == IO_FILE_TYPE_FILE || _fdtab[fd].type == IO_FILE_TYPE_CONSOLE) {
    if (WriteFile(_fdtab[fd].hnd, buf, count, (DWORD *)&nwritten, NULL) == FALSE) {
      WCETRACE(WCE_IO, "WriteFile: hnd %x error %d\n", _fdtab[fd].hnd, GetLastError());
      errno = EIO;
      return(-1);
    }
  } else if (_fdtab[fd].type == IO_FILE_TYPE_SOCKET) {
    if ((nwritten = (int)M$_send((SOCKET)_fdtab[fd].hnd, buf, count, 0)) == SOCKET_ERROR) {
      werr = WSAGetLastError();
      WCETRACE(WCE_IO, "send: sock %d error %d",  _fdtab[fd].hnd, werr);
      errno = _winerr2errno(werr);
      return(-1);
    }
  } else if (_fdtab[fd].type == IO_FILE_TYPE_NULL) {
    WCETRACE(WCE_IO, "warning - write called w/IO_FILE_TYPE_NULL");
    nwritten = count;
  }

  return nwritten;
}

off_t
_lseek_r(struct _reent *reent, int fd, off_t offset, int whence) {
  off_t newpos;
  int method;
  WCETRACE(WCE_IO, "lseek(%d, %d, %d)", fd, offset, whence);

  FDCHECK(fd);

  switch (whence) {
  case SEEK_SET:
    method = FILE_BEGIN;
    break;
  case SEEK_CUR:
    method = FILE_CURRENT;
    break;
  case SEEK_END:
    method = FILE_END;
    break;
  default:
    method = FILE_BEGIN;
  }

  if (_fdtab[fd].type == IO_FILE_TYPE_FILE) {
    if ((newpos = SetFilePointer(_fdtab[fd].hnd, (LONG)offset, NULL, (DWORD)method)) == -1) {
      WCETRACE(WCE_IO, "SetFilePointer(%x): error %d", _fdtab[fd].hnd, GetLastError());
      errno = EIO;
      newpos = -1;
    } 
  } else {
    errno = EINVAL;
    newpos = -1;
  }

  WCETRACE(WCE_IO, "lseek returns %d", newpos);
  return(newpos);
}

int
_getpid_r(struct _reent *reent)
{
  int pid;

  pid = GetCurrentProcessId();
  return(pid);
}

int
isatty(int fd)
{
  WCETRACE(WCE_IO, "isatty(%d)", fd);

  if (!fdsinitialized)
    return(FALSE);

  if (_fdtab[fd].type == IO_FILE_TYPE_CONSOLE) {
    WCETRACE(WCE_IO, "isatty(%d): yes", fd);
    return(TRUE);
  }

  WCETRACE(WCE_IO, "isatty(%d): no", fd);
  return(FALSE);
}


