/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxdef-globals.h"

#include "polyxdef-ui-fragspecs.h"
#include "polyxdef-ui-polchemdef.h"



gboolean 
polyxdef_ui_polchemdef_wnd_setup_fragspecs_notebook_page (PxmDefCtxt *defctxt,
							  GladeXML *xml)
{
  GtkWidget *widget = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);
  


  /* The fragspecs' notebook page must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "fragspecs_available_vbox");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "fragspecs_available_vbox", widget);

  if (FALSE == 
      polyxdef_ui_polchemdef_wnd_setup_fragspecs_treeview (defctxt))
    {
      g_critical (_("%s@%d: failed to set up the fragspecs' treeview\n"),
	     __FILE__, __LINE__);
      
      return FALSE;
    }

  widget = glade_xml_get_widget (xml, "polyxdef_fragspec_add_fgs_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_fragspec_add_fgs_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_fragspec_add_fgs_button),
		    defctxt);
  
  widget = glade_xml_get_widget (xml, "polyxdef_fragspec_add_fgr_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_fragspec_add_fgr_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_fragspec_add_fgr_button),
		    defctxt);
  
  widget = glade_xml_get_widget (xml, "polyxdef_fragspec_remove_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_fragspec_remove_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_fragspec_remove_button),
		    defctxt);
  
  
  return TRUE;
}


gboolean
polyxdef_ui_polchemdef_wnd_setup_fragspecs_treeview (PxmDefCtxt *defctxt)
{
  GtkWidget *vbox = NULL;

  GPtrArray *fragspecGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;

  gint col_offset;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  fragspecGPA = defctxt->polchemdef->fragspecGPA;
  g_assert (fragspecGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			   "fragspecs_available_vbox");
  g_assert (vbox != NULL);
  

  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxdef_ui_polchemdef_wnd_create_fragspecs_treeview_model 
    (fragspecGPA);


  /* Set to the model a datum with a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "window", defctxt->definition_wnd);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "fragspec_treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "fragspec_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  

  /* Fragspec name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGS_NAME);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, 
						 "text",
						 
						 COLUMN_FGS_NAME,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /* Fragspec end column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGS_END);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("End"),
						 renderer, 
						 "text",
						 
						 COLUMN_FGS_END,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /* Fragspec actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", (gint *) 
		     COLUMN_FGS_ACTFORM);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Actform"),
						 renderer, "text",

						 COLUMN_FGS_ACTFORM,

						 "visible",
						 COLUMN_FGS_VISIBLE,

						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragspec comment column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", (gint *) 
		     COLUMN_FGS_COMMENT);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Comment"),
						 renderer, "text",

						 COLUMN_FGS_COMMENT,

						 "visible",
						 COLUMN_FGS_VISIBLE,

						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_NAME);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, "text",
						 
						 COLUMN_FGR_NAME,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Fragrule prev column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_PREV);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Prev"),
						 renderer, "text",
						 
						 COLUMN_FGR_PREV,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Fragrule this column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_THIS);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("This"),
						 renderer, "text",
						 
						 COLUMN_FGR_THIS,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Fragrule next column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_NEXT);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Next"),
						 renderer, "text",
						 
						 COLUMN_FGR_NEXT,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Fragrule actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_ACTFORM);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Actform"),
						 renderer, "text",
						 
						 COLUMN_FGR_ACTFORM,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Fragrule comment column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_COMMENT);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Comment"),
						 renderer, "text",
						 
						 COLUMN_FGR_COMMENT,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}



GtkTreeModel *
polyxdef_ui_polchemdef_wnd_create_fragspecs_treeview_model (GPtrArray *fragspecGPA)
{
  GtkTreeStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;
  gint jter = 0;

  gchar *help = NULL;
  
  PxmFragSpec *fgs = NULL;
  PxmFragRule *fgr = NULL;
  
  model = gtk_tree_store_new (COLUMN_FGS_COL_COUNT,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN);


  /* Add data for each fragspec to the tree store.
   */
  for (iter = 0 ; iter < fragspecGPA->len ; iter++)
    {
      fgs = g_ptr_array_index (fragspecGPA, iter);
      g_assert (fgs != NULL);
      
      gtk_tree_store_append (model, &tree_iter, NULL);

      if (fgs->end == PXM_FRAG_END_LEFT)
	help = g_strdup ("LE");
      else if (fgs->end == PXM_FRAG_END_RIGHT)
	help = g_strdup ("RE");
      else if (fgs->end == PXM_FRAG_END_NONE)
	help = g_strdup ("NE");
      else
	g_critical (_("%s@%d: incorrect fragspec 'end' member\n"),
	       __FILE__, __LINE__);

      gtk_tree_store_set 
	(model, &tree_iter,
	 /* fragspec data */
	 COLUMN_FGS_NAME, fgs->name,
	 COLUMN_FGS_END, help,
	 COLUMN_FGS_ACTFORM, (fgs->actform != NULL)? fgs->actform : "",
	 COLUMN_FGS_COMMENT, (fgs->comment != NULL)? fgs->comment : "",
	 /* fragrule data */
	 COLUMN_FGR_NAME, "",
	 COLUMN_FGR_PREV, "",
	 COLUMN_FGR_THIS, "",
	 COLUMN_FGR_NEXT, "",
	 COLUMN_FGR_ACTFORM, "",
	 COLUMN_FGR_COMMENT, "",
	 /* column behaviour */
	 COLUMN_FGS_VISIBLE, TRUE,
	 COLUMN_FGS_EDITABLE, TRUE,
	 -1);

      g_free (help);
      
      /* Add children, namely cleaverules of current fragspec, if any.
       */
      for (jter = 0 ; jter < fgs->fgrGPA->len ; jter++)
	{
	  GtkTreeIter tree_jter;
	  
	  fgr = g_ptr_array_index (fgs->fgrGPA, jter);
	  g_assert (fgr != NULL);
	  
	  gtk_tree_store_append (model, &tree_jter, &tree_iter);

	  gtk_tree_store_set 
	    (model, &tree_jter,
	     /* fragspec data */
	     COLUMN_FGS_NAME, "",
	     COLUMN_FGS_END, "",
	     COLUMN_FGS_ACTFORM, "",
	     COLUMN_FGS_COMMENT, "",
	     /* fragrule data */
	     COLUMN_FGR_NAME, fgr->name,
	     COLUMN_FGR_PREV, (fgr->prev != NULL)? fgr->prev : "",
	     COLUMN_FGR_THIS, (fgr->this != NULL)? fgr->this : "",
	     COLUMN_FGR_NEXT, (fgr->next != NULL)? fgr->next : "",
	     COLUMN_FGR_ACTFORM, fgr->actform,
	     COLUMN_FGR_COMMENT, (fgr->comment != NULL)? fgr->comment : "",
	     /* column behaviour */
	     COLUMN_FGS_VISIBLE, TRUE,
	     COLUMN_FGS_EDITABLE, TRUE,
	     -1);
	}
    }

  return GTK_TREE_MODEL (model);
}


void
polyxdef_ui_polchemdef_wnd_fragspecs_cell_edited (GtkCellRendererText *cell,
						  const gchar *path_string,
						  const gchar *new_text,
						  gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmPolchemdef *polchemdef = NULL;
  PxmFragSpec *fgs = NULL;
  PxmFragRule *fgr = NULL;
  
  GPtrArray *fragspecGPA = NULL;

  gint *column = NULL;
  gint idx_fgs = -1;
  gint idx_fgr = -1;
  gint depth = -1;
  
  gint result = 0;
    
  gchar *old_text = NULL;
  gchar *help = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  polchemdef = defctxt->polchemdef;
  g_assert (polchemdef != NULL);
  
  fragspecGPA = polchemdef->fragspecGPA;
  g_assert (fragspecGPA != NULL);

  g_assert (defctxt->polchemdef->atomGPA != NULL);

  


  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "fragspec_treeview_model");
  g_assert (model != NULL);

  column = g_object_get_data (G_OBJECT (cell), "column");

  gtk_tree_model_get_iter (model, &tree_iter, path);

  depth = gtk_tree_path_get_depth (path);

  switch (GPOINTER_TO_INT (column))
    {
    case COLUMN_FGS_NAME:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragspec row,
	 * and not a fragrule row.
	 */
	if (depth > 1)
	  {
	    help = 
	      g_strdup_printf (_("Select or create a fragspec first"));
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return;
	  }
	
	/* Before proceeding and accepting the new text, we should 
	 * make sure that no other fragspec already exists by the same
	 * name.
	 */
	result = 
	  pxmchem_fragspec_get_index_by_name ((gchar *) new_text, 
					      fragspecGPA);
	
	if (-1 != result)
	  {
	    help = g_strdup_printf (_("A fragspec by the same name"
				      " exists already: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	pxmchem_fragspec_set_name (fgs, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGS_END:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragspec row,
	 * and not a fragrule row.
	 */
	if (depth > 1)
	  {
	    return;
	  }
	
	/* Make sure the string is either "LE", "RE" or "NE".
	 */
	if (0 != strcmp ((gchar *) new_text, "LE")
	    && 0 != strcmp ((gchar *) new_text, "RE")
	    && 0 != strcmp ((gchar *) new_text, "NE"))
	  {
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       _("The 'End' datum must be"
		 " in: 'LE','RE','NE'"),
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	   
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);
	
	/* Get the index of the item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	if (0 == strcmp ((gchar *) new_text, "LE"))
	  pxmchem_fragspec_set_end (fgs, PXM_FRAG_END_LEFT);
	else if (0 == strcmp ((gchar *) new_text, "RE"))
	  pxmchem_fragspec_set_end (fgs, PXM_FRAG_END_RIGHT);
	else 
	  pxmchem_fragspec_set_end (fgs, PXM_FRAG_END_NONE);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGS_ACTFORM:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragspec row,
	 * and not a fragrule row.
	 */
	if (depth > 1)
	  {
	    return;
	  }
		
	/* Before proceeding and accepting the new text, we should
	 * check that the actform in it is valid. Only if the new_text
	 * has a length, and is different than the '~' that we have
	 * put in the treeview when filling it. This is because the
	 * DTD stipulates that the actform of a fragspec can be
	 * omitted (which is NULL or empty in the structure), and an
	 * emtpy actform would not validate below.
	 */
	result = TRUE;

	result = pxmchem_actform_check ((gchar *) new_text,
					defctxt->polchemdef->atomGPA);
	
	if (FALSE == result)
	  {
	    help = g_strdup_printf (_("The new actform"
				      " is invalid: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);

	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement. However,
	   since the actform can be either NULL or empty, we have to make
	   sure that the proper value get in the fgs->actform.
	*/
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	
	g_free (old_text);
	
	/* Get the index of the item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);

	pxmchem_fragspec_set_actform (fgs, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGS_COMMENT:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragspec row,
	 * and not a fragrule row.
	 */
	if (depth > 1)
	  {
	    return;
	  }
	
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	pxmchem_fragspec_set_comment (fgs, (gchar *) new_text);


	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGR_NAME:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragrule row,
	 * and not a fragspec row.
	 */
	if (depth != 2)
	  {
	    return;
	  }
	
	/* Before proceeding and accepting the new text, we should 
	 * make sure that no other fragrule already exists by the same
	 * name in the fgrGPA array belonging to the fragspec.
	 */

	/* Get the index of the fragspec (main) item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	result = 
	  pxmchem_fragrule_get_index_by_name ((gchar *) new_text,
					      fgs->fgrGPA);
	
	if (-1 != result)
	  {
	    help = g_strdup_printf (_("A fragrule by the same name"
				      " exists already: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the fragrule (sub) item that is edited.
	 */
	idx_fgr = gtk_tree_path_get_indices (path) [1];

	fgr = g_ptr_array_index (fgs->fgrGPA, idx_fgr);
	g_assert (fgr != NULL);

	pxmchem_fragrule_set_name (fgr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGR_PREV:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragrule row,
	 * and not a fragspec row.
	 */
	if (depth != 2)
	  {
	    return;
	  }
	
	/* But first yet, verify that the monomer code is existing in
	 * the polymer chemistry definition ! The DTD stipulates that the
	 * monomer code for PREV, like for THIS and NEXT may be empty!
	 */
	if (strlen (new_text) > 0 
	    &&
	    -1 == 
	    pxmchem_monomer_get_index_by_code ((gchar *) new_text,
					       polchemdef->monomerGPA))
	  {
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       _("Code unknown"), 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    return ;
	  }

	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the fragspec (main) item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	/* Get the index of the fragrule (sub) item that is edited.
	 */
	idx_fgr = gtk_tree_path_get_indices (path) [1];

	fgr = g_ptr_array_index (fgs->fgrGPA, idx_fgr);
	g_assert (fgr != NULL);

	pxmchem_fragrule_set_prev (fgr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGR_THIS:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragrule row,
	 * and not a fragspec row.
	 */
	if (depth != 2)
	  {
	    return;
	  }
	
	/* But first yet, verify that the monomer code is existing in
	 * the polymer chemistry definition ! The DTD stipulates that the
	 * monomer code for PREV, like for THIS and NEXT may be empty!
	 */
	if (strlen (new_text) > 0 
	    &&
	    -1 == 
	    pxmchem_monomer_get_index_by_code ((gchar *) new_text,
					       polchemdef->monomerGPA))
	  {
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       _("Code unknown"), 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    return ;
	  }

	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the fragspec (main) item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	/* Get the index of the fragrule (sub) item that is edited.
	 */
	idx_fgr = gtk_tree_path_get_indices (path) [1];

	fgr = g_ptr_array_index (fgs->fgrGPA, idx_fgr);
	g_assert (fgr != NULL);

	pxmchem_fragrule_set_this (fgr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGR_NEXT:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragrule row,
	 * and not a fragspec row.
	 */
	if (depth != 2)
	  {
	    return;
	  }
	
	/* But first yet, verify that the monomer code is existing in
	 * the polymer chemistry definition ! The DTD stipulates that the
	 * monomer code for PREV, like for THIS and NEXT may be empty!
	 */
	if (strlen (new_text) > 0 
	    &&
	    -1 == 
	    pxmchem_monomer_get_index_by_code ((gchar *) new_text,
					       polchemdef->monomerGPA))
	  {
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       _("Code unknown"), 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    return ;
	  }

	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the fragspec (main) item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	/* Get the index of the fragrule (sub) item that is edited.
	 */
	idx_fgr = gtk_tree_path_get_indices (path) [1];

	fgr = g_ptr_array_index (fgs->fgrGPA, idx_fgr);
	g_assert (fgr != NULL);

	pxmchem_fragrule_set_next (fgr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGR_ACTFORM:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragrule row,
	 * and not a fragspec row.
	 */
	if (depth != 2)
	  {
	    return;
	  }
	
	result = pxmchem_actform_check ((gchar *) new_text,
					defctxt->polchemdef->atomGPA);

	if (FALSE == result)
	  {
	    help = g_strdup_printf (_("The new actform"
				      " is invalid: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);

	    g_free (help);
	    
	    return ;
	  }

	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the fragspec (main) item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	/* Get the index of the fragrule (sub) item that is edited.
	 */
	idx_fgr = gtk_tree_path_get_indices (path) [1];

	fgr = g_ptr_array_index (fgs->fgrGPA, idx_fgr);
	g_assert (fgr != NULL);

	pxmchem_fragrule_set_actform (fgr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_FGR_COMMENT:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a fragrule row,
	 * and not a fragspec row.
	 */
	if (depth != 2)
	  {
	    return;
	  }
	
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the fragspec (main) item that is edited.
	 */
	idx_fgs = gtk_tree_path_get_indices (path) [0];

	fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
	
	/* Get the index of the fragrule (sub) item that is edited.
	 */
	idx_fgr = gtk_tree_path_get_indices (path) [1];

	fgr = g_ptr_array_index (fgs->fgrGPA, idx_fgr);
	g_assert (fgr != NULL);

	pxmchem_fragrule_set_comment (fgr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    }

  gtk_tree_path_free (path);
}


void
polyxdef_ui_polchemdef_wnd_fragspec_add_fgs_button (GtkWidget *widget,
						    gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeViewColumn* column = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;
  GtkTreeIter tree_iter_parent;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_fgs = -1;
  gint depth = -1;
      
  PxmPolchemdef *polchemdef = NULL;

  PxmFragSpec *fgs = NULL;
  PxmFragSpec *fgs_test = NULL;
  
  GPtrArray *fragspecGPA = NULL;


  /****************** GENERAL PHILOSOPHY **************************/

  /* A fragspec is made with the following intrinsic members: 
   * name, end, actform and comment.
   *
   * It may also contain any number of fragrules.
   * 
   * If a fragspec contains fragrule objects, they are represented
   * as items in a treenode at a depth incremented by one.
   */

  /* That means that depending on the treeview situation, the actions
     that we need to take in response to the clicking onto the ADD
     button are different.
  */

  /* If no selection is made at all, just append a new fragspec at the
     end of the treeview.
    
     If a selection is made:
   
     - if the selected item is a fragspec row, then a new fragspec
     row is to be added after this selected fragspec row.
	 
     - if the selected row is a fragrule row, then a new fragspec row
     is to be added after the fragspec row of which the fragrule is
     currently selected.
  */

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  polchemdef = defctxt->polchemdef;
  g_assert (polchemdef != NULL);
  
  fragspecGPA = polchemdef->fragspecGPA;
  g_assert (fragspecGPA != NULL);
  


  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "fragspec_treeview_model");
  g_assert (model != NULL);


  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), "fragspec_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, that means that the user
   * wants to add a fragspec item and not a fragrule.
   */
  if (result == FALSE)
    {
      /* Allocate a new fragspec so that we can later fill its
       * member data.
       */
      fgs = pxmchem_fragspec_new ();
      
      pxmchem_fragspec_set_name (fgs, _("enter name here"));
      pxmchem_fragspec_set_end (fgs, PXM_FRAG_END_NONE);
      pxmchem_fragspec_set_actform (fgs, _("enter actform here"));
      pxmchem_fragspec_set_comment (fgs, _("enter comment here"));
      
      g_ptr_array_add (fragspecGPA, fgs);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
      
      gtk_tree_store_append (GTK_TREE_STORE (model), &tree_iter, NULL);
      
      gtk_tree_store_set 
	((GtkTreeStore *) model, &tree_iter,

	 /* fragspec data */
	 COLUMN_FGS_NAME, fgs->name,
	 COLUMN_FGS_END, "NE",
	 COLUMN_FGS_ACTFORM, fgs->actform,
	 COLUMN_FGS_COMMENT, fgs->comment,

	 /* fragrule data */
	 COLUMN_FGR_NAME, "",
	 COLUMN_FGR_PREV, "",
	 COLUMN_FGR_THIS, "",
	 COLUMN_FGR_NEXT, "",
	 COLUMN_FGR_ACTFORM, "",
	 COLUMN_FGR_COMMENT, "",

	 /* column behaviour */
	 COLUMN_FGS_VISIBLE, TRUE,
	 COLUMN_FGS_EDITABLE, TRUE,
	 -1);
      
      /* And now we want to ease the user's work on modifying the fake
	 fragspec values into more useful ones: go to the place where
	 the item was stored in the tree store, and put the cursor on the
	 right row and put the keyboard focus on it also, so that it get
	 highlighted...
      */
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				      &tree_iter);
      g_assert (path != NULL);
      
      /*
	{
	gchar *help = NULL;
	help =  gtk_tree_path_to_string  (path);
	debug_printf (("path = %s\n", help));
	g_free (help);
	}
      */
      
      gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				    path);
      
      column = gtk_tree_view_get_column (treeview, COLUMN_FGS_NAME);
      
      gtk_tree_selection_select_iter (tree_selection, &tree_iter);
      
      gtk_tree_view_set_cursor_on_cell (treeview,
					path,
					column,
					NULL,
					FALSE);
      
      gtk_widget_grab_focus (GTK_WIDGET (treeview));
      
      gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				    path,
				    column,
				    TRUE,
				    0.5,
				    0.5);
      gtk_tree_path_free (path);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd,
	 _("Added new fragspec;"
	   " set its data!"), 
	 POLYXMASS_LONG_MSG_TIMEOUT); 
      return;
    }
  
  /* There is currently a selection, that means that the user wants to
     add an item. We'll have to decide if it is a fragspec or a
     fragrule.
  */

  /* Get the path to the selected item, which can be a fragspec row or
     a fragrule row, which are two very different cases that we must
     handle. Specifically, if depth == 1, the selected row is a
     fragspec, if depth == 2, then it is a fragrule.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
  
  /* Get the depth of the currently selected item, which is 1 if it is
     a fragspec item or 2 if it is a fragrule item.
  */
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth=%d\n", depth));
  */
  
  /* Whatever the depth, we can get the index of the fragspec that is
     concerned, be it selected or one of its fragrule children.
  */
  idx_fgs = gtk_tree_path_get_indices (path) [0];
   
  /*
    debug_printf (("idx_fgs=%d\n", idx_fgs);
  */
    
  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Now, if the depth == 2, that means that the item that is selected
     is a fragrule. Since we only want to add a new fragspec, we
     understand that the user wants to insert/add a fgs object after
     the fragspec object that is actually the parent of the currently
     selected fragrule.
  */
  if (depth == 2)
    {
      result = gtk_tree_model_iter_parent (GTK_TREE_MODEL (model), 
					   &tree_iter_parent, 
					   &tree_iter_sel);
      g_assert (result != FALSE);
      
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_parent);
    }
  else
    {
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_sel);
    }
  
  /* Allocate a new fragspec so that we can later fill its member
     data.
  */
  fgs = pxmchem_fragspec_new ();
  
  pxmchem_fragspec_set_name (fgs, _("enter name here"));
  pxmchem_fragspec_set_end (fgs, PXM_FRAG_END_NONE);
  pxmchem_fragspec_set_actform (fgs, _("enter actform here"));
  pxmchem_fragspec_set_comment (fgs, _("enter comment here"));

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  /* Get the index at which we will store the data. This index is the
     one that we will use in order to insert/add the newly allocated
     cls to fragspecGPA.
  */
  idx_fgs = gtk_tree_path_get_indices (path) [0];

  if (idx_fgs + 1 >= fragspecGPA->len)
    {
      g_ptr_array_add (fragspecGPA, fgs);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else if (idx_fgs + 1 < fragspecGPA->len)
    {
      fgs_test = g_ptr_array_insert_val (fragspecGPA, idx_fgs, fgs);
      g_assert (fgs == fgs_test);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else
    {
      g_error (_("%s@%d: failed to insert/add fragspec item to array\n"),
	     __FILE__, __LINE__);
    }
  
  gtk_tree_path_free (path);

  gtk_tree_store_set 
    ((GtkTreeStore *) model, &tree_iter,

     /* fragspec data */
     COLUMN_FGS_NAME, fgs->name,
     COLUMN_FGS_END, "NE",
     COLUMN_FGS_ACTFORM, fgs->actform,
     COLUMN_FGS_COMMENT, fgs->comment,

     /* fragrule data */
     COLUMN_FGR_NAME, "",
     COLUMN_FGR_PREV, "",
     COLUMN_FGR_THIS, "",
     COLUMN_FGR_NEXT, "",
     COLUMN_FGR_ACTFORM, "",
     COLUMN_FGR_COMMENT, "",

     /* column behaviour */
     COLUMN_FGS_VISIBLE, TRUE,
     COLUMN_FGS_EDITABLE, TRUE,
     -1);
  
  
 
  /* And now we want to ease the user's work on modifying the fake
     name values into more useful ones: go to the place where the item
     was stored in the tree store, and put the cursor on the right row
     and put the keyboard focus on it also, so that it get
     highlighted...
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  /*
    {
    gchar *help = NULL;
    help =  gtk_tree_path_to_string  (path);
    debug_printf (("path = %s\n", help));
    g_free (help);
    }
  */

  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				NULL,
				TRUE,
				0.5,
				0.5);

  gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				path);

  column = gtk_tree_view_get_column (treeview, COLUMN_FGS_NAME);

  gtk_tree_selection_select_iter (tree_selection, &tree_iter);

  gtk_tree_view_set_cursor_on_cell (treeview,
				    path,
				    column,
				    NULL,
				    FALSE);
  
  gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				column,
				TRUE,
				0.5,
				0.5);

  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set 
    ((GtkWindow *) defctxt->definition_wnd,
     _("Added new fragspec;"
       " set its data!"), 
     POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}


void
polyxdef_ui_polchemdef_wnd_fragspec_add_fgr_button (GtkWidget *widget, 
						    gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeViewColumn* column = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_fgs = -1;
  gint idx_fgr = -1;
  gint depth = -1;
      
  PxmPolchemdef *polchemdef = NULL;

  PxmFragSpec *fgs = NULL;
  PxmFragRule *fgr = NULL;
  PxmFragRule *fgr_test = NULL;
  
  GPtrArray *fragspecGPA = NULL;


  /****************** GENERAL PHILOSOPHY **************************/

  /* A fragspec is made with the following intrinsic members: 
     name, end, actform and comment.
   
     It may also contain any number of fragrules.
    
     If a fragspec contains fragrule objects, they are represented
     as items in a treenode at a depth incremented by one.
  */

  /* That means that depending on the treeview situation, the actions
     that we need to take in response to the clicking onto the 
     ADD button are different.
  */

  /* If no selection is made at all, just return.
    
  If a selection is made:
   
  - if the selected item is a fragspec row, then a new fragrule row
  is to be added to this selected fragspec row, and a new subtree
  may need to be created if no one exists already.
   
  - if the selected row is a fragrule row, then a new fragrule row
  is to be added to the array of fragrules.
  */


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  polchemdef = defctxt->polchemdef;
  g_assert (polchemdef != NULL);
  
  fragspecGPA = polchemdef->fragspecGPA;
  g_assert (fragspecGPA != NULL);
  


  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "fragspec_treeview_model");
  g_assert (model != NULL);
  
  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), "fragspec_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, return.
   */
  if (result == FALSE)
    return;
  
  /* There is currently a selection, that means that the user wants to
     add an item. We'll have to decide if it is a fragspec or a
     fragrule.
  */

  /* Get the path to the selected item, which can be a fragspec row or
     a fragrule row, which are two very different cases that we must
     handle. Specifically, if depth == 1, the selected row is a
     fragspec, if depth == 2, then it is a fragrule.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1 if it is
     a fragspec item or 2 if it is a fragrule item.
  */
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth is %d\n", depth));
  */
  
  /* Whatever the depth, we can get the index of the fragspec that is
     concerned, be it selected or one of its fragrule children.
  */
  idx_fgs = gtk_tree_path_get_indices (path) [0];
  
  /*
    debug_printf (("idx_fgs=%d\n", idx_fgs));
  */
    
  if (depth == 2)
    {
      idx_fgr = gtk_tree_path_get_indices (path) [1];
      /*
	debug_printf (("idx_fgr is %d\n", idx_fgr));
      */
    }

  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Now, if the depth == 2, that means that the item that is selected
     is a fragrule. This means that we just want to insert/add a
     new fragrule to the fragspec at idx_fgs.
  */
  if (depth == 1)
    {
      /* The item that is selected is a fragspec item (at index
	 idx_fgs). Since we want to add a fragrule item, we must
	 insert a fragrule item as the first child of the
	 selected fragspec item.
      */
      gtk_tree_store_prepend (GTK_TREE_STORE (model),
			      &tree_iter,
			      &tree_iter_sel);
    }
  else if (depth == 2)
    {
      /* A fragrule item was selected. All we do is insert the new
	 fragrule item after the currently selected one.
      */
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_sel);
    }
  else
    {
      g_error (_("%s@%d: failed to insert/add fragrule to treeview\n"),
	     __FILE__, __LINE__);
    }

  
  /* Allocate a new fragrule so that we can later fill its member
     data.
  */
  /* Get the fragspec!
   */
  fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
  g_assert (fgs != NULL);
  /*
    debug_printf (("fragspec is %s\n", fgs->name));
  */

  fgr = pxmchem_fragrule_new ();

  pxmchem_fragrule_set_name (fgr, _("enter name here"));
  pxmchem_fragrule_set_prev (fgr, _("enter prev here"));
  pxmchem_fragrule_set_this (fgr, _("enter this here"));
  pxmchem_fragrule_set_next (fgr, _("enter next here"));
  pxmchem_fragrule_set_actform (fgr, _("enter actform here"));
  pxmchem_fragrule_set_comment (fgr, _("enter comment here"));

  /* But where do we have to put that stuff in fgs->fgrGPA? If a
     fragrule was selected (depth == 2) we had determined the idx_fgr
     of the selected fragrule. Otherwise idx_fgr is -1.
  */
  if (depth == 1)
    {
      /* The selected item was a fragspec at index idx_fgs, so we want
	 to insert at index 0 our new items.
      */
      /*
	debug_printf (("Insert the fragrule at index 0\n"));
      */

      fgr_test = g_ptr_array_insert_val (fgs->fgrGPA, 0, fgr);
      g_assert (fgr == fgr_test);
      
	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else /* depth == 2 */
    {
      /* 
	 We actually know that a fragrule element was selected,
	 at index idx_fgr... If the selected item was the last one, we
	 append the new one otherwise we insert the new one at the
	 index of the currently selected one +1.
      */
      if (idx_fgr + 1 >= fgs->fgrGPA->len)
	{
	  /*
	    debug_printf (("Add the fragrule at end of array\n"));
	  */

	  g_ptr_array_add (fgs->fgrGPA, fgr);

	  defctxt->is_modified = TRUE;
	  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	}
      else if (idx_fgr + 1 < fgs->fgrGPA->len)
	{
	  /*
	    debug_printf (("Insert the fragrule at index %d\n",
	    idx_fgr + 1));
	  */

	  fgr_test = 
	    g_ptr_array_insert_val (fgs->fgrGPA, idx_fgr + 1, fgr);
	  g_assert (fgr == fgr_test);
	  
	  defctxt->is_modified = TRUE;
	  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	}
      else
	{
	  g_error (_("%s@%d: failed to insert/add fragrule to array\n"),
		 __FILE__, __LINE__);
	}
    }


  gtk_tree_store_set 
    ((GtkTreeStore *) (model), &tree_iter,

     /* fragspec data */
     COLUMN_FGS_NAME, "",
     COLUMN_FGS_END, "",
     COLUMN_FGS_ACTFORM, "",
     COLUMN_FGS_COMMENT, "",

     /* fragrule data */
     COLUMN_FGR_NAME, fgr->name,
     COLUMN_FGR_PREV, fgr->prev,
     COLUMN_FGR_THIS, fgr->this,
     COLUMN_FGR_NEXT, fgr->next,
     COLUMN_FGR_ACTFORM, fgr->actform,
     COLUMN_FGR_COMMENT, fgr->comment,

     /* column behaviour */
     COLUMN_FGS_VISIBLE, TRUE,
     COLUMN_FGS_EDITABLE, TRUE,
     -1);


  /* And now we want to ease the user's work on modifying the fake
     fragrule values into more useful ones: go to the place where the
     item was stored in the tree store, and put the cursor on the
     right row and put the keyboard focus on it also, so that it get
     highlighted...
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  /*
    {
    gchar *help = NULL;
    help =  gtk_tree_path_to_string  (path);
    debug_printf (("path = %s\n", help));
    g_free (help);
    }
  */

  gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				path);

  column = gtk_tree_view_get_column (treeview, COLUMN_FGR_NAME);

  gtk_tree_selection_select_iter (tree_selection, &tree_iter);
  
  gtk_tree_view_set_cursor_on_cell (treeview,
				    path,
				    column,
				    NULL,
				    FALSE);
  
  gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				column,
				TRUE,
				0.5,
				0.5);
  
  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set 
    ((GtkWindow *) defctxt->definition_wnd, 
     _("Added new fragrule;"
       " set its data!"), 
     POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}


void
polyxdef_ui_polchemdef_wnd_fragspec_remove_button (GtkWidget *widget, 
						   gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;

  GtkTreeIter tree_iter;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_fgs = -1;
  gint idx_fgr = -1;
  gint depth = -1;
      
  PxmPolchemdef *polchemdef = NULL;

  PxmFragSpec *fgs = NULL;
  PxmFragRule *fgr = NULL;
  
  GPtrArray *fragspecGPA = NULL;

  /* We only remove the item that is selected. If no item is selected,
   * we return.
   */


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  polchemdef = defctxt->polchemdef;
  g_assert (polchemdef != NULL);
  
  fragspecGPA = polchemdef->fragspecGPA;
  g_assert (fragspecGPA != NULL);
  


  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "fragspec_treeview_model");
  g_assert (model != NULL);
  
  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "fragspec_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter);

  /* If there is no current selection, return.
   */
  if (result == FALSE)
    return;
  
  /* Get the path to the selected item, which can be a fragspec row
   * or a fragrule row, which are two very different cases that
   * we must handle. Specifically, if depth == 1, the selected row
   * is a fragspec, if depth == 2, then it is a fragrule.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1
   * if it is a fragspec item or 2 if it is a fragrule item.
   */
  depth = gtk_tree_path_get_depth (path);
  //printf ("depth=%d\n", depth);
  
  /* Whatever the depth, we can get the index of the fragspec
   * that is concerned, be it selected or one of its fragrule
   * children.
   */
  idx_fgs = gtk_tree_path_get_indices (path) [0];
  //printf ("idx_fgs=%d\n", idx_fgs);
    
  /* If depth == 2, that means that a fragrule is selected. We must
   * get to know at what index it is and we will remove it.
   */
  if (depth == 2)
    {
      idx_fgr = gtk_tree_path_get_indices (path) [1];
      //printf ("idx_fgr=%d\n", idx_fgr);
      
      /* Remove this fragrule !
       */
      fgs = g_ptr_array_index (fragspecGPA, idx_fgs);
      g_assert (fgs != NULL);
      
      fgr = g_ptr_array_remove_index (fgs->fgrGPA, idx_fgr);
      g_assert (fgr != NULL);
      pxmchem_fragrule_free (fgr);
      
	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

      gtk_tree_store_remove (GTK_TREE_STORE (model),
			     &tree_iter);
    }
  else if (depth == 1)
    {
      /* Remove this fragspec !
       */
      fgs = g_ptr_array_remove_index (fragspecGPA, idx_fgs);
      g_assert (fgs != NULL);
      
      pxmchem_fragspec_free (fgs);
      
	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
      
      gtk_tree_store_remove (GTK_TREE_STORE (model),
			     &tree_iter);
    }
  else
    {
      g_error (_("%s@%d: failed to remove item from treeview\n"),
	     __FILE__, __LINE__);
    }

  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  return ;
}
