/*****
*
* Copyright (C) 2000, 2002, 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <assert.h>

#include "detect.h"
#include "rules.h"
#include "rules-type.h"


static plugin_detect_t plugin;
static subscribtion_t subscribtion[] = {
        { p_tcp, NULL },
        { p_udp, NULL },
        { p_end, NULL }
};


typedef struct cnx_infos 
{        
        uint8_t port[65536 / 8];
        
        uint16_t high_port_cnx_count;
        uint16_t low_port_cnx_count;
        
        uint16_t first_port;
        uint16_t last_port;

        packet_container_t *packet;
        
        const char *kind;
        long first_cnxtime;
        prelude_timer_t timer;
} cnx_infos_t;




static ip_t *ignore = NULL;
static int is_enabled = 0;
static unsigned int plug_id;
static unsigned int cnx_ttl = 60;
static uint16_t max_low_port_count = 5;
static uint16_t max_high_port_count = 80;



static nids_alert_t *setup_alert(void) 
{
        static nids_alert_t alert;
        static int initialized = 0;
        static idmef_impact_t impact;
        
        if ( ! initialized ) {
                initialized = 1;
                nids_alert_init(&alert);
                
                alert.impact = &impact;
                impact.type = recon;
                impact.severity = impact_high;
                impact.completion = succeeded;
                impact.description.string = NULL;
                idmef_string_set_constant(&alert.classification.name, "Scanning attack");
        }

        return &alert;
}




/*
 *
 */
static void do_report_if_needed(cnx_infos_t *cnx, packet_container_t *packet) 
{
        if ( cnx->high_port_cnx_count >= max_high_port_count ||
             cnx->low_port_cnx_count >= max_low_port_count ) {
                
                struct timeval tv;

                tv.tv_sec = packet->pcap_hdr->ts.tv_sec;
                tv.tv_usec = packet->pcap_hdr->ts.tv_usec;
                
                nids_alert((plugin_generic_t *) &plugin, packet, setup_alert(),
                           "%s scanning attempt: %u cnx (low port) and %u cnx (high port) in %ld seconds - "
                           "Port range is %d - %d",
                           cnx->kind, cnx->low_port_cnx_count, cnx->high_port_cnx_count, 
                           tv.tv_sec - cnx->first_cnxtime, cnx->first_port, cnx->last_port);
        }
}



/*
 *
 */
static void expire_cnx(void *arg) 
{
        hostdb_t *h = arg;
        cnx_infos_t *cnx = hostdb_get_plugin_data(h, plug_id);

        hostdb_del(h, plug_id);
        timer_destroy(&cnx->timer);
        
        do_report_if_needed(cnx, cnx->packet);
        packet_release(cnx->packet);
        
        free(cnx);
}



/*
 * This list is based on the Snort one.
 */
static const char *guess_tcp_scan_kind(uint8_t flags) 
{
        flags = flags & ~(TH_CWR | TH_ECNECHO);
        
        if ( flags & TH_ACK ) {
                switch(flags) {
                        
                case (TH_ACK):
                case (TH_SYN | TH_ACK):
                case (TH_FIN | TH_ACK):
                case (TH_RST | TH_ACK):
                case (TH_ACK | TH_PSH):
                case (TH_ACK | TH_URG):
                case (TH_ACK | TH_URG | TH_PSH):
                case (TH_FIN | TH_ACK | TH_PSH):
                case (TH_FIN | TH_ACK | TH_URG):
                case (TH_FIN | TH_ACK | TH_URG | TH_PSH):
                case (TH_RST | TH_ACK | TH_PSH):   
                        break;
                        
                case (TH_SYN | TH_RST | TH_ACK | TH_FIN | TH_PSH | TH_URG):
                        return "Full Xmas";

                case (TH_SYN | TH_PSH | TH_ACK | TH_URG):
                        return "Spau";
                        
                default:
                        return "Invalid ACK";
                }
                
        } else {
        
                switch(flags) {
                        
                case (TH_RST):
                        break;
                        
                case (TH_SYN) :
                        return "Syn";
                        
                case (TH_FIN):
                        return "Fin";
                        
                case (TH_SYN | TH_FIN):
                        return "Syn Fin";
                        
                case 0:
                        return "Null";
                        
                case (TH_FIN | TH_PSH | TH_URG):
                        return "Xmas";
                        
                case (TH_URG):
                case (TH_PSH):
                case (TH_URG | TH_FIN):
                case (TH_PSH | TH_FIN):
                case (TH_URG | TH_PSH):
                        return "Vecna";

                case (TH_SYN | TH_FIN | TH_PSH | TH_URG):
                        return "Nmap";
                        
                default:
                        return "Unknow (no ack)";
                }
        }
        
        return NULL;
}




/*
 * Return 0 on success, -1 if it was already set.
 */
static int set_cnx_port(cnx_infos_t *cnx, uint16_t dport) 
{
        int ret;
        
        
        /*
         * dport / 8 give use the key to access our array.
         * 0x01 << (dport % 8) set the bit corresponding to dport to 1. 
         */
        ret = cnx->port[dport / 8] & (0x01 << (dport % 8));        
        if ( ret )
                return -1;

        /*
         * We shouldn't have any wrap_arround problem with
         * counter, as the port can be counter only one time.
         */
        cnx->port[dport / 8] |= (0x01 << (dport % 8));
        
        if ( dport > 1024 )
                cnx->high_port_cnx_count++;
        else 
                cnx->low_port_cnx_count++;

        return 0;
}




/*
 *
 */
static cnx_infos_t *new_cnx(packet_container_t *p, hostdb_t *h,  const char *kind, uint16_t dport) 
{
        cnx_infos_t *tmp;
        
        tmp = malloc(sizeof(cnx_infos_t));
        if ( ! tmp ) {
                log(LOG_ERR, "memory exhausted.");
                return NULL;
        }
        
        memset(tmp->port, 0, sizeof(tmp->port));
        tmp->low_port_cnx_count = tmp->high_port_cnx_count = 0;
        
        set_cnx_port(tmp, dport);
                
        tmp->first_cnxtime = p->pcap_hdr->ts.tv_sec;
        tmp->last_port = tmp->first_port = dport;
        tmp->kind = kind;
        tmp->packet = p;
        
        packet_lock(p);
        
        timer_set_expire(&tmp->timer, cnx_ttl);
        timer_set_data(&tmp->timer, h);
        timer_set_callback(&tmp->timer, expire_cnx);
        
        timer_init(&tmp->timer);

        return tmp;
}



/*
 *
 */
static void modify_cnx(cnx_infos_t *cnx, packet_container_t *pc, uint16_t dport) 
{
        int ret;
        
        if ( dport < cnx->first_port )
                cnx->first_port = dport;
        
        else if ( dport > cnx->last_port )
                cnx->last_port = dport;

        /*
         * If we could set this port (not already set),
         * reset the timer identifying this connection.
         */
        ret = set_cnx_port(cnx, dport);       
        if ( ret == 0 ) 
                timer_reset(&cnx->timer);
                
        packet_release(cnx->packet);
        packet_lock(pc);
        cnx->packet = pc;
}



/*
 *
 */
static void update_hdb_entry(hostdb_t *h, packet_container_t *p, uint16_t dport, const char *kind) 
{
        cnx_infos_t *tmp;

        /*
         * If this plugin didn't registered any data in the host
         * database at this time, create a new cnx_infos_t structure.
         */
        tmp = (cnx_infos_t *) hostdb_get_plugin_data(h, plug_id);
        if ( ! tmp ) {
                                
                tmp = new_cnx(p, h, kind, dport);
                if ( ! tmp )
                        return;
                
                hostdb_set_plugin_data(h, plug_id, tmp);
        }

        else modify_cnx(tmp, p, dport);
}



/*
 *
 */
static void create_hdb_entry(packet_container_t *p, const iphdr_t *ip, uint16_t dport, const char *kind)
{
        hostdb_t *hdb;
        cnx_infos_t *tmp;
        
        hdb = hostdb_new(p, ip);
        if (! hdb)
                return;
        
        tmp = new_cnx(p, hdb, kind, dport);
        if ( ! tmp )
                return;
        
        hostdb_set_plugin_data(hdb, plug_id, tmp);
}



/*
 *
 */
static void generic_packet(packet_container_t *p, const iphdr_t *ip, uint16_t dport, const char *kind)
{
        hostdb_t *hdb;

        timer_lock_critical_region();
        
        hdb = hostdb_search(ip);
        if ( ! hdb ) 
                create_hdb_entry(p, ip, dport, kind);
        else 
                update_hdb_entry(hdb, p, dport, kind);

        timer_unlock_critical_region();
}



static void tcp_packet(packet_container_t *p, const iphdr_t *ip, const tcphdr_t *tcp) 
{
        uint16_t dport;
        const char *kind;

        kind = guess_tcp_scan_kind(tcp->th_flags);
        if ( ! kind )
                return;

        dport = extract_uint16(&tcp->th_dport);
        generic_packet(p, ip, dport, kind);
}




/*
 *
 */
static void scandetect_run(packet_container_t *packet) 
{
        uint16_t dport;
        const iphdr_t *ip;
	
        /*
         * tcp / udp always come with IP. No check needed.
         */
        ip = packet->packet[packet->network_layer_depth].p.ip;
        
        /*
         * We don't want to analyze packet from this host.
         */
        if ( ignore && (ip->ip_src.s_addr & ignore->netmask) == ignore->ip ) 
                return;
        
        switch (packet->packet[packet->depth].proto) {
                
        case p_tcp:
                tcp_packet(packet, ip, packet->packet[packet->depth].p.tcp);
                break;

        case p_udp:
                dport = extract_uint16(&packet->packet[packet->depth].p.udp_hdr->uh_dport);
                generic_packet(packet, ip, dport, "UDP");
                break;

        default:
                /*
                 * This should never happen with current subscribtion rules.
                 */
                log(LOG_ERR, "Unknow protocol %d.\n", packet->packet[packet->depth].proto);
                assert(0);
        }
}



static int set_ignore_host(prelude_option_t *opt, const char *args) 
{        
        ignore = parse_ip(args);
        if ( ! ignore )
                return prelude_option_error;

        return prelude_option_success;
}


static int get_ignore_host(char *buf, size_t size) 
{
        buf[0] = '\0'; /* FIXME */
        return prelude_option_success;
}



static int set_high_port_count(prelude_option_t *opt, const char *args) 
{
        max_high_port_count = atoi(args);
        return prelude_option_success;
}


static int get_high_port_count(char *buf, size_t size) 
{
        snprintf(buf, size, "%u", max_high_port_count);
        return prelude_option_success;
}



static int set_low_port_count(prelude_option_t *opt, const char *args) 
{
        max_low_port_count = atoi(args);
        return prelude_option_success;
}


static int get_low_port_count(char *buf, size_t size) 
{
        snprintf(buf, size, "%u", max_low_port_count);
        return prelude_option_success;
}



static int set_cnx_ttl(prelude_option_t *opt, const char *args) 
{
        cnx_ttl = atoi(args);
        return prelude_option_success;
}


static int get_cnx_ttl(char *buf, size_t size) 
{
        snprintf(buf, size, "%d", cnx_ttl);
        return prelude_option_success;
}


static int set_scandetect_state(prelude_option_t *opt, const char *optarg) 
{
        int ret;
        
        if ( is_enabled == 1 ) {
                ret = plugin_unsubscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;        
                is_enabled = 0;
        }

        else {
                ret = plugin_subscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;
                is_enabled = 1;
        }
        
        return prelude_option_success;
}


static int get_scandetect_state(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", (is_enabled == 1) ? "enabled" : "disabled");
        return prelude_option_success;
}


plugin_generic_t *plugin_init(int argc, char **argv)
{
        prelude_option_t *opt;

        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 0, "scandetect",
                                 "Set ScanDetect plugin option", no_argument,
                                 set_scandetect_state, get_scandetect_state);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'i', "ignore-host",
                           "Host to ignore", required_argument, set_ignore_host, get_ignore_host);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'h', "high-port-count",
                           "Number of cnx on high port before issuing an alert",
                           required_argument, set_high_port_count, get_high_port_count);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'l', "low-port-count",
                           "Number of cnx on low port before issuing an alert",
                           required_argument, set_low_port_count, get_low_port_count);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'c', "cnx-ttl",
                           "The time under witch cnx-count must occur",
                           required_argument, set_cnx_ttl, get_cnx_ttl);

        /*
         * we need an unique ID for hostdb API use.
         */
        plug_id = plugin_request_new_id();
        
        plugin_set_name(&plugin, "ScanDetect");
        plugin_set_author(&plugin, "Yoann Vandoorselaere");
        plugin_set_contact(&plugin, "yoann@prelude-ids.org");
        plugin_set_desc(&plugin, "Detect almost all kind of scanning.");
        plugin_set_running_func(&plugin, scandetect_run);
        plugin_set_subscribtion(&plugin, subscribtion);
        
        return (plugin_generic_t *) &plugin;
}


















