/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: parameters.cc,v $
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:37  slash
 * Initial release.
 *
 */

#include <stdio.h>

#include <libintl.h>
#define _(String) gettext(String)

#include <fstream.h>
#include <strstream.h>

#include "main.h"
#include "parameters.h"

#include <magick/magick.h>


//---------------------------------------------------------------------------
Parameters::Parameters()
{
    output_format_ = HTML;

    // Clear the ImageSize pointers.
    prev_ = NULL;
    next_ = NULL;
    first_ = NULL;
    last_ = NULL;
    note_ = NULL;
    slide_ = NULL;
    header_ = NULL;

    // The default config-file is located in the user's home-directory.
    default_config_file_ = string(getenv("HOME")) + "/" + USER_CONFIG_FILE;

    // Default path to the themes.
    default_path_to_themes_ = DEFAULT_PATH_TO_THEMES;

    // Default theme.
    default_html_theme_ = DEFAULT_THEME;

    // Disabling of the XML-settings.
    default_disable_xml_settings_ = 0;
    cf_disable_xml_settings_ = -1;
    cl_disable_xml_settings_ = -1;
    
    // Resolution for the HTML-output.
    default_html_resolution_ = DEFAULT_RESOLUTION;
    cf_html_resolution_ = -1;
    xml_html_resolution_ = -1;
    cl_html_resolution_ = -1;

    // Background-graphic.
    default_html_disable_background_graphic_ = 0;
    cl_html_disable_background_graphic_ = -1;

    // Header-graphic.
    default_html_disable_header_graphic_ = 0;
    cl_html_disable_header_graphic_ = -1;

    // First and Last button.
    default_html_disable_first_last_button_ = 0;
    cl_html_disable_first_last_button_ = -1;

    // Note button.
    default_html_disable_note_button_ = 0;
    cl_html_disable_note_button_ = -1;

    // Sloppy buttons.
    default_html_sloppy_buttons_ = 0;
    cf_html_sloppy_buttons_ = -1;
    xml_html_sloppy_buttons_ = -1;
    cl_html_sloppy_buttons_ = -1;

    // Cell padding.
    default_html_cell_padding_ = 20;
    xml_html_cell_padding_ = -1;

    // Navigation buttons on the top.
    default_html_top_navigation_ = 0;
    cf_html_top_navigation_ = -1;
    xml_html_top_navigation_ = -1;
    cl_html_top_navigation_ = -1;

    // Outline for the LaTeX-output.
    default_latex_print_outline_ = 0;
    cf_latex_print_outline_ = -1;
    xml_latex_print_outline_ = -1;
    cl_latex_print_outline_ = -1;
}

//---------------------------------------------------------------------------
Parameters::~Parameters()
{
    ZapPointer(prev_);
    ZapPointer(next_);
    ZapPointer(first_);
    ZapPointer(last_);
    ZapPointer(note_);
    ZapPointer(slide_);
    ZapPointer(header_);
}


//---------------------------------------------------------------------------
int Parameters::readConfigFile()
{
    string &config_file = getConfigFile();
    ifstream input(config_file.c_str());
    if (!input) {
        if (config_file == cl_config_file_) {
            // Only write an error-message, if the file given with the
            // commandline-option -c doesn't exist.
            printf(_("The config-file \"%s\" doesn't exist."),
                   config_file.c_str());
            ENDL;
        }
        return -1;
    }

    printf(_("Reading the config-file \"%s\" ... "), config_file.c_str());

    unsigned int current_line_number = 0;
    unsigned int number_of_errors = 0;

    char option[BUFFER_SIZE];
    char equals;
    char value[BUFFER_SIZE];

    while (!input.eof()) {
        input.getline(Buffer_, BUFFER_SIZE);
        current_line_number++;

        // Skip empty lines or lines containing comments.
        if ((strlen(Buffer_) == 0) || (Buffer_[0] == '#'))  continue;

        istrstream line(Buffer_);
        line >> option >> equals >> value;  // "option = value"

        // Check the line for errors
        if (strlen(option) == 0) {
            ENDL;
            printf("\"%s\", %s %d :\n", config_file.c_str(),
                   _("line"), current_line_number);
            printf(_("    There is an option missing."));
            ENDL;
            number_of_errors++;

        } else if (strlen(value) == 0) {
            ENDL;
            printf("\"%s\", %s %d :\n", config_file.c_str(), _("line"),
                   current_line_number);
            printf(_("    The value for the option \"%s\" is missing."),
                   option);
            ENDL;
            number_of_errors++;

        } else if (equals != '=') {
            ENDL;
            printf("\"%s\", %s %d :\n", config_file.c_str(),
                   _("line"), current_line_number);
            printf(_("    A '=' is missing there."));
            ENDL;
            number_of_errors++;
        }

        // Parse the line of the config-file
        if (!strcmp(option, OPTION_PATH_TO_THEMES)) {
            cf_path_to_themes_ = value;

        } else if (!strcmp(option, OPTION_THEME)) {
            cf_html_theme_ = value;

        } else if (!strcmp(option, OPTION_DISABLE_XML_SETTINGS)) {
            if (!strcmp(value,"yes")) {
                cf_disable_xml_settings_ = 1;
            } else if (!strcmp(value,"no")) {
                cf_disable_xml_settings_ = 0;
            } else {
                ENDL;
                printf("\"%s\", %s %d :\n", config_file.c_str(),
                       _("line"), current_line_number);
                printf(_("    The value of the option \"%s\" must be either \"yes\" or \"no\"."), OPTION_DISABLE_XML_SETTINGS);
                ENDL;
                number_of_errors++;
            }

        } else if (!strcmp(option,OPTION_RESOLUTION)) {
            if (!strcmp(value, "640x480")) {
                cf_html_resolution_ = 640;
            } else if (!strcmp(value, "800x600")) {
                cf_html_resolution_ = 800;
            } else if (!strcmp(value, "800x600")) {
                cf_html_resolution_ = 1024;
            } else {
                ENDL;
                printf("\"%s\", %s %d :\n", config_file.c_str(),
                       _("line"), current_line_number);
                printf(_("    The resolution must be \"640x480\", \"800x600\" or \"1024x768\"."));
                ENDL;
                number_of_errors++;
            }

        } else if (!strcmp(option,OPTION_SLOPPY_BUTTONS)) {
            if (!strcmp(value,"yes")) {
                cf_html_sloppy_buttons_ = 1;
            } else if (!strcmp(value,"no")) {
                cf_html_sloppy_buttons_ = 0;
            } else {
                ENDL;
                printf("\"%s\", %s %d :\n", config_file.c_str(),
                       _("line"), current_line_number);
                printf(_("    The value of the option \"%s\" must be either \"yes\" or \"no\"."), OPTION_SLOPPY_BUTTONS);
                ENDL;
                number_of_errors++;
            }
            
        } else if (!strcmp(option,OPTION_TOP_NAVIGATION)) {
            if (!strcmp(value,"yes")) {
                cf_html_top_navigation_ = 1;
            } else if (!strcmp(value,"no")) {
                cf_html_top_navigation_ = 0;
            } else {
                ENDL;
                printf("\"%s\", %s %d :\n", config_file.c_str(),
                       _("line"), current_line_number);
                printf(_("    The value of the option \"%s\" must be either \"yes\" or \"no\"."), OPTION_TOP_NAVIGATION);
                ENDL;
                number_of_errors++;
            }

        } else if (!strcmp(option,OPTION_PRINT_OUTLINE)) {
            if (!strcmp(value,"yes")) {
                cf_latex_print_outline_ = 1;
            } else if (!strcmp(value,"no")) {
                cf_latex_print_outline_ = 0;
            } else {
                ENDL;
                printf("\"%s\", %s %d :\n", config_file.c_str(),
                       _("line"), current_line_number);
                printf(_("    The value of the option \"%s\" must be either \"yes\" or \"no\"."), OPTION_PRINT_OUTLINE);
                ENDL;
                number_of_errors++;
            }

        } else {
            ENDL;
            printf("\"%s\", %s %d :\n", config_file.c_str(),
                   _("line"), current_line_number);
            printf(_("    The option \"%s\" is not valid"), option);
            ENDL;
            number_of_errors++;
        }
    }

    if (number_of_errors > 0) {
        ENDL;
        if (number_of_errors == 1) {
            printf(_("The error in the config-file will be ignored."));
            ENDL;
            printf(_("Please fix it."));
        } else {
            printf(_("The errors in the config-file will be ignored."));
            ENDL;
            printf(_("Please fix them."));
        }
    } else {
        printf(_("finished."));
    }
    ENDL;

    return number_of_errors;
}


//---------------------------------------------------------------------------
int Parameters::initPathToTheme()
{
    // First, determine the selected path to the themes.
    string path_to_themes = getPathToThemes();

    // Next, determine the selected theme.
    string theme = getTheme();
    if (theme[theme.length()-1] != '/') {
        theme += "/";
    }
    

    // Check, if there is a file "config" in the theme-directory
    string config = string(path_to_themes) + theme + "config";
    ifstream theme_config_file(config.c_str());
    if (!theme_config_file) {
        printf(_("The theme \"%s\" doesn't exist in %s."), theme.c_str(),
               path_to_themes.c_str());
        ENDL;
        printf(_("Use the option \"-t\" to see a list of all available themes."));
        ENDL;
        return -1;
    }

    full_path_to_theme_ = string(path_to_themes) + theme;

    return 0;
}

//---------------------------------------------------------------------------
void Parameters::initImageSizes()
{
    string filename;
    ImageSize size;

    // Initialize the size of the previous-button
    filename = full_path_to_theme_ + BUTTON_PREV;
    if (!ping_image(filename.c_str(),&size)) {
        prev_ = new ImageSize;
        *prev_ = size;
    }

    // Initialize the size of the next-button
    filename = full_path_to_theme_ + BUTTON_NEXT;
    if (!ping_image(filename.c_str(),&size)) {
        next_ = new ImageSize;
        *next_ = size;
    }

    // Initialize the size of the first-button
    filename = full_path_to_theme_ + BUTTON_FIRST;
    if (!ping_image(filename.c_str(),&size)) {
        first_ = new ImageSize;
        *first_ = size;
    }

    // Initialize the size of the last-button
    filename = full_path_to_theme_ + BUTTON_LAST;
    if (!ping_image(filename.c_str(),&size)) {
        last_ = new ImageSize;
        *last_ = size;
    }

    // Initialize the size of the note-button
    filename = full_path_to_theme_ + BUTTON_NOTE;
    if (!ping_image(filename.c_str(),&size)) {
        note_ = new ImageSize;
        *note_ = size;
    }

    // Initialize the size of the background-graphic for the slide
    if (isSlideBackgroundEnabled()) {
        filename = full_path_to_theme_ + BG_SLIDE + JPG_EXT;
        if (!ping_image(filename.c_str(),&size)) {
            slide_ = new ImageSize;
            *slide_ = size;
        } else {
            printf("          %s", _("The background-graphic for the slides will be disabled."));
            ENDL;
            default_html_disable_background_graphic_ = 1;
        }
    }

    // Initialize the size of the backgound-graphic for the header
    if (isHeaderBackgroundEnabled()) {
        filename = full_path_to_theme_ + BG_HEADER;
        switch (getResolution()) {
          case 640:  filename += "_640"; break;
          case 800:  filename += "_800"; break;
          case 1024: filename += "_1024"; break;
        }
        filename += GIF_EXT;

        if (!ping_image(filename.c_str(),&size)) {
            header_ = new ImageSize;
            *header_ = size;
        } else {
            printf("          %s", _("The background-graphic for the header will be disabled."));
            ENDL;
            default_html_disable_header_graphic_ = 1;
        }
    }
}


//---------------------------------------------------------------------------
string &Parameters::getConfigFile()
{
    if (!cl_config_file_.empty()) {
        return cl_config_file_;
    } else {
        return default_config_file_;
    }
}

//---------------------------------------------------------------------------
string Parameters::getPathToThemes() const
{
    string path_to_themes;

    if (!cl_path_to_themes_.empty()) {
        path_to_themes = cl_path_to_themes_;
    } else if (!cf_path_to_themes_.empty()) {
        path_to_themes = cf_path_to_themes_;
    } else {
        path_to_themes = default_path_to_themes_;
    }

    if (path_to_themes[path_to_themes.length()-1] != '/') {
        path_to_themes += "/";
    }

    return path_to_themes;
}

//---------------------------------------------------------------------------
string &Parameters::getTheme()
{
    if (!cl_html_theme_.empty()) {
        return cl_html_theme_;
    } else if (!isXMLSettingsDisabled() && !xml_html_theme_.empty()) {
        return xml_html_theme_;
    } else if (!cf_html_theme_.empty()) {
        return cf_html_theme_;
    } else {
        return default_html_theme_;
    }
}

//---------------------------------------------------------------------------
int Parameters::isXMLSettingsDisabled() const
{
    if (cl_disable_xml_settings_ != -1)  return cl_disable_xml_settings_;
    if (cf_disable_xml_settings_ != -1)  return cf_disable_xml_settings_;
    return default_disable_xml_settings_;
}

//---------------------------------------------------------------------------
int Parameters::getResolution() const
{
    if (cl_html_resolution_ != -1)  return cl_html_resolution_;
    if (!isXMLSettingsDisabled()) {
        if (xml_html_resolution_ != -1)  return xml_html_resolution_;
    }
    if (cf_html_resolution_ != -1)  return cf_html_resolution_;
    return default_html_resolution_;
}

//---------------------------------------------------------------------------
int Parameters::isSlideBackgroundEnabled() const
{
    if (cl_html_disable_background_graphic_ != -1)  return 0;
    return 1;
}

//---------------------------------------------------------------------------
int Parameters::isHeaderBackgroundEnabled() const
{
    if (cl_html_disable_header_graphic_ != -1)  return 0;
    return 1;
}

//---------------------------------------------------------------------------
int Parameters::isFirstAndLastButtonEnabled() const
{
    if (cl_html_disable_first_last_button_ != -1)  return 0;
    return 1;
}

//---------------------------------------------------------------------------
int Parameters::isNoteButtonEnabled() const
{
    if (cl_html_disable_note_button_ != -1)  return 0;
    return 1;
}

//---------------------------------------------------------------------------
int Parameters::isSloppyButtonsEnabled() const
{
    if (cl_html_sloppy_buttons_ != -1)  return cl_html_sloppy_buttons_;
    if (!isXMLSettingsDisabled()) {
        if (xml_html_sloppy_buttons_ != -1)  return xml_html_sloppy_buttons_;
    }
    if (cf_html_sloppy_buttons_ != -1)  return cf_html_sloppy_buttons_;
    return default_html_sloppy_buttons_;
}

//---------------------------------------------------------------------------
int Parameters::getCellPadding() const
{
    if (!isXMLSettingsDisabled()) {
        if (xml_html_cell_padding_ != -1)  return xml_html_cell_padding_;
    }
    return default_html_cell_padding_;
}

//---------------------------------------------------------------------------
int Parameters::isTopNavigationEnabled() const
{
    if (cl_html_top_navigation_ != -1)  return cl_html_top_navigation_;
    if (!isXMLSettingsDisabled()) {
        if (xml_html_top_navigation_ != -1)  return xml_html_top_navigation_;
    }
    if (cf_html_top_navigation_ != -1)  return cf_html_top_navigation_;
    return default_html_top_navigation_;
}

//---------------------------------------------------------------------------
int Parameters::isOutlineEnabled() const
{
    if (cl_latex_print_outline_ != -1)  return cl_latex_print_outline_;
    if (!isXMLSettingsDisabled()) {
        if (xml_latex_print_outline_ != -1)  return xml_latex_print_outline_;
    }
    if (cf_latex_print_outline_ != -1)  return cf_latex_print_outline_;
    return default_latex_print_outline_;
}
