/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: slidenode.h,v $
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:50  slash
 * Initial release.
 *
 */

#ifndef SLIDENODE_H
#define SLIDENODE_H

#include "tocinterface.h"
#include "node.h"

//---------------------------------------------------------------------------
/**
 * This class represents one slide.
 * It contains all data, that is needed for writing the apropriate HTML-File.
 * The member variable environment_ points to any environment,
 * where the content of the slide is stored.
 * A slide can also consist of subslides, which will be written to own
 * HTML-files.
 * The subslides itself are generated on-the-fly, when the method write() is
 * called.
 */
class SlideNode : public Node, TOCInterface {
  public:
    //-----------------------------------------------------------------------
    /**
     * Creates a slide with the given sequence-number.
     *
     * @param node A pointer to the slide-node in the XML-tree.
     */
    SlideNode(xmlNodePtr node);

    //-----------------------------------------------------------------------
    /**
     * Destructor.
     */
    ~SlideNode();


    //-----------------------------------------------------------------------
    /**
     * This method writes the content of the slide to a HTML-file.
     * If the slide contains subslides, all subslides will be written to
     * own HTML-files.
     * The writing itself is done by calling the helper-method
     * writeSubSlide().
     *
     * @see writeSubSlide
     */
    void writeHTML(int parameter = 0) const;

    //-----------------------------------------------------------------------
    /**
     * This method writes the content of the slide to the global output
     * filestream Output_.
     * The format of the output must be LaTeX.
     */
    void writeLaTeX() const;

    //-----------------------------------------------------------------------
    /**
     * This method writes the content of the slide to the global output
     * filestream Output_.
     * The format of the output must be XML.
     */
    void writeXML() const;


    //-----------------------------------------------------------------------
    /**
     * This method initializes the filename of the first and the last slide.
     */
    static void initFilenameOfFirstAndLastSlide();


    //-----------------------------------------------------------------------
    /**
     * This method creates the entry for the table of contents.
     */
    void writeTOCEntry() const;


  private:

    //-----------------------------------------------------------------------
    /**
     * This helper-method writes a (sub)slide to a HTML-File.
     * The filename of the HTML-file is generated on-the-fly by the value
     * of sequence_number_ and the value of the parameter sub_slide.
     * The writing of the header and footer of the HTML-file and the content
     * is done by calling the helper-methods writeHeader() and writeFooter().
     *
     * @param sub_slide The subslide that should be written.
     *                  If -1, there doesn't exist any subslides.
     * @see writeHeader, writeFooter	      
     */
    void writeSubSlideHTML(const int sub_slide) const;

    //-----------------------------------------------------------------------
    /**
     * This helper-method prints the HTML-header and the header of the
     * (sub)slide to the global output filestream Output_.
     *
     * @param prev_slide The filename of the previous (sub)slide
     * @param next_slide The filename of the next (sub)slide
     */
    void writeHTMLHeader(const char *prev_slide,
                         const char *next_slide) const;

    //-----------------------------------------------------------------------
    /**
     * This helper-method prints the navigation-buttons to the global output
     * filestream Output_.
     *
     * @param prev_slide The filename of the previous (sub)slide
     * @param next_slide The filename of the next (sub)slide
     */
    void writeHTMLNavigationButtons(const char *prev_slide,
                                    const char *next_slide) const;

    //-----------------------------------------------------------------------
    /**
     * This helper-method prints the HTML-table, which contains the buttons,
     * and the HTML-footer of the (sub)slide to the global output filestream
     * Output_.
     *
     * @param prev_slide The filename of the previous (sub)slide
     * @param next_slide The filename of the next (sub)slide
     */
    void writeHTMLFooter(const char *prev_slide,
                         const char *next_slide) const;


  private:

    //-----------------------------------------------------------------------
    /**
     * The sequence-number of the slide.
     * The first slide will get the sequence-number 1.
     */
    unsigned int sequence_number_;

    //-----------------------------------------------------------------------
    /**
     * The number of subslides or -1, if there are no subslides.
     */
    int number_of_subslides_;

    //-----------------------------------------------------------------------
    /**
     * The title of the slide.
     */
    char *title_;

    //-----------------------------------------------------------------------
    /**
     * The filename of additional notes to the slide.
     */
    char *note_;

    //-----------------------------------------------------------------------
    /**
     * 0, if this slide should be skipped in the LaTeX-output, else 1.
     */
    int latex_;


  public:

    //-----------------------------------------------------------------------
    /**
     * This vector stores all slidenodes.
     */
    static vector<SlideNode*> All_Slides_;


    //-----------------------------------------------------------------------
    /**
     * The filename of the first slide of the presentation.
     */
    static char Filename_Of_First_Slide_[64];

    //-----------------------------------------------------------------------
    /**
     * The filename of the last slide of the presentation.
     */
    static char Filename_Of_Last_Slide_[64];


    //-----------------------------------------------------------------------
    /**
     * The title of the section.
     */
    static const char* Section_Title_;

    //-----------------------------------------------------------------------
    /**
     * The filename of the first slide in the section.
     */
    static char Section_Filename_[64];

    //-----------------------------------------------------------------------
    /**
     * Indicates, if the title of the section was written
     * in the outline-version of the LaTeX-output.
     */
    static int Section_Written_In_Outline_;
};

#endif //SLIDENODE_H
