--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_cs_nonce_locks(
  partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(cs_nonce_lock_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',nonce BYTEA PRIMARY KEY CHECK (LENGTH(nonce)=32)'
      ',op_hash BYTEA NOT NULL CHECK (LENGTH(op_hash)=64)'
      ',max_denomination_serial INT8 NOT NULL'
    ') %s ;'
    ,'cs_nonce_locks'
    ,'PARTITION BY HASH (nonce)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'ensures a Clause Schnorr client nonce is locked for use with an operation identified by a hash'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'actual nonce submitted by the client'
    ,'nonce'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash (RC for refresh, blind coin hash for withdraw) the nonce may be used with'
    ,'op_hash'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Maximum number of a CS denomination serial the nonce could be used with, for GC'
    ,'max_denomination_serial'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_cs_nonce_locks(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'cs_nonce_locks';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_cs_nonce_lock_serial_id_key'
    ' UNIQUE (cs_nonce_lock_serial_id)'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('cs_nonce_locks'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('cs_nonce_locks'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
