/*
 * copyright (C) 2003 the ffmpeg project
 *
 * This file is part of Libav.
 *
 * Libav is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Libav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Libav; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_VP3DATA_H
#define AVCODEC_VP3DATA_H

#include <stdint.h>
#include <stdlib.h>

/* these coefficients dequantize intraframe Y plane coefficients
 * (note: same as JPEG) */
static const int16_t vp31_intra_y_dequant[64] =
{       16,  11,  10,  16,  24,  40,  51,  61,
        12,  12,  14,  19,  26,  58,  60,  55,
        14,  13,  16,  24,  40,  57,  69,  56,
        14,  17,  22,  29,  51,  87,  80,  62,
        18,  22,  37,  58,  68, 109, 103,  77,
        24,  35,  55,  64,  81, 104, 113,  92,
        49,  64,  78,  87, 103, 121, 120, 101,
        72,  92,  95,  98, 112, 100, 103,  99
};

/* these coefficients dequantize intraframe C plane coefficients
 * (note: same as JPEG) */
static const int16_t vp31_intra_c_dequant[64] =
{       17,  18,     24,     47,     99,     99,     99,     99,
        18,  21,     26,     66,     99,     99,     99,     99,
        24,  26,     56,     99,     99,     99,     99,     99,
        47,  66,     99,     99,     99,     99,     99,     99,
        99,  99,     99,     99,     99,     99,     99,     99,
        99,  99,     99,     99,     99,     99,     99,     99,
        99,  99,     99,     99,     99,     99,     99,     99,
        99,  99,     99,     99,     99,     99,     99,     99
};

/* these coefficients dequantize interframe coefficients (all planes) */
static const int16_t vp31_inter_dequant[64] =
{   16,  16,  16,  20,  24,  28,  32,  40,
    16,  16,  20,  24,  28,  32,  40,  48,
    16,  20,  24,  28,  32,  40,  48,  64,
    20,  24,  28,  32,  40,  48,  64,  64,
    24,  28,  32,  40,  48,  64,  64,  64,
    28,  32,  40,  48,  64,  64,  64,  96,
    32,  40,  48,  64,  64,  64,  96,  128,
    40,  48,  64,  64,  64,  96,  128, 128
};

static const int16_t vp31_dc_scale_factor[64] =
{ 220, 200, 190, 180, 170, 170, 160, 160,
  150, 150, 140, 140, 130, 130, 120, 120,
  110, 110, 100, 100, 90,  90,  90,  80,
  80,  80,  70,  70,  70,  60,  60,  60,
  60,  50,  50,  50,  50,  40,  40,  40,
  40,  40,  30,  30,  30,  30,  30,  30,
  30,  20,  20,  20,  20,  20,  20,  20,
  20,  10,  10,  10,  10,  10,  10,  10
};

static const uint32_t vp31_ac_scale_factor[64] =
{ 500,  450,  400,  370,  340,  310, 285, 265,
  245,  225,  210,  195,  185,  180, 170, 160,
  150,  145,  135,  130,  125,  115, 110, 107,
  100,   96,   93,   89,   85,   82,  75,  74,
   70,   68,   64,   60,   57,   56,  52,  50,
   49,   45,   44,   43,   40,   38,  37,  35,
   33,   32,   30,   29,   28,   25,  24,  22,
   21,   19,   18,   17,   15,   13,  12,  10
};

static const uint8_t vp31_filter_limit_values[64] =
{  30, 25, 20, 20, 15, 15, 14, 14,
   13, 13, 12, 12, 11, 11, 10, 10,
    9,  9,  8,  8,  7,  7,  7,  7,
    6,  6,  6,  6,  5,  5,  5,  5,
    4,  4,  4,  4,  3,  3,  3,  3,
    2,  2,  2,  2,  2,  2,  2,  2,
    0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0
};

static const uint16_t superblock_run_length_vlc_table[34][2] = {
    { 0, 1 },

    { 4, 3 },    { 5, 3 },

    { 0xC, 4 },  { 0xD, 4 },

    { 0x38, 6 },    { 0x39, 6 },    { 0x3A, 6 },    { 0x3B, 6 },

    { 0xF0, 8 },    { 0xF1, 8 },    { 0xF2, 8 },    { 0xF3, 8 },
    { 0xF4, 8 },    { 0xF5, 8 },    { 0xF6, 8 },    { 0xF7, 8 },

    { 0x3E0, 10 },    { 0x3E1, 10 },    { 0x3E2, 10 },    { 0x3E3, 10 },
    { 0x3E4, 10 },    { 0x3E5, 10 },    { 0x3E6, 10 },    { 0x3E7, 10 },
    { 0x3E8, 10 },    { 0x3E9, 10 },    { 0x3EA, 10 },    { 0x3EB, 10 },
    { 0x3EC, 10 },    { 0x3ED, 10 },    { 0x3EE, 10 },    { 0x3EF, 10 },

    { 0x3F, 6 }  /* this last VLC is a special case for reading 12 more
                    bits from stream and adding the value 34 */
};

static const uint16_t fragment_run_length_vlc_table[30][2] = {
    /* 1 -> 2 */
    { 0x0, 2 },    { 0x1, 2 },

    /* 3 -> 4 */
    { 0x4, 3 },    { 0x5, 3 },

    /* 5 -> 6 */
    { 0xC, 4 },    { 0xD, 4 },

    /* 7 -> 10 */
    { 0x38, 6 },   { 0x39, 6 },
    { 0x3A, 6 },   { 0x3B, 6 },

    /* 11 -> 14 */
    { 0x78, 7 },   { 0x79, 7 },
    { 0x7A, 7 },   { 0x7B, 7 },

    /* 15 -> 30 */
    { 0x1F0, 9 },  { 0x1F1, 9 },  { 0x1F2, 9 },  { 0x1F3, 9 },
    { 0x1F4, 9 },  { 0x1F5, 9 },  { 0x1F6, 9 },  { 0x1F7, 9 },
    { 0x1F8, 9 },  { 0x1F9, 9 },  { 0x1FA, 9 },  { 0x1FB, 9 },
    { 0x1FC, 9 },  { 0x1FD, 9 },  { 0x1FE, 9 },  { 0x1FF, 9 }
};

static const uint8_t mode_code_vlc_table[8][2] = {
    { 0, 1 },      { 2, 2 },
    { 6, 3 },      { 14, 4 },
    { 30, 5 },     { 62, 6 },
    { 126, 7 },    { 127, 7 }
};

static const uint8_t motion_vector_vlc_table[63][2] = {
    { 0, 3 },
    { 1, 3 },
    { 2, 3 },

    { 6, 4 },    { 7, 4 },

    { 8, 4 },    { 9, 4 },

    { 40, 6 },    { 41, 6 },    { 42, 6 },    { 43, 6 },
    { 44, 6 },    { 45, 6 },    { 46, 6 },    { 47, 6 },

    { 96, 7 },    { 97, 7 },    { 98, 7 },    { 99, 7 },
    { 100, 7 },   { 101, 7 },   { 102, 7 },   { 103, 7 },
    { 104, 7 },   { 105, 7 },   { 106, 7 },   { 107, 7 },
    { 108, 7 },   { 109, 7 },   { 110, 7 },   { 111, 7 },

    { 0xE0, 8 },  { 0xE1, 8 },  { 0xE2, 8 },  { 0xE3, 8 },
    { 0xE4, 8 },  { 0xE5, 8 },  { 0xE6, 8 },  { 0xE7, 8 },
    { 0xE8, 8 },  { 0xE9, 8 },  { 0xEA, 8 },  { 0xEB, 8 },
    { 0xEC, 8 },  { 0xED, 8 },  { 0xEE, 8 },  { 0xEF, 8 },

    { 0xF0, 8 },  { 0xF1, 8 },  { 0xF2, 8 },  { 0xF3, 8 },
    { 0xF4, 8 },  { 0xF5, 8 },  { 0xF6, 8 },  { 0xF7, 8 },
    { 0xF8, 8 },  { 0xF9, 8 },  { 0xFA, 8 },  { 0xFB, 8 },
    { 0xFC, 8 },  { 0xFD, 8 },  { 0xFE, 8 },  { 0xFF, 8 }
};

static const int motion_vector_table[63] = {
    0, 1, -1,
    2, -2,
    3, -3,
    4, -4, 5, -5, 6, -6, 7, -7,
    8, -8, 9, -9, 10, -10, 11, -11, 12, -12, 13, -13, 14, -14, 15, -15,
    16, -16, 17, -17, 18, -18, 19, -19, 20, -20, 21, -21, 22, -22, 23, -23,
    24, -24, 25, -25, 26, -26, 27, -27, 28, -28, 29, -29, 30, -30, 31, -31
};

static const int8_t fixed_motion_vector_table[64] = {
   0,   0,   1,  -1,   2,  -2,   3,  -3,
   4,  -4,   5,  -5,   6,  -6,   7,  -7,
   8,  -8,   9,  -9,  10, -10,  11, -11,
  12, -12,  13, -13,  14, -14,  15, -15,
  16, -16,  17, -17,  18, -18,  19, -19,
  20, -20,  21, -21,  22, -22,  23, -23,
  24, -24,  25, -25,  26, -26,  27, -27,
  28, -28,  29, -29,  30, -30,  31, -31
};

/* only tokens 0..6 indicate eob runs */
static const int eob_run_base[7] = {
    1, 2, 3, 4, 8, 16, 0
};
static const int eob_run_get_bits[7] = {
    0, 0, 0, 2, 3, 4, 12
};

static const int zero_run_base[32] = {
    0, 0, 0, 0, 0, 0, 0,  /* 0..6 are never used */
    0, 0,  /* 7..8 */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 9..22 */
    1, 2, 3, 4, 5,  /* 23..27 */
    6, 10, 1, 2  /* 28..31 */
};
static const int zero_run_get_bits[32] = {
    0, 0, 0, 0, 0, 0, 0,  /* 0..6 are never used */
    3, 6,  /* 7..8 */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  /* 9..22 */
    0, 0, 0, 0, 0,  /* 23..27 */
    2, 3, 0, 1  /* 28..31 */
};

static const int coeff_get_bits[32] = {
    0, 0, 0, 0, 0, 0, 0,  /* 0..6 are never used */
    0, 0, 0, 0, 0, 0,  /* 7..12 use constant coeffs */
    1, 1, 1, 1,  /* 13..16 are constants but still need sign bit */
    2, 3, 4, 5, 6, 10,  /* 17..22, for reading large coeffs */
    1, 1, 1, 1, 1, 1, 1,  /* 23..29 are constants but still need sign bit */
    2, 2  /* 30..31 */
};

static const int16_t coeff_table_token_7_8[1] = { 0 };
static const int16_t coeff_table_token_9[1] = { 1 };
static const int16_t coeff_table_token_10[1] = { -1 };
static const int16_t coeff_table_token_11[1] = { 2 };
static const int16_t coeff_table_token_12[1] = { -2 };

static const int16_t coeff_table_token_13[2] = { 3, -3 };
static const int16_t coeff_table_token_14[2] = { 4, -4 };
static const int16_t coeff_table_token_15[2] = { 5, -5 };
static const int16_t coeff_table_token_16[2] = { 6, -6 };

static const int16_t coeff_table_token_23_24_25_26_27_28_29[2] = { 1, -1 };
static const int16_t coeff_table_token_30[4] = { 2, 3, -2, -3 };
static const int16_t coeff_table_token_31[4] = { 2, 3, -2, -3 };

static const int16_t coeff_table_token_17[4] = {
    7, 8, -7, -8
};

static const int16_t coeff_table_token_18[8] = {
    9, 10, 11, 12, -9, -10, -11, -12
};

static const int16_t coeff_table_token_19[16] = {
    13, 14, 15, 16, 17, 18, 19, 20, -13, -14, -15, -16, -17, -18, -19, -20
};

static const int16_t coeff_table_token_20[32] = {
    21, 22, 23, 24, 25, 26, 27, 28,
    29, 30, 31, 32, 33, 34, 35, 36,
    -21, -22, -23, -24, -25, -26, -27, -28,
    -29, -30, -31, -32, -33, -34, -35, -36
};

static const int16_t coeff_table_token_21[64] = {
    37, 38, 39, 40, 41, 42, 43, 44,
    45, 46, 47, 48, 49, 50, 51, 52,
    53, 54, 55, 56, 57, 58, 59, 60,
    61, 62, 63, 64, 65, 66, 67, 68,
    -37, -38, -39, -40, -41, -42, -43, -44,
    -45, -46, -47, -48, -49, -50, -51, -52,
    -53, -54, -55, -56, -57, -58, -59, -60,
    -61, -62, -63, -64, -65, -66, -67, -68
};

static const int16_t coeff_table_token_22[1024] = {
    69, 70, 71, 72, 73, 74, 75, 76,
    77, 78, 79, 80, 81, 82, 83, 84,
    85, 86, 87, 88, 89, 90, 91, 92,
    93, 94, 95, 96, 97, 98, 99, 100,
    101, 102, 103, 104, 105, 106, 107, 108,
    109, 110, 111, 112, 113, 114, 115, 116,
    117, 118, 119, 120, 121, 122, 123, 124,
    125, 126, 127, 128, 129, 130, 131, 132,
    133, 134, 135, 136, 137, 138, 139, 140,
    141, 142, 143, 144, 145, 146, 147, 148,
    149, 150, 151, 152, 153, 154, 155, 156,
    157, 158, 159, 160, 161, 162, 163, 164,
    165, 166, 167, 168, 169, 170, 171, 172,
    173, 174, 175, 176, 177, 178, 179, 180,
    181, 182, 183, 184, 185, 186, 187, 188,
    189, 190, 191, 192, 193, 194, 195, 196,
    197, 198, 199, 200, 201, 202, 203, 204,
    205, 206, 207, 208, 209, 210, 211, 212,
    213, 214, 215, 216, 217, 218, 219, 220,
    221, 222, 223, 224, 225, 226, 227, 228,
    229, 230, 231, 232, 233, 234, 235, 236,
    237, 238, 239, 240, 241, 242, 243, 244,
    245, 246, 247, 248, 249, 250, 251, 252,
    253, 254, 255, 256, 257, 258, 259, 260,
    261, 262, 263, 264, 265, 266, 267, 268,
    269, 270, 271, 272, 273, 274, 275, 276,
    277, 278, 279, 280, 281, 282, 283, 284,
    285, 286, 287, 288, 289, 290, 291, 292,
    293, 294, 295, 296, 297, 298, 299, 300,
    301, 302, 303, 304, 305, 306, 307, 308,
    309, 310, 311, 312, 313, 314, 315, 316,
    317, 318, 319, 320, 321, 322, 323, 324,
    325, 326, 327, 328, 329, 330, 331, 332,
    333, 334, 335, 336, 337, 338, 339, 340,
    341, 342, 343, 344, 345, 346, 347, 348,
    349, 350, 351, 352, 353, 354, 355, 356,
    357, 358, 359, 360, 361, 362, 363, 364,
    365, 366, 367, 368, 369, 370, 371, 372,
    373, 374, 375, 376, 377, 378, 379, 380,
    381, 382, 383, 384, 385, 386, 387, 388,
    389, 390, 391, 392, 393, 394, 395, 396,
    397, 398, 399, 400, 401, 402, 403, 404,
    405, 406, 407, 408, 409, 410, 411, 412,
    413, 414, 415, 416, 417, 418, 419, 420,
    421, 422, 423, 424, 425, 426, 427, 428,
    429, 430, 431, 432, 433, 434, 435, 436,
    437, 438, 439, 440, 441, 442, 443, 444,
    445, 446, 447, 448, 449, 450, 451, 452,
    453, 454, 455, 456, 457, 458, 459, 460,
    461, 462, 463, 464, 465, 466, 467, 468,
    469, 470, 471, 472, 473, 474, 475, 476,
    477, 478, 479, 480, 481, 482, 483, 484,
    485, 486, 487, 488, 489, 490, 491, 492,
    493, 494, 495, 496, 497, 498, 499, 500,
    501, 502, 503, 504, 505, 506, 507, 508,
    509, 510, 511, 512, 513, 514, 515, 516,
    517, 518, 519, 520, 521, 522, 523, 524,
    525, 526, 527, 528, 529, 530, 531, 532,
    533, 534, 535, 536, 537, 538, 539, 540,
    541, 542, 543, 544, 545, 546, 547, 548,
    549, 550, 551, 552, 553, 554, 555, 556,
    557, 558, 559, 560, 561, 562, 563, 564,
    565, 566, 567, 568, 569, 570, 571, 572,
    573, 574, 575, 576, 577, 578, 579, 580,
    -69, -70, -71, -72, -73, -74, -75, -76,
    -77, -78, -79, -80, -81, -82, -83, -84,
    -85, -86, -87, -88, -89, -90, -91, -92,
    -93, -94, -95, -96, -97, -98, -99, -100,
    -101, -102, -103, -104, -105, -106, -107, -108,
    -109, -110, -111, -112, -113, -114, -115, -116,
    -117, -118, -119, -120, -121, -122, -123, -124,
    -125, -126, -127, -128, -129, -130, -131, -132,
    -133, -134, -135, -136, -137, -138, -139, -140,
    -141, -142, -143, -144, -145, -146, -147, -148,
    -149, -150, -151, -152, -153, -154, -155, -156,
    -157, -158, -159, -160, -161, -162, -163, -164,
    -165, -166, -167, -168, -169, -170, -171, -172,
    -173, -174, -175, -176, -177, -178, -179, -180,
    -181, -182, -183, -184, -185, -186, -187, -188,
    -189, -190, -191, -192, -193, -194, -195, -196,
    -197, -198, -199, -200, -201, -202, -203, -204,
    -205, -206, -207, -208, -209, -210, -211, -212,
    -213, -214, -215, -216, -217, -218, -219, -220,
    -221, -222, -223, -224, -225, -226, -227, -228,
    -229, -230, -231, -232, -233, -234, -235, -236,
    -237, -238, -239, -240, -241, -242, -243, -244,
    -245, -246, -247, -248, -249, -250, -251, -252,
    -253, -254, -255, -256, -257, -258, -259, -260,
    -261, -262, -263, -264, -265, -266, -267, -268,
    -269, -270, -271, -272, -273, -274, -275, -276,
    -277, -278, -279, -280, -281, -282, -283, -284,
    -285, -286, -287, -288, -289, -290, -291, -292,
    -293, -294, -295, -296, -297, -298, -299, -300,
    -301, -302, -303, -304, -305, -306, -307, -308,
    -309, -310, -311, -312, -313, -314, -315, -316,
    -317, -318, -319, -320, -321, -322, -323, -324,
    -325, -326, -327, -328, -329, -330, -331, -332,
    -333, -334, -335, -336, -337, -338, -339, -340,
    -341, -342, -343, -344, -345, -346, -347, -348,
    -349, -350, -351, -352, -353, -354, -355, -356,
    -357, -358, -359, -360, -361, -362, -363, -364,
    -365, -366, -367, -368, -369, -370, -371, -372,
    -373, -374, -375, -376, -377, -378, -379, -380,
    -381, -382, -383, -384, -385, -386, -387, -388,
    -389, -390, -391, -392, -393, -394, -395, -396,
    -397, -398, -399, -400, -401, -402, -403, -404,
    -405, -406, -407, -408, -409, -410, -411, -412,
    -413, -414, -415, -416, -417, -418, -419, -420,
    -421, -422, -423, -424, -425, -426, -427, -428,
    -429, -430, -431, -432, -433, -434, -435, -436,
    -437, -438, -439, -440, -441, -442, -443, -444,
    -445, -446, -447, -448, -449, -450, -451, -452,
    -453, -454, -455, -456, -457, -458, -459, -460,
    -461, -462, -463, -464, -465, -466, -467, -468,
    -469, -470, -471, -472, -473, -474, -475, -476,
    -477, -478, -479, -480, -481, -482, -483, -484,
    -485, -486, -487, -488, -489, -490, -491, -492,
    -493, -494, -495, -496, -497, -498, -499, -500,
    -501, -502, -503, -504, -505, -506, -507, -508,
    -509, -510, -511, -512, -513, -514, -515, -516,
    -517, -518, -519, -520, -521, -522, -523, -524,
    -525, -526, -527, -528, -529, -530, -531, -532,
    -533, -534, -535, -536, -537, -538, -539, -540,
    -541, -542, -543, -544, -545, -546, -547, -548,
    -549, -550, -551, -552, -553, -554, -555, -556,
    -557, -558, -559, -560, -561, -562, -563, -564,
    -565, -566, -567, -568, -569, -570, -571, -572,
    -573, -574, -575, -576, -577, -578, -579, -580
};

static const int16_t *const coeff_tables[32] = {
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    coeff_table_token_7_8,

    coeff_table_token_7_8,
    coeff_table_token_9,
    coeff_table_token_10,
    coeff_table_token_11,
    coeff_table_token_12,
    coeff_table_token_13,
    coeff_table_token_14,
    coeff_table_token_15,

    coeff_table_token_16,
    coeff_table_token_17,
    coeff_table_token_18,
    coeff_table_token_19,
    coeff_table_token_20,
    coeff_table_token_21,
    coeff_table_token_22,
    coeff_table_token_23_24_25_26_27_28_29,

    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_30,
    coeff_table_token_31
};

static const uint16_t dc_bias[16][32][2] = {
  {  /* DC bias table 0 */
    { 0x2D, 6 },
    { 0x26, 7 },
    { 0x166, 9 },
    { 0x4E, 8 },
    { 0x2CE, 10 },
    { 0x59E, 11 },
    { 0x27D, 11 },
    { 0x8, 5 },
    { 0x4F9, 12 },
    { 0xF, 4 },
    { 0xE, 4 },
    { 0x1B, 5 },
    { 0x6, 4 },
    { 0x8, 4 },
    { 0x5, 4 },
    { 0x1A, 5 },
    { 0x15, 5 },
    { 0x7, 4 },
    { 0xC, 4 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0x9, 4 },
    { 0x17, 5 },
    { 0x29, 6 },
    { 0x28, 6 },
    { 0xB2, 8 },
    { 0x4F8, 12 },
    { 0x59F, 11 },
    { 0x9E, 9 },
    { 0x13F, 10 },
    { 0x12, 6 },
    { 0x58, 7 }
  },
  {  /* DC bias table 1 */
    { 0x10, 5 },
    { 0x47, 7 },
    { 0x1FF, 9 },
    { 0x8C, 8 },
    { 0x3FC, 10 },
    { 0x46A, 11 },
    { 0x469, 11 },
    { 0x22, 6 },
    { 0x11A1, 13 },
    { 0xE, 4 },
    { 0xD, 4 },
    { 0x4, 4 },
    { 0x5, 4 },
    { 0x9, 4 },
    { 0x6, 4 },
    { 0x1E, 5 },
    { 0x16, 5 },
    { 0x7, 4 },
    { 0xC, 4 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xA, 4 },
    { 0x17, 5 },
    { 0x7D, 7 },
    { 0x7E, 7 },
    { 0x11B, 9 },
    { 0x8D1, 12 },
    { 0x3FD, 10 },
    { 0x46B, 11 },
    { 0x11A0, 13 },
    { 0x7C, 7 },
    { 0xFE, 8 }
  },
  {  /* DC bias table 2 */
    { 0x16, 5 },
    { 0x20, 6 },
    { 0x86, 8 },
    { 0x87, 8 },
    { 0x367, 10 },
    { 0x6CC, 11 },
    { 0x6CB, 11 },
    { 0x6E, 7 },
    { 0x366D, 14 },
    { 0xF, 4 },
    { 0xE, 4 },
    { 0x4, 4 },
    { 0x5, 4 },
    { 0xA, 4 },
    { 0x6, 4 },
    { 0x1A, 5 },
    { 0x11, 5 },
    { 0x7, 4 },
    { 0xC, 4 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0x9, 4 },
    { 0x17, 5 },
    { 0x6F, 7 },
    { 0x6D, 7 },
    { 0x364, 10 },
    { 0xD9A, 12 },
    { 0x6CA, 11 },
    { 0x1B37, 13 },
    { 0x366C, 14 },
    { 0x42, 7 },
    { 0xD8, 8 }
  },
  {  /* DC bias table 3 */
    { 0x0, 4 },
    { 0x2D, 6 },
    { 0xF7, 8 },
    { 0x58, 7 },
    { 0x167, 9 },
    { 0x2CB, 10 },
    { 0x2CA, 10 },
    { 0xE, 6 },
    { 0x1661, 13 },
    { 0x3, 3 },
    { 0x2, 3 },
    { 0x8, 4 },
    { 0x9, 4 },
    { 0xD, 4 },
    { 0x2, 4 },
    { 0x1F, 5 },
    { 0x17, 5 },
    { 0x1, 4 },
    { 0xC, 4 },
    { 0xE, 4 },
    { 0xA, 4 },
    { 0x6, 5 },
    { 0x78, 7 },
    { 0xF, 6 },
    { 0x7A, 7 },
    { 0x164, 9 },
    { 0x599, 11 },
    { 0x2CD, 10 },
    { 0xB31, 12 },
    { 0x1660, 13 },
    { 0x79, 7 },
    { 0xF6, 8 }
  },
  {  /* DC bias table 4 */
    { 0x3, 4 },
    { 0x3C, 6 },
    { 0xF, 7 },
    { 0x7A, 7 },
    { 0x1D, 8 },
    { 0x20, 9 },
    { 0x72, 10 },
    { 0x6, 6 },
    { 0x399, 13 },
    { 0x4, 3 },
    { 0x5, 3 },
    { 0x5, 4 },
    { 0x6, 4 },
    { 0xE, 4 },
    { 0x4, 4 },
    { 0x0, 4 },
    { 0x19, 5 },
    { 0x2, 4 },
    { 0xD, 4 },
    { 0x7, 4 },
    { 0x1F, 5 },
    { 0x30, 6 },
    { 0x11, 8 },
    { 0x31, 6 },
    { 0x5, 6 },
    { 0x21, 9 },
    { 0xE7, 11 },
    { 0x38, 9 },
    { 0x1CD, 12 },
    { 0x398, 13 },
    { 0x7B, 7 },
    { 0x9, 7 }
  },
  {  /* DC bias table 5 */
    { 0x9, 4 },
    { 0x2, 5 },
    { 0x74, 7 },
    { 0x7, 6 },
    { 0xEC, 8 },
    { 0xD1, 9 },
    { 0x1A6, 10 },
    { 0x6, 6 },
    { 0xD21, 13 },
    { 0x5, 3 },
    { 0x6, 3 },
    { 0x8, 4 },
    { 0x7, 4 },
    { 0xF, 4 },
    { 0x4, 4 },
    { 0x0, 4 },
    { 0x1C, 5 },
    { 0x2, 4 },
    { 0x5, 4 },
    { 0x3, 4 },
    { 0xC, 5 },
    { 0x35, 7 },
    { 0x1A7, 10 },
    { 0x1B, 6 },
    { 0x77, 7 },
    { 0x1A5, 10 },
    { 0x349, 11 },
    { 0xD0, 9 },
    { 0x691, 12 },
    { 0xD20, 13 },
    { 0x75, 7 },
    { 0xED, 8 }
  },
  {  /* DC bias table 6 */
    { 0xA, 4 },
    { 0xC, 5 },
    { 0x12, 6 },
    { 0x1B, 6 },
    { 0xB7, 8 },
    { 0x16C, 9 },
    { 0x99, 9 },
    { 0x5A, 7 },
    { 0x16D8, 13 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x0, 3 },
    { 0x5, 4 },
    { 0x17, 5 },
    { 0xE, 5 },
    { 0x2, 4 },
    { 0x3, 4 },
    { 0xF, 5 },
    { 0x1A, 6 },
    { 0x4D, 8 },
    { 0x2DB3, 14 },
    { 0x2C, 6 },
    { 0x11, 6 },
    { 0x2DA, 10 },
    { 0x5B7, 11 },
    { 0x98, 9 },
    { 0xB6D, 12 },
    { 0x2DB2, 14 },
    { 0x10, 6 },
    { 0x27, 7 }
  },
  {  /* DC bias table 7 */
    { 0xD, 4 },
    { 0xF, 5 },
    { 0x1D, 6 },
    { 0x8, 5 },
    { 0x51, 7 },
    { 0x56, 8 },
    { 0xAF, 9 },
    { 0x2A, 7 },
    { 0x148A, 13 },
    { 0x7, 3 },
    { 0x0, 2 },
    { 0x8, 4 },
    { 0x9, 4 },
    { 0xC, 4 },
    { 0x6, 4 },
    { 0x17, 5 },
    { 0xB, 5 },
    { 0x16, 5 },
    { 0x15, 5 },
    { 0x9, 5 },
    { 0x50, 7 },
    { 0xAE, 9 },
    { 0x2917, 14 },
    { 0x1C, 6 },
    { 0x14, 6 },
    { 0x290, 10 },
    { 0x523, 11 },
    { 0x149, 9 },
    { 0xA44, 12 },
    { 0x2916, 14 },
    { 0x53, 7 },
    { 0xA5, 8 }
  },
  {  /* DC bias table 8 */
    { 0x1, 4 },
    { 0x1D, 6 },
    { 0xF5, 8 },
    { 0xF4, 8 },
    { 0x24D, 10 },
    { 0x499, 11 },
    { 0x498, 11 },
    { 0x1, 5 },
    { 0x21, 6 },
    { 0x6, 3 },
    { 0x5, 3 },
    { 0x6, 4 },
    { 0x5, 4 },
    { 0x2, 4 },
    { 0x7, 5 },
    { 0x25, 6 },
    { 0x7B, 7 },
    { 0x1C, 6 },
    { 0x20, 6 },
    { 0xD, 6 },
    { 0x48, 7 },
    { 0x92, 8 },
    { 0x127, 9 },
    { 0xE, 4 },
    { 0x4, 4 },
    { 0x11, 5 },
    { 0xC, 6 },
    { 0x3C, 6 },
    { 0xF, 5 },
    { 0x0, 5 },
    { 0x1F, 5 },
    { 0x13, 5 }
  },
  {  /* DC bias table 9 */
    { 0x5, 4 },
    { 0x3C, 6 },
    { 0x40, 7 },
    { 0xD, 7 },
    { 0x31, 9 },
    { 0x61, 10 },
    { 0x60, 10 },
    { 0x2, 5 },
    { 0xF5, 8 },
    { 0x6, 3 },
    { 0x5, 3 },
    { 0x7, 4 },
    { 0x6, 4 },
    { 0x2, 4 },
    { 0x9, 5 },
    { 0x25, 6 },
    { 0x7, 6 },
    { 0x21, 6 },
    { 0x24, 6 },
    { 0x10, 6 },
    { 0x41, 7 },
    { 0xF4, 8 },
    { 0x19, 8 },
    { 0xE, 4 },
    { 0x3, 4 },
    { 0x11, 5 },
    { 0x11, 6 },
    { 0x3F, 6 },
    { 0x3E, 6 },
    { 0x7B, 7 },
    { 0x0, 4 },
    { 0x13, 5 }
  },
  {  /* DC bias table 10 */
    { 0xA, 4 },
    { 0x7, 5 },
    { 0x1, 6 },
    { 0x9, 6 },
    { 0x131, 9 },
    { 0x261, 10 },
    { 0x260, 10 },
    { 0x15, 6 },
    { 0x1, 7 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0x8, 4 },
    { 0x7, 4 },
    { 0x6, 4 },
    { 0x12, 5 },
    { 0x2F, 6 },
    { 0x14, 6 },
    { 0x27, 6 },
    { 0x2D, 6 },
    { 0x16, 6 },
    { 0x4D, 7 },
    { 0x99, 8 },
    { 0x0, 7 },
    { 0x4, 4 },
    { 0x1, 4 },
    { 0x5, 5 },
    { 0x17, 6 },
    { 0x2E, 6 },
    { 0x2C, 6 },
    { 0x8, 6 },
    { 0x6, 5 },
    { 0x1, 5 }
  },
  {  /* DC bias table 11 */
    { 0x0, 3 },
    { 0xE, 5 },
    { 0x17, 6 },
    { 0x2A, 6 },
    { 0x10, 7 },
    { 0xF9, 10 },
    { 0xF8, 10 },
    { 0x1E, 7 },
    { 0x3F, 8 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x6, 4 },
    { 0xF, 5 },
    { 0x5, 5 },
    { 0x16, 6 },
    { 0x29, 6 },
    { 0x2B, 6 },
    { 0x15, 6 },
    { 0x50, 7 },
    { 0x11, 7 },
    { 0x7D, 9 },
    { 0x4, 4 },
    { 0x17, 5 },
    { 0x6, 5 },
    { 0x14, 6 },
    { 0x2C, 6 },
    { 0x2D, 6 },
    { 0xE, 6 },
    { 0x9, 6 },
    { 0x51, 7 }
  },
  {  /* DC bias table 12 */
    { 0x2, 3 },
    { 0x18, 5 },
    { 0x2F, 6 },
    { 0xD, 5 },
    { 0x53, 7 },
    { 0x295, 10 },
    { 0x294, 10 },
    { 0xA4, 8 },
    { 0x7C, 8 },
    { 0x0, 2 },
    { 0x7, 3 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x1B, 5 },
    { 0xC, 5 },
    { 0x28, 6 },
    { 0x6A, 7 },
    { 0x1E, 6 },
    { 0x1D, 6 },
    { 0x69, 7 },
    { 0xD7, 8 },
    { 0x7D, 8 },
    { 0x14B, 9 },
    { 0x19, 5 },
    { 0x16, 5 },
    { 0x2E, 6 },
    { 0x1C, 6 },
    { 0x2B, 6 },
    { 0x2A, 6 },
    { 0x68, 7 },
    { 0x3F, 7 },
    { 0xD6, 8 }
  },
  {  /* DC bias table 13 */
    { 0x2, 3 },
    { 0x1B, 5 },
    { 0xC, 5 },
    { 0x18, 5 },
    { 0x29, 6 },
    { 0x7F, 8 },
    { 0x2F0, 10 },
    { 0x198, 9 },
    { 0x179, 9 },
    { 0x0, 2 },
    { 0x7, 3 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x1A, 5 },
    { 0xD, 5 },
    { 0x2A, 6 },
    { 0x64, 7 },
    { 0x1E, 6 },
    { 0x67, 7 },
    { 0x5F, 7 },
    { 0xCD, 8 },
    { 0x7E, 8 },
    { 0x2F1, 10 },
    { 0x16, 5 },
    { 0xE, 5 },
    { 0x2E, 6 },
    { 0x65, 7 },
    { 0x2B, 6 },
    { 0x28, 6 },
    { 0x3E, 7 },
    { 0xBD, 8 },
    { 0x199, 9 }
  },
  {  /* DC bias table 14 */
    { 0x2, 3 },
    { 0x7, 4 },
    { 0x16, 5 },
    { 0x6, 4 },
    { 0x36, 6 },
    { 0x5C, 7 },
    { 0x15D, 9 },
    { 0x15C, 9 },
    { 0x2BF, 10 },
    { 0x0, 2 },
    { 0x7, 3 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x18, 5 },
    { 0x34, 6 },
    { 0x2A, 6 },
    { 0x5E, 7 },
    { 0x6A, 7 },
    { 0x64, 7 },
    { 0x5D, 7 },
    { 0xCB, 8 },
    { 0xAD, 8 },
    { 0x2BE, 10 },
    { 0x14, 5 },
    { 0x33, 6 },
    { 0x6E, 7 },
    { 0x5F, 7 },
    { 0x6F, 7 },
    { 0x6B, 7 },
    { 0xCA, 8 },
    { 0xAC, 8 },
    { 0x15E, 9 }
  },
  {  /* DC bias table 15 */
    { 0xF, 4 },
    { 0x1D, 5 },
    { 0x18, 5 },
    { 0xB, 4 },
    { 0x19, 5 },
    { 0x29, 6 },
    { 0xD6, 8 },
    { 0x551, 11 },
    { 0xAA1, 12 },
    { 0x1, 2 },
    { 0x0, 2 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x1B, 5 },
    { 0x38, 6 },
    { 0x28, 6 },
    { 0x57, 7 },
    { 0x6A, 7 },
    { 0x68, 7 },
    { 0x56, 7 },
    { 0xE5, 8 },
    { 0x155, 9 },
    { 0xAA0, 12 },
    { 0x73, 7 },
    { 0x69, 7 },
    { 0xD7, 8 },
    { 0xAB, 8 },
    { 0xE4, 8 },
    { 0xA9, 8 },
    { 0x151, 9 },
    { 0x150, 9 },
    { 0x2A9, 10 }
  }
};

static const uint16_t ac_bias_0[16][32][2] = {
  {  /* AC bias group 1, table 0 */
    { 0x8, 5 },
    { 0x25, 7 },
    { 0x17A, 9 },
    { 0x2F7, 10 },
    { 0xBDB, 12 },
    { 0x17B4, 13 },
    { 0x2F6B, 14 },
    { 0x1D, 5 },
    { 0x2F6A, 14 },
    { 0x8, 4 },
    { 0x7, 4 },
    { 0x1, 4 },
    { 0x2, 4 },
    { 0xA, 4 },
    { 0x6, 4 },
    { 0x0, 4 },
    { 0x1C, 5 },
    { 0x9, 4 },
    { 0xD, 4 },
    { 0xF, 4 },
    { 0xC, 4 },
    { 0x3, 4 },
    { 0xA, 5 },
    { 0x16, 5 },
    { 0x13, 6 },
    { 0x5D, 7 },
    { 0x24, 7 },
    { 0xBC, 8 },
    { 0x5C, 7 },
    { 0x5EC, 11 },
    { 0xB, 5 },
    { 0x5F, 7 }
  },
  {  /* AC bias group 1, table 1 */
    { 0xF, 5 },
    { 0x10, 6 },
    { 0x4B, 8 },
    { 0xC6, 8 },
    { 0x31D, 10 },
    { 0xC71, 12 },
    { 0xC70, 12 },
    { 0x1, 4 },
    { 0xC73, 12 },
    { 0x8, 4 },
    { 0x9, 4 },
    { 0x2, 4 },
    { 0x3, 4 },
    { 0xB, 4 },
    { 0x6, 4 },
    { 0x0, 4 },
    { 0x1C, 5 },
    { 0x5, 4 },
    { 0xD, 4 },
    { 0xF, 4 },
    { 0xA, 4 },
    { 0x19, 5 },
    { 0x13, 6 },
    { 0x1D, 5 },
    { 0x30, 6 },
    { 0x62, 7 },
    { 0x24, 7 },
    { 0x4A, 8 },
    { 0x18F, 9 },
    { 0xC72, 12 },
    { 0xE, 5 },
    { 0x11, 6 }
  },
  {  /* AC bias group 1, table 2 */
    { 0x1B, 5 },
    { 0x3, 6 },
    { 0x8D, 8 },
    { 0x40, 7 },
    { 0x239, 10 },
    { 0x471, 11 },
    { 0x8E0, 12 },
    { 0x3, 4 },
    { 0x11C3, 13 },
    { 0xA, 4 },
    { 0x9, 4 },
    { 0x4, 4 },
    { 0x5, 4 },
    { 0xE, 4 },
    { 0x7, 4 },
    { 0x1, 4 },
    { 0x1E, 5 },
    { 0x6, 4 },
    { 0xC, 4 },
    { 0xB, 4 },
    { 0x2, 4 },
    { 0x0, 5 },
    { 0x41, 7 },
    { 0x1F, 5 },
    { 0x22, 6 },
    { 0x2, 6 },
    { 0x8F, 8 },
    { 0x8C, 8 },
    { 0x11D, 9 },
    { 0x11C2, 13 },
    { 0x1A, 5 },
    { 0x21, 6 }
  },
  {  /* AC bias group 1, table 3 */
    { 0x1F, 5 },
    { 0x3, 6 },
    { 0x3, 7 },
    { 0x43, 7 },
    { 0xB, 9 },
    { 0x15, 10 },
    { 0x51, 12 },
    { 0x3, 4 },
    { 0x50, 12 },
    { 0xD, 4 },
    { 0xC, 4 },
    { 0x4, 4 },
    { 0x6, 4 },
    { 0xE, 4 },
    { 0xA, 4 },
    { 0x1, 4 },
    { 0x1E, 5 },
    { 0x5, 4 },
    { 0x9, 4 },
    { 0x7, 4 },
    { 0x11, 5 },
    { 0x2, 6 },
    { 0x4, 8 },
    { 0x2, 4 },
    { 0x2D, 6 },
    { 0x20, 6 },
    { 0x42, 7 },
    { 0x1, 7 },
    { 0x0, 7 },
    { 0x29, 11 },
    { 0x17, 5 },
    { 0x2C, 6 }
  },
  {  /* AC bias group 1, table 4 */
    { 0x3, 4 },
    { 0x1F, 6 },
    { 0x3A, 7 },
    { 0x5D, 7 },
    { 0x173, 9 },
    { 0x2E4, 10 },
    { 0x172D, 13 },
    { 0x4, 4 },
    { 0x172C, 13 },
    { 0xF, 4 },
    { 0xE, 4 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0xC, 4 },
    { 0xA, 4 },
    { 0x1, 4 },
    { 0x16, 5 },
    { 0x2, 4 },
    { 0x5, 4 },
    { 0x1A, 5 },
    { 0x2F, 6 },
    { 0x38, 7 },
    { 0x5CA, 11 },
    { 0x6, 4 },
    { 0x37, 6 },
    { 0x1E, 6 },
    { 0x3B, 7 },
    { 0x39, 7 },
    { 0xB8, 8 },
    { 0xB97, 12 },
    { 0x0, 4 },
    { 0x36, 6 }
  },
  {  /* AC bias group 1, table 5 */
    { 0x6, 4 },
    { 0x37, 6 },
    { 0x5D, 7 },
    { 0xC, 6 },
    { 0xB9, 8 },
    { 0x2E3, 10 },
    { 0x5C4, 11 },
    { 0x4, 4 },
    { 0x1715, 13 },
    { 0x0, 3 },
    { 0xF, 4 },
    { 0x8, 4 },
    { 0x7, 4 },
    { 0xC, 4 },
    { 0x9, 4 },
    { 0x1D, 5 },
    { 0x16, 5 },
    { 0x1C, 5 },
    { 0x1A, 5 },
    { 0xB, 5 },
    { 0x5E, 7 },
    { 0x170, 9 },
    { 0x1714, 13 },
    { 0xA, 4 },
    { 0xA, 5 },
    { 0x36, 6 },
    { 0x5F, 7 },
    { 0x1B, 7 },
    { 0x1A, 7 },
    { 0xB8B, 12 },
    { 0x2, 4 },
    { 0x7, 5 }
  },
  {  /* AC bias group 1, table 6 */
    { 0xC, 4 },
    { 0xB, 5 },
    { 0x79, 7 },
    { 0x22, 6 },
    { 0xF0, 8 },
    { 0x119, 9 },
    { 0x230, 10 },
    { 0x1D, 5 },
    { 0x8C4, 12 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xA, 4 },
    { 0x9, 4 },
    { 0xB, 4 },
    { 0x7, 4 },
    { 0x1C, 5 },
    { 0x3D, 6 },
    { 0xD, 5 },
    { 0x8, 5 },
    { 0x15, 6 },
    { 0x8D, 8 },
    { 0x118B, 13 },
    { 0x118A, 13 },
    { 0xD, 4 },
    { 0x10, 5 },
    { 0x9, 5 },
    { 0x14, 6 },
    { 0x47, 7 },
    { 0xF1, 8 },
    { 0x463, 11 },
    { 0x1F, 5 },
    { 0xC, 5 }
  },
  {  /* AC bias group 1, table 7 */
    { 0x0, 3 },
    { 0x1A, 5 },
    { 0x33, 6 },
    { 0xC, 5 },
    { 0x46, 7 },
    { 0x1E3, 9 },
    { 0x3C5, 10 },
    { 0x17, 5 },
    { 0x1E21, 13 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x9, 4 },
    { 0xA, 4 },
    { 0x7, 4 },
    { 0x1B, 5 },
    { 0x3D, 6 },
    { 0x1B, 6 },
    { 0x22, 6 },
    { 0x79, 7 },
    { 0xF0, 8 },
    { 0x1E20, 13 },
    { 0x1E23, 13 },
    { 0x1E22, 13 },
    { 0xE, 4 },
    { 0x16, 5 },
    { 0x18, 5 },
    { 0x32, 6 },
    { 0x1A, 6 },
    { 0x47, 7 },
    { 0x789, 11 },
    { 0x1F, 5 },
    { 0x10, 5 }
  },
  {  /* AC bias group 1, table 8 */
    { 0x1D, 5 },
    { 0x61, 7 },
    { 0x4E, 8 },
    { 0x9E, 9 },
    { 0x27C, 11 },
    { 0x9F5, 13 },
    { 0x9F4, 13 },
    { 0x3, 4 },
    { 0x60, 7 },
    { 0x0, 3 },
    { 0xF, 4 },
    { 0xB, 4 },
    { 0xA, 4 },
    { 0x9, 4 },
    { 0x5, 4 },
    { 0xD, 5 },
    { 0x31, 6 },
    { 0x8, 5 },
    { 0x38, 6 },
    { 0x12, 6 },
    { 0x26, 7 },
    { 0x13F, 10 },
    { 0x4FB, 12 },
    { 0xD, 4 },
    { 0x2, 4 },
    { 0xC, 5 },
    { 0x39, 6 },
    { 0x1C, 6 },
    { 0xF, 5 },
    { 0x1D, 6 },
    { 0x8, 4 },
    { 0x19, 5 }
  },
  {  /* AC bias group 1, table 9 */
    { 0x7, 4 },
    { 0x19, 6 },
    { 0xAB, 8 },
    { 0xAA, 8 },
    { 0x119, 10 },
    { 0x461, 12 },
    { 0x460, 12 },
    { 0x1B, 5 },
    { 0x47, 8 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xC, 4 },
    { 0xB, 4 },
    { 0x9, 4 },
    { 0x5, 4 },
    { 0xD, 5 },
    { 0x35, 6 },
    { 0x3D, 6 },
    { 0x3C, 6 },
    { 0x18, 6 },
    { 0x22, 7 },
    { 0x8D, 9 },
    { 0x231, 11 },
    { 0xE, 4 },
    { 0x1F, 5 },
    { 0x9, 5 },
    { 0x2B, 6 },
    { 0x10, 6 },
    { 0x34, 6 },
    { 0x54, 7 },
    { 0x8, 4 },
    { 0x14, 5 }
  },
  {  /* AC bias group 1, table 10 */
    { 0xC, 4 },
    { 0x5, 5 },
    { 0x8, 6 },
    { 0x5B, 7 },
    { 0x4D, 9 },
    { 0x131, 11 },
    { 0x261, 12 },
    { 0x1A, 5 },
    { 0x12, 7 },
    { 0x0, 3 },
    { 0xF, 4 },
    { 0xA, 4 },
    { 0x9, 4 },
    { 0x6, 4 },
    { 0x1B, 5 },
    { 0x6, 5 },
    { 0x1C, 6 },
    { 0x2C, 6 },
    { 0x15, 6 },
    { 0x5A, 7 },
    { 0x27, 8 },
    { 0x99, 10 },
    { 0x260, 12 },
    { 0xE, 4 },
    { 0x4, 4 },
    { 0xF, 5 },
    { 0x7, 5 },
    { 0x1D, 6 },
    { 0xB, 5 },
    { 0x14, 6 },
    { 0x8, 4 },
    { 0x17, 5 }
  },
  {  /* AC bias group 1, table 11 */
    { 0xF, 4 },
    { 0x13, 5 },
    { 0x75, 7 },
    { 0x24, 6 },
    { 0x95, 8 },
    { 0x251, 10 },
    { 0x4A0, 11 },
    { 0x10, 5 },
    { 0xC8, 8 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x1, 4 },
    { 0x0, 4 },
    { 0x1A, 5 },
    { 0x11, 5 },
    { 0x2C, 6 },
    { 0x65, 7 },
    { 0x74, 7 },
    { 0x4B, 7 },
    { 0xC9, 8 },
    { 0x129, 9 },
    { 0x943, 12 },
    { 0x942, 12 },
    { 0x3, 3 },
    { 0xA, 4 },
    { 0x1C, 5 },
    { 0x18, 5 },
    { 0x33, 6 },
    { 0x17, 5 },
    { 0x2D, 6 },
    { 0x1B, 5 },
    { 0x3B, 6 }
  },
  {  /* AC bias group 1, table 12 */
    { 0x3, 3 },
    { 0x1A, 5 },
    { 0x2D, 6 },
    { 0x38, 6 },
    { 0x28, 7 },
    { 0x395, 10 },
    { 0xE51, 12 },
    { 0x37, 6 },
    { 0xE4, 8 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0x1F, 5 },
    { 0x1E, 5 },
    { 0x17, 5 },
    { 0x3A, 6 },
    { 0x73, 7 },
    { 0x2A, 7 },
    { 0x2B, 7 },
    { 0x29, 7 },
    { 0x1CB, 9 },
    { 0x729, 11 },
    { 0x1CA1, 13 },
    { 0x1CA0, 13 },
    { 0x4, 3 },
    { 0xA, 4 },
    { 0x4, 4 },
    { 0x18, 5 },
    { 0x36, 6 },
    { 0xB, 5 },
    { 0x2C, 6 },
    { 0x19, 5 },
    { 0x3B, 6 }
  },
  {  /* AC bias group 1, table 13 */
    { 0x4, 3 },
    { 0x4, 4 },
    { 0x3F, 6 },
    { 0x17, 5 },
    { 0x75, 7 },
    { 0x1F5, 9 },
    { 0x7D1, 11 },
    { 0x17, 6 },
    { 0x1F6, 9 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0x1B, 5 },
    { 0x1A, 5 },
    { 0xA, 5 },
    { 0x32, 6 },
    { 0x74, 7 },
    { 0xF8, 8 },
    { 0xF9, 8 },
    { 0x1F7, 9 },
    { 0x3E9, 10 },
    { 0xFA0, 12 },
    { 0x1F43, 13 },
    { 0x1F42, 13 },
    { 0x3, 3 },
    { 0xA, 4 },
    { 0x1E, 5 },
    { 0x1C, 5 },
    { 0x3B, 6 },
    { 0x18, 5 },
    { 0x16, 6 },
    { 0x16, 5 },
    { 0x33, 6 }
  },
  {  /* AC bias group 1, table 14 */
    { 0x4, 3 },
    { 0x7, 4 },
    { 0x18, 5 },
    { 0x1E, 5 },
    { 0x36, 6 },
    { 0x31, 7 },
    { 0x177, 9 },
    { 0x77, 7 },
    { 0x176, 9 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0x1A, 5 },
    { 0x19, 5 },
    { 0x3A, 6 },
    { 0x19, 6 },
    { 0x5C, 7 },
    { 0xBA, 8 },
    { 0x61, 8 },
    { 0xC1, 9 },
    { 0x180, 10 },
    { 0x302, 11 },
    { 0x607, 12 },
    { 0x606, 12 },
    { 0x2, 3 },
    { 0xA, 4 },
    { 0x1F, 5 },
    { 0x1C, 5 },
    { 0x37, 6 },
    { 0x16, 5 },
    { 0x76, 7 },
    { 0xD, 5 },
    { 0x2F, 6 }
  },
  {  /* AC bias group 1, table 15 */
    { 0x0, 3 },
    { 0xA, 4 },
    { 0x1A, 5 },
    { 0xC, 4 },
    { 0x1D, 5 },
    { 0x39, 6 },
    { 0x78, 7 },
    { 0x5E, 7 },
    { 0x393, 11 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x16, 5 },
    { 0xF, 5 },
    { 0x2E, 6 },
    { 0x5F, 7 },
    { 0x73, 8 },
    { 0xE5, 9 },
    { 0x1C8, 10 },
    { 0xE4A, 13 },
    { 0x1C97, 14 },
    { 0x1C96, 14 },
    { 0xE49, 13 },
    { 0xE48, 13 },
    { 0x4, 3 },
    { 0x6, 4 },
    { 0x1F, 5 },
    { 0x1B, 5 },
    { 0x1D, 6 },
    { 0x38, 6 },
    { 0x38, 7 },
    { 0x3D, 6 },
    { 0x79, 7 }
  }
};

static const uint16_t ac_bias_1[16][32][2] = {
  {  /* AC bias group 2, table 0 */
    { 0xB, 5 },
    { 0x2B, 7 },
    { 0x54, 8 },
    { 0x1B7, 9 },
    { 0x6D9, 11 },
    { 0xDB1, 12 },
    { 0xDB0, 12 },
    { 0x2, 4 },
    { 0xAB, 9 },
    { 0x9, 4 },
    { 0xA, 4 },
    { 0x7, 4 },
    { 0x8, 4 },
    { 0xF, 4 },
    { 0xC, 4 },
    { 0x3, 4 },
    { 0x1D, 5 },
    { 0x4, 4 },
    { 0xB, 4 },
    { 0x6, 4 },
    { 0x1A, 5 },
    { 0x3, 6 },
    { 0xAA, 9 },
    { 0x1, 4 },
    { 0x0, 5 },
    { 0x14, 6 },
    { 0x6C, 7 },
    { 0xDA, 8 },
    { 0x2, 6 },
    { 0x36D, 10 },
    { 0x1C, 5 },
    { 0x37, 6 }
  },
  {  /* AC bias group 2, table 1 */
    { 0x1D, 5 },
    { 0x4, 6 },
    { 0xB6, 8 },
    { 0x6A, 8 },
    { 0x5B9, 11 },
    { 0x16E1, 13 },
    { 0x16E0, 13 },
    { 0x7, 4 },
    { 0x16F, 9 },
    { 0xC, 4 },
    { 0xD, 4 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0xF, 4 },
    { 0xA, 4 },
    { 0x3, 4 },
    { 0x17, 5 },
    { 0x2, 4 },
    { 0x4, 4 },
    { 0x1C, 5 },
    { 0x2C, 6 },
    { 0x6B, 8 },
    { 0xB71, 12 },
    { 0x5, 4 },
    { 0x3, 5 },
    { 0x1B, 6 },
    { 0x5A, 7 },
    { 0x34, 7 },
    { 0x5, 6 },
    { 0x2DD, 10 },
    { 0x0, 4 },
    { 0xC, 5 }
  },
  {  /* AC bias group 2, table 2 */
    { 0x3, 4 },
    { 0x7F, 7 },
    { 0xA1, 8 },
    { 0xA0, 8 },
    { 0x20C, 10 },
    { 0x834, 12 },
    { 0x106B, 13 },
    { 0x7, 4 },
    { 0x82, 8 },
    { 0xE, 4 },
    { 0xD, 4 },
    { 0xB, 4 },
    { 0xC, 4 },
    { 0x0, 3 },
    { 0x9, 4 },
    { 0x2, 4 },
    { 0x11, 5 },
    { 0x1E, 5 },
    { 0x15, 5 },
    { 0x3E, 6 },
    { 0x40, 7 },
    { 0x41B, 11 },
    { 0x106A, 13 },
    { 0x6, 4 },
    { 0xA, 5 },
    { 0x29, 6 },
    { 0x7E, 7 },
    { 0x51, 7 },
    { 0x21, 6 },
    { 0x107, 9 },
    { 0x4, 4 },
    { 0xB, 5 }
  },
  {  /* AC bias group 2, table 3 */
    { 0x7, 4 },
    { 0x1B, 6 },
    { 0xF6, 8 },
    { 0xE9, 8 },
    { 0x3A1, 10 },
    { 0x740, 11 },
    { 0xE82, 12 },
    { 0x1F, 5 },
    { 0x1EF, 9 },
    { 0x1, 3 },
    { 0x2, 3 },
    { 0xB, 4 },
    { 0xC, 4 },
    { 0xD, 4 },
    { 0x8, 4 },
    { 0x1C, 5 },
    { 0x3, 5 },
    { 0x12, 5 },
    { 0x2, 5 },
    { 0x75, 7 },
    { 0x1D1, 9 },
    { 0x1D07, 13 },
    { 0x1D06, 13 },
    { 0xA, 4 },
    { 0x13, 5 },
    { 0x3B, 6 },
    { 0x1A, 6 },
    { 0x7A, 7 },
    { 0x3C, 6 },
    { 0x1EE, 9 },
    { 0x0, 4 },
    { 0xC, 5 }
  },
  {  /* AC bias group 2, table 4 */
    { 0xD, 4 },
    { 0x3D, 6 },
    { 0x42, 7 },
    { 0x37, 7 },
    { 0xD9, 9 },
    { 0x362, 11 },
    { 0x6C6, 12 },
    { 0x1F, 5 },
    { 0x86, 8 },
    { 0x1, 3 },
    { 0x2, 3 },
    { 0xC, 4 },
    { 0xB, 4 },
    { 0xA, 4 },
    { 0x1, 4 },
    { 0xF, 5 },
    { 0x25, 6 },
    { 0x3C, 6 },
    { 0x1A, 6 },
    { 0x87, 8 },
    { 0x1B0, 10 },
    { 0xD8F, 13 },
    { 0xD8E, 13 },
    { 0xE, 4 },
    { 0x13, 5 },
    { 0xC, 5 },
    { 0x24, 6 },
    { 0x20, 6 },
    { 0x11, 5 },
    { 0x6D, 8 },
    { 0x0, 4 },
    { 0xE, 5 }
  },
  {  /* AC bias group 2, table 5 */
    { 0x0, 3 },
    { 0x12, 5 },
    { 0x76, 7 },
    { 0x77, 7 },
    { 0x14D, 9 },
    { 0x533, 11 },
    { 0x14C9, 13 },
    { 0x13, 5 },
    { 0xA5, 8 },
    { 0x2, 3 },
    { 0x3, 3 },
    { 0xB, 4 },
    { 0xC, 4 },
    { 0x8, 4 },
    { 0x1A, 5 },
    { 0x2B, 6 },
    { 0x75, 7 },
    { 0x74, 7 },
    { 0xA7, 8 },
    { 0x298, 10 },
    { 0x14C8, 13 },
    { 0x14CB, 13 },
    { 0x14CA, 13 },
    { 0xF, 4 },
    { 0x1C, 5 },
    { 0x7, 5 },
    { 0x2A, 6 },
    { 0x28, 6 },
    { 0x1B, 5 },
    { 0xA4, 8 },
    { 0x2, 4 },
    { 0x6, 5 }
  },
  {  /* AC bias group 2, table 6 */
    { 0x2, 3 },
    { 0x1A, 5 },
    { 0x2B, 6 },
    { 0x3A, 6 },
    { 0xED, 8 },
    { 0x283, 10 },
    { 0xA0A, 12 },
    { 0x4, 5 },
    { 0xA1, 8 },
    { 0x4, 3 },
    { 0x3, 3 },
    { 0xB, 4 },
    { 0xC, 4 },
    { 0x1F, 5 },
    { 0x6, 5 },
    { 0x77, 7 },
    { 0xA3, 8 },
    { 0xA2, 8 },
    { 0x140, 9 },
    { 0x1417, 13 },
    { 0x1416, 13 },
    { 0xA09, 12 },
    { 0xA08, 12 },
    { 0x0, 3 },
    { 0x1E, 5 },
    { 0x7, 5 },
    { 0x2A, 6 },
    { 0x29, 6 },
    { 0x1C, 5 },
    { 0xEC, 8 },
    { 0x1B, 5 },
    { 0x5, 5 }
  },
  {  /* AC bias group 2, table 7 */
    { 0x2, 3 },
    { 0x2, 4 },
    { 0x18, 5 },
    { 0x1D, 5 },
    { 0x35, 6 },
    { 0xE4, 8 },
    { 0x1CF, 11 },
    { 0x1D, 7 },
    { 0x72, 9 },
    { 0x4, 3 },
    { 0x5, 3 },
    { 0x6, 4 },
    { 0x7, 4 },
    { 0x6, 5 },
    { 0x73, 7 },
    { 0x38, 8 },
    { 0x1CE, 11 },
    { 0x39B, 12 },
    { 0x398, 12 },
    { 0x733, 13 },
    { 0x732, 13 },
    { 0x735, 13 },
    { 0x734, 13 },
    { 0x0, 3 },
    { 0x1F, 5 },
    { 0x1B, 5 },
    { 0x34, 6 },
    { 0xF, 6 },
    { 0x1E, 5 },
    { 0xE5, 8 },
    { 0x19, 5 },
    { 0x38, 6 }
  },
  {  /* AC bias group 2, table 8 */
    { 0x16, 5 },
    { 0x50, 7 },
    { 0x172, 9 },
    { 0x2E7, 10 },
    { 0x1732, 13 },
    { 0x2E67, 14 },
    { 0x2E66, 14 },
    { 0x6, 4 },
    { 0x51, 7 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xD, 4 },
    { 0xC, 4 },
    { 0x9, 4 },
    { 0x1C, 5 },
    { 0x9, 5 },
    { 0x1C, 6 },
    { 0x1D, 6 },
    { 0x5D, 7 },
    { 0xB8, 8 },
    { 0x5CD, 11 },
    { 0x1731, 13 },
    { 0x1730, 13 },
    { 0xF, 4 },
    { 0x5, 4 },
    { 0xF, 5 },
    { 0x8, 5 },
    { 0x29, 6 },
    { 0x1D, 5 },
    { 0x2F, 6 },
    { 0x8, 4 },
    { 0x15, 5 }
  },
  {  /* AC bias group 2, table 9 */
    { 0x9, 4 },
    { 0x21, 6 },
    { 0x40, 7 },
    { 0xAD, 8 },
    { 0x2B0, 10 },
    { 0x1589, 13 },
    { 0x1588, 13 },
    { 0x1C, 5 },
    { 0x5F, 7 },
    { 0x0, 3 },
    { 0xF, 4 },
    { 0xD, 4 },
    { 0xC, 4 },
    { 0x6, 4 },
    { 0x11, 5 },
    { 0x2A, 6 },
    { 0x57, 7 },
    { 0x5E, 7 },
    { 0x41, 7 },
    { 0x159, 9 },
    { 0x563, 11 },
    { 0x158B, 13 },
    { 0x158A, 13 },
    { 0x1, 3 },
    { 0x5, 4 },
    { 0x14, 5 },
    { 0x3B, 6 },
    { 0x2E, 6 },
    { 0x4, 4 },
    { 0x3A, 6 },
    { 0x7, 4 },
    { 0x16, 5 }
  },
  {  /* AC bias group 2, table 10 */
    { 0xE, 4 },
    { 0x7, 5 },
    { 0x46, 7 },
    { 0x45, 7 },
    { 0x64, 9 },
    { 0x32A, 12 },
    { 0x657, 13 },
    { 0x18, 5 },
    { 0xD, 6 },
    { 0x0, 3 },
    { 0xF, 4 },
    { 0xA, 4 },
    { 0xB, 4 },
    { 0x1A, 5 },
    { 0x36, 6 },
    { 0x47, 7 },
    { 0x44, 7 },
    { 0x18, 7 },
    { 0x33, 8 },
    { 0xCB, 10 },
    { 0x656, 13 },
    { 0x329, 12 },
    { 0x328, 12 },
    { 0x2, 3 },
    { 0x6, 4 },
    { 0x19, 5 },
    { 0xE, 5 },
    { 0x37, 6 },
    { 0x9, 4 },
    { 0xF, 5 },
    { 0x2, 4 },
    { 0x10, 5 }
  },
  {  /* AC bias group 2, table 11 */
    { 0x3, 3 },
    { 0x18, 5 },
    { 0x23, 6 },
    { 0x77, 7 },
    { 0x194, 9 },
    { 0x1956, 13 },
    { 0x32AF, 14 },
    { 0x3A, 6 },
    { 0x76, 7 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x1F, 5 },
    { 0x1E, 5 },
    { 0x14, 5 },
    { 0x22, 6 },
    { 0x64, 7 },
    { 0x197, 9 },
    { 0x196, 9 },
    { 0x32B, 10 },
    { 0x654, 11 },
    { 0x32AE, 14 },
    { 0x1955, 13 },
    { 0x1954, 13 },
    { 0x0, 3 },
    { 0x9, 4 },
    { 0x1C, 5 },
    { 0x15, 5 },
    { 0x10, 5 },
    { 0xD, 4 },
    { 0x17, 5 },
    { 0x16, 5 },
    { 0x33, 6 }
  },
  {  /* AC bias group 2, table 12 */
    { 0x5, 3 },
    { 0x6, 4 },
    { 0x3E, 6 },
    { 0x10, 5 },
    { 0x48, 7 },
    { 0x93F, 12 },
    { 0x24FA, 14 },
    { 0x32, 6 },
    { 0x67, 7 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x1B, 5 },
    { 0x1E, 5 },
    { 0x34, 6 },
    { 0x66, 7 },
    { 0x92, 8 },
    { 0x126, 9 },
    { 0x24E, 10 },
    { 0x49E, 11 },
    { 0x49F7, 15 },
    { 0x49F6, 15 },
    { 0x24F9, 14 },
    { 0x24F8, 14 },
    { 0x0, 3 },
    { 0x7, 4 },
    { 0x18, 5 },
    { 0x11, 5 },
    { 0x3F, 6 },
    { 0xE, 4 },
    { 0x13, 5 },
    { 0x35, 6 },
    { 0x25, 6 }
  },
  {  /* AC bias group 2, table 13 */
    { 0x5, 3 },
    { 0x8, 4 },
    { 0x12, 5 },
    { 0x1C, 5 },
    { 0x1C, 6 },
    { 0xEA, 9 },
    { 0x1D75, 14 },
    { 0x1E, 6 },
    { 0x66, 7 },
    { 0x1, 3 },
    { 0x2, 3 },
    { 0x1B, 5 },
    { 0x1A, 5 },
    { 0x1F, 6 },
    { 0x3B, 7 },
    { 0x74, 8 },
    { 0x1D6, 10 },
    { 0x3AF, 11 },
    { 0x1D74, 14 },
    { 0x1D77, 14 },
    { 0x1D76, 14 },
    { 0xEB9, 13 },
    { 0xEB8, 13 },
    { 0xF, 4 },
    { 0x6, 4 },
    { 0x13, 5 },
    { 0x3B, 6 },
    { 0x3A, 6 },
    { 0x0, 3 },
    { 0x18, 5 },
    { 0x32, 6 },
    { 0x67, 7 }
  },
  {  /* AC bias group 2, table 14 */
    { 0x4, 3 },
    { 0xA, 4 },
    { 0x1B, 5 },
    { 0xC, 4 },
    { 0xD, 5 },
    { 0xE6, 8 },
    { 0x684, 11 },
    { 0x72, 7 },
    { 0xE7, 8 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x17, 5 },
    { 0x16, 5 },
    { 0x18, 6 },
    { 0xD1, 8 },
    { 0x1A0, 9 },
    { 0x686, 11 },
    { 0xD0F, 12 },
    { 0xD0A, 12 },
    { 0x1A17, 13 },
    { 0x1A16, 13 },
    { 0x1A1D, 13 },
    { 0x1A1C, 13 },
    { 0xF, 4 },
    { 0x1D, 5 },
    { 0xE, 5 },
    { 0x35, 6 },
    { 0x38, 6 },
    { 0x0, 3 },
    { 0xF, 5 },
    { 0x19, 6 },
    { 0x69, 7 }
  },
  {  /* AC bias group 2, table 15 */
    { 0x3, 3 },
    { 0xC, 4 },
    { 0x1B, 5 },
    { 0x0, 3 },
    { 0x3, 4 },
    { 0x2E, 6 },
    { 0x51, 9 },
    { 0xBC, 8 },
    { 0x53, 9 },
    { 0x4, 3 },
    { 0x2, 3 },
    { 0x16, 5 },
    { 0x15, 5 },
    { 0x15, 7 },
    { 0x50, 9 },
    { 0xA4, 10 },
    { 0x294, 12 },
    { 0x52B, 13 },
    { 0x52A, 13 },
    { 0x52D, 13 },
    { 0x52C, 13 },
    { 0x52F, 13 },
    { 0x52E, 13 },
    { 0xE, 4 },
    { 0x1A, 5 },
    { 0x4, 5 },
    { 0x28, 6 },
    { 0x29, 6 },
    { 0xF, 4 },
    { 0xB, 6 },
    { 0x5F, 7 },
    { 0xBD, 8 }
  }
};

static const uint16_t ac_bias_2[16][32][2] = {
  {  /* AC bias group 3, table 0 */
    { 0x3, 4 },
    { 0x9, 6 },
    { 0xD0, 8 },
    { 0x1A3, 9 },
    { 0x344, 10 },
    { 0xD14, 12 },
    { 0x1A2B, 13 },
    { 0x4, 4 },
    { 0x15, 7 },
    { 0x0, 3 },
    { 0xF, 4 },
    { 0xB, 4 },
    { 0xC, 4 },
    { 0xE, 4 },
    { 0x9, 4 },
    { 0x1B, 5 },
    { 0xA, 5 },
    { 0x14, 5 },
    { 0xD, 5 },
    { 0x2A, 6 },
    { 0x14, 7 },
    { 0x68B, 11 },
    { 0x1A2A, 13 },
    { 0x8, 4 },
    { 0xB, 5 },
    { 0x2B, 6 },
    { 0xB, 6 },
    { 0x69, 7 },
    { 0x35, 6 },
    { 0x8, 6 },
    { 0x7, 4 },
    { 0xC, 5 }
  },
  {  /* AC bias group 3, table 1 */
    { 0xA, 4 },
    { 0x3C, 6 },
    { 0x32, 7 },
    { 0x30, 7 },
    { 0xC5, 9 },
    { 0x621, 12 },
    { 0x620, 12 },
    { 0x1F, 5 },
    { 0x33, 7 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xE, 4 },
    { 0xD, 4 },
    { 0xC, 4 },
    { 0x4, 4 },
    { 0xD, 5 },
    { 0x26, 6 },
    { 0x27, 6 },
    { 0x14, 6 },
    { 0x63, 8 },
    { 0x189, 10 },
    { 0x623, 12 },
    { 0x622, 12 },
    { 0xB, 4 },
    { 0x12, 5 },
    { 0x3D, 6 },
    { 0x22, 6 },
    { 0x15, 6 },
    { 0xB, 5 },
    { 0x23, 6 },
    { 0x7, 4 },
    { 0x10, 5 }
  },
  {  /* AC bias group 3, table 2 */
    { 0xF, 4 },
    { 0xC, 5 },
    { 0x43, 7 },
    { 0x10, 6 },
    { 0x44, 8 },
    { 0x114, 10 },
    { 0x455, 12 },
    { 0x18, 5 },
    { 0x23, 7 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xE, 4 },
    { 0xD, 4 },
    { 0x9, 4 },
    { 0x19, 5 },
    { 0x9, 5 },
    { 0x17, 6 },
    { 0x16, 6 },
    { 0x42, 7 },
    { 0x8B, 9 },
    { 0x454, 12 },
    { 0x457, 12 },
    { 0x456, 12 },
    { 0xB, 4 },
    { 0x15, 5 },
    { 0xA, 5 },
    { 0x29, 6 },
    { 0x20, 6 },
    { 0xD, 5 },
    { 0x28, 6 },
    { 0x7, 4 },
    { 0x11, 5 }
  },
  {  /* AC bias group 3, table 3 */
    { 0x1, 3 },
    { 0x1A, 5 },
    { 0x29, 6 },
    { 0x2A, 6 },
    { 0xA0, 8 },
    { 0x285, 10 },
    { 0x1425, 13 },
    { 0x2, 5 },
    { 0x0, 7 },
    { 0x2, 3 },
    { 0x3, 3 },
    { 0xC, 4 },
    { 0xB, 4 },
    { 0x8, 4 },
    { 0x12, 5 },
    { 0x1, 6 },
    { 0x51, 7 },
    { 0x1, 7 },
    { 0x143, 9 },
    { 0x508, 11 },
    { 0x1424, 13 },
    { 0x1427, 13 },
    { 0x1426, 13 },
    { 0xF, 4 },
    { 0x1C, 5 },
    { 0x3, 5 },
    { 0x37, 6 },
    { 0x2B, 6 },
    { 0x13, 5 },
    { 0x36, 6 },
    { 0x1D, 5 },
    { 0x1, 5 }
  },
  {  /* AC bias group 3, table 4 */
    { 0x4, 3 },
    { 0x1F, 5 },
    { 0x3D, 6 },
    { 0x6, 5 },
    { 0x16, 7 },
    { 0x53, 9 },
    { 0x14A, 11 },
    { 0x34, 6 },
    { 0x2A, 8 },
    { 0x2, 3 },
    { 0x3, 3 },
    { 0xB, 4 },
    { 0xC, 4 },
    { 0x1C, 5 },
    { 0x37, 6 },
    { 0x17, 7 },
    { 0x2B, 8 },
    { 0x28, 8 },
    { 0xA4, 10 },
    { 0x52D, 13 },
    { 0x52C, 13 },
    { 0x52F, 13 },
    { 0x52E, 13 },
    { 0x0, 3 },
    { 0x1D, 5 },
    { 0x7, 5 },
    { 0x4, 5 },
    { 0x35, 6 },
    { 0x14, 5 },
    { 0x36, 6 },
    { 0x15, 5 },
    { 0x3C, 6 }
  },
  {  /* AC bias group 3, table 5 */
    { 0x4, 3 },
    { 0xA, 4 },
    { 0x7, 5 },
    { 0x1D, 5 },
    { 0x9, 6 },
    { 0x1F3, 9 },
    { 0x7C7, 11 },
    { 0x8, 6 },
    { 0x1F0, 9 },
    { 0x3, 3 },
    { 0x2, 3 },
    { 0xD, 4 },
    { 0xC, 4 },
    { 0x17, 5 },
    { 0x7D, 7 },
    { 0x1F2, 9 },
    { 0x7C6, 11 },
    { 0x7C5, 11 },
    { 0x1F12, 13 },
    { 0x3E27, 14 },
    { 0x3E26, 14 },
    { 0x1F11, 13 },
    { 0x1F10, 13 },
    { 0x0, 3 },
    { 0x1E, 5 },
    { 0x6, 5 },
    { 0x39, 6 },
    { 0x38, 6 },
    { 0x3F, 6 },
    { 0x2C, 6 },
    { 0x5, 5 },
    { 0x2D, 6 }
  },
  {  /* AC bias group 3, table 6 */
    { 0x2, 3 },
    { 0x7, 4 },
    { 0x18, 5 },
    { 0x3, 4 },
    { 0x5, 5 },
    { 0x35, 7 },
    { 0x4F, 9 },
    { 0x12, 7 },
    { 0x4E5, 13 },
    { 0x5, 3 },
    { 0x4, 3 },
    { 0xD, 4 },
    { 0xE, 4 },
    { 0x33, 6 },
    { 0x26, 8 },
    { 0x9D, 10 },
    { 0x4E4, 13 },
    { 0x4E7, 13 },
    { 0x4E6, 13 },
    { 0x4E1, 13 },
    { 0x4E0, 13 },
    { 0x4E3, 13 },
    { 0x4E2, 13 },
    { 0x0, 3 },
    { 0x1F, 5 },
    { 0xC, 5 },
    { 0x3D, 6 },
    { 0x3C, 6 },
    { 0x32, 6 },
    { 0x34, 7 },
    { 0x1B, 6 },
    { 0x8, 6 }
  },
  {  /* AC bias group 3, table 7 */
    { 0x0, 3 },
    { 0x4, 4 },
    { 0x1C, 5 },
    { 0xF, 4 },
    { 0x2, 4 },
    { 0x7, 5 },
    { 0x75, 7 },
    { 0xE8, 8 },
    { 0x1D2A, 13 },
    { 0x5, 3 },
    { 0x4, 3 },
    { 0xD, 4 },
    { 0xC, 4 },
    { 0x77, 7 },
    { 0xE96, 12 },
    { 0x3A57, 14 },
    { 0x3A56, 14 },
    { 0x3A5D, 14 },
    { 0x3A5C, 14 },
    { 0x3A5F, 14 },
    { 0x3A5E, 14 },
    { 0x1D29, 13 },
    { 0x1D28, 13 },
    { 0x3, 3 },
    { 0x6, 5 },
    { 0xA, 5 },
    { 0x2C, 7 },
    { 0x17, 6 },
    { 0x76, 7 },
    { 0x1D3, 9 },
    { 0x3A4, 10 },
    { 0x2D, 7 }
  },
  {  /* AC bias group 3, table 8 */
    { 0xA, 4 },
    { 0x24, 6 },
    { 0xBF, 8 },
    { 0x85, 8 },
    { 0x211, 10 },
    { 0x842, 12 },
    { 0x1087, 13 },
    { 0x18, 5 },
    { 0x20, 6 },
    { 0x1, 3 },
    { 0x2, 3 },
    { 0xE, 4 },
    { 0xD, 4 },
    { 0x7, 4 },
    { 0x13, 5 },
    { 0x25, 6 },
    { 0x5E, 7 },
    { 0x43, 7 },
    { 0xBE, 8 },
    { 0x109, 9 },
    { 0x1086, 13 },
    { 0x841, 12 },
    { 0x840, 12 },
    { 0xF, 4 },
    { 0x1, 4 },
    { 0x11, 5 },
    { 0x0, 5 },
    { 0x2E, 6 },
    { 0x19, 5 },
    { 0x1, 5 },
    { 0x6, 4 },
    { 0x16, 5 }
  },
  {  /* AC bias group 3, table 9 */
    { 0x2, 3 },
    { 0xF, 5 },
    { 0x6F, 7 },
    { 0x61, 7 },
    { 0x374, 10 },
    { 0x1BA8, 13 },
    { 0x3753, 14 },
    { 0x12, 5 },
    { 0x36, 6 },
    { 0x0, 3 },
    { 0x1, 3 },
    { 0xA, 4 },
    { 0xB, 4 },
    { 0x1A, 5 },
    { 0x31, 6 },
    { 0x60, 7 },
    { 0xDC, 8 },
    { 0x1BB, 9 },
    { 0x6EB, 11 },
    { 0x1BAB, 13 },
    { 0x3752, 14 },
    { 0x3755, 14 },
    { 0x3754, 14 },
    { 0xE, 4 },
    { 0x6, 4 },
    { 0x13, 5 },
    { 0xE, 5 },
    { 0x3E, 6 },
    { 0x8, 4 },
    { 0x1E, 5 },
    { 0x19, 5 },
    { 0x3F, 6 }
  },
  {  /* AC bias group 3, table 10 */
    { 0x3, 3 },
    { 0x1C, 5 },
    { 0x25, 6 },
    { 0x24, 6 },
    { 0x1DA, 9 },
    { 0x1DBD, 13 },
    { 0x3B7C, 14 },
    { 0x3C, 6 },
    { 0x3D, 6 },
    { 0x0, 3 },
    { 0x1, 3 },
    { 0xB, 4 },
    { 0xA, 4 },
    { 0xB, 5 },
    { 0x77, 7 },
    { 0xEC, 8 },
    { 0x3B6, 10 },
    { 0x76E, 11 },
    { 0x1DBF, 13 },
    { 0x76FB, 15 },
    { 0x76FA, 15 },
    { 0x3B79, 14 },
    { 0x3B78, 14 },
    { 0xD, 4 },
    { 0x1F, 5 },
    { 0x13, 5 },
    { 0xA, 5 },
    { 0x8, 5 },
    { 0xC, 4 },
    { 0x8, 4 },
    { 0x9, 5 },
    { 0x3A, 6 }
  },
  {  /* AC bias group 3, table 11 */
    { 0x5, 3 },
    { 0x3, 4 },
    { 0x4, 5 },
    { 0x10, 5 },
    { 0x8F, 8 },
    { 0x475, 11 },
    { 0x11D1, 13 },
    { 0x79, 7 },
    { 0x27, 6 },
    { 0x2, 3 },
    { 0x3, 3 },
    { 0x1, 4 },
    { 0x0, 4 },
    { 0x26, 6 },
    { 0x46, 7 },
    { 0x11C, 9 },
    { 0x477, 11 },
    { 0x8ED, 12 },
    { 0x11D0, 13 },
    { 0x11D3, 13 },
    { 0x11D2, 13 },
    { 0x11D9, 13 },
    { 0x11D8, 13 },
    { 0xD, 4 },
    { 0x1F, 5 },
    { 0x12, 5 },
    { 0x5, 5 },
    { 0x3D, 6 },
    { 0xC, 4 },
    { 0xE, 4 },
    { 0x22, 6 },
    { 0x78, 7 }
  },
  {  /* AC bias group 3, table 12 */
    { 0x5, 3 },
    { 0xC, 4 },
    { 0x1B, 5 },
    { 0x0, 4 },
    { 0x6, 6 },
    { 0x3E2, 10 },
    { 0x3E3D, 14 },
    { 0xF, 7 },
    { 0x34, 6 },
    { 0x3, 3 },
    { 0x2, 3 },
    { 0x1E, 5 },
    { 0x1D, 5 },
    { 0x7D, 7 },
    { 0x1F0, 9 },
    { 0x7C6, 11 },
    { 0x3E3C, 14 },
    { 0x3E3F, 14 },
    { 0x3E3E, 14 },
    { 0x3E39, 14 },
    { 0x3E38, 14 },
    { 0x3E3B, 14 },
    { 0x3E3A, 14 },
    { 0x8, 4 },
    { 0x1C, 5 },
    { 0x2, 5 },
    { 0x3F, 6 },
    { 0x35, 6 },
    { 0x9, 4 },
    { 0x1, 3 },
    { 0xE, 7 },
    { 0xF9, 8 }
  },
  {  /* AC bias group 3, table 13 */
    { 0x4, 3 },
    { 0xB, 4 },
    { 0x1, 4 },
    { 0xA, 4 },
    { 0x1E, 6 },
    { 0xE0, 9 },
    { 0xE1E, 13 },
    { 0x71, 8 },
    { 0x39, 7 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0xD, 5 },
    { 0xC, 5 },
    { 0x20, 7 },
    { 0x1C2, 10 },
    { 0x1C3F, 14 },
    { 0x1C3E, 14 },
    { 0xE19, 13 },
    { 0xE18, 13 },
    { 0xE1B, 13 },
    { 0xE1A, 13 },
    { 0xE1D, 13 },
    { 0xE1C, 13 },
    { 0x0, 4 },
    { 0x9, 5 },
    { 0x1D, 6 },
    { 0x1F, 6 },
    { 0x11, 6 },
    { 0x5, 4 },
    { 0x1, 3 },
    { 0x43, 8 },
    { 0x42, 8 }
  },
  {  /* AC bias group 3, table 14 */
    { 0x4, 3 },
    { 0xD, 4 },
    { 0x7, 4 },
    { 0x2, 3 },
    { 0x14, 5 },
    { 0x16C, 9 },
    { 0x16D1, 13 },
    { 0x2DF, 10 },
    { 0x16E, 9 },
    { 0x0, 2 },
    { 0x7, 3 },
    { 0x2C, 6 },
    { 0x2B, 6 },
    { 0x2DE, 10 },
    { 0x16D0, 13 },
    { 0x16D3, 13 },
    { 0x16D2, 13 },
    { 0x2DB5, 14 },
    { 0x2DB4, 14 },
    { 0x2DB7, 14 },
    { 0x2DB6, 14 },
    { 0x16D9, 13 },
    { 0x16D8, 13 },
    { 0xC, 5 },
    { 0x2A, 6 },
    { 0x5A, 7 },
    { 0x1B, 6 },
    { 0x1A, 6 },
    { 0x17, 5 },
    { 0xC, 4 },
    { 0x5B7, 11 },
    { 0x5B5, 11 }
  },
  {  /* AC bias group 3, table 15 */
    { 0x2, 2 },
    { 0xF, 4 },
    { 0x1C, 5 },
    { 0xC, 4 },
    { 0x3B, 6 },
    { 0x1AC, 9 },
    { 0x1AD8, 13 },
    { 0x35B3, 14 },
    { 0x35B2, 14 },
    { 0x1, 2 },
    { 0x0, 2 },
    { 0x69, 7 },
    { 0x68, 7 },
    { 0x35BD, 14 },
    { 0x35BC, 14 },
    { 0x35BF, 14 },
    { 0x35BE, 14 },
    { 0x35B9, 14 },
    { 0x35B8, 14 },
    { 0x35BB, 14 },
    { 0x35BA, 14 },
    { 0x35B5, 14 },
    { 0x35B4, 14 },
    { 0x1A9, 9 },
    { 0x1A8, 9 },
    { 0x35A, 10 },
    { 0xD7, 8 },
    { 0xD5, 8 },
    { 0x3A, 6 },
    { 0x1B, 5 },
    { 0x35B7, 14 },
    { 0x35B6, 14 }
  }
};

static const uint16_t ac_bias_3[16][32][2] = {
  {  /* AC bias group 4, table 0 */
    { 0x0, 3 },
    { 0x10, 5 },
    { 0x72, 7 },
    { 0x71, 7 },
    { 0x154, 9 },
    { 0xAAB, 12 },
    { 0xAA8, 12 },
    { 0x14, 5 },
    { 0x70, 7 },
    { 0x2, 3 },
    { 0x3, 3 },
    { 0xC, 4 },
    { 0xB, 4 },
    { 0x3, 4 },
    { 0x11, 5 },
    { 0x73, 7 },
    { 0x54, 7 },
    { 0xAB, 8 },
    { 0x2AB, 10 },
    { 0x1553, 13 },
    { 0x1552, 13 },
    { 0x1555, 13 },
    { 0x1554, 13 },
    { 0xD, 4 },
    { 0x1E, 5 },
    { 0x12, 5 },
    { 0x3E, 6 },
    { 0x2B, 6 },
    { 0x2, 4 },
    { 0x3F, 6 },
    { 0x1D, 5 },
    { 0x13, 5 }
  },
  {  /* AC bias group 4, table 1 */
    { 0x3, 3 },
    { 0x1F, 5 },
    { 0x29, 6 },
    { 0x3D, 6 },
    { 0xC, 7 },
    { 0x69, 10 },
    { 0x345, 13 },
    { 0x2, 5 },
    { 0x28, 6 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0xE, 4 },
    { 0xC, 4 },
    { 0x15, 5 },
    { 0x7, 6 },
    { 0x1B, 8 },
    { 0x6B, 10 },
    { 0x6A, 10 },
    { 0x344, 13 },
    { 0x347, 13 },
    { 0x346, 13 },
    { 0x1A1, 12 },
    { 0x1A0, 12 },
    { 0xB, 4 },
    { 0x1A, 5 },
    { 0x12, 5 },
    { 0x0, 5 },
    { 0x3C, 6 },
    { 0x8, 4 },
    { 0x1B, 5 },
    { 0x13, 5 },
    { 0x1, 5 }
  },
  {  /* AC bias group 4, table 2 */
    { 0x4, 3 },
    { 0x4, 4 },
    { 0x3F, 6 },
    { 0x14, 5 },
    { 0x56, 7 },
    { 0x15C, 9 },
    { 0x15D5, 13 },
    { 0x3C, 6 },
    { 0x2A, 6 },
    { 0x0, 3 },
    { 0x1, 3 },
    { 0xE, 4 },
    { 0xD, 4 },
    { 0xC, 5 },
    { 0xAF, 8 },
    { 0x2BB, 10 },
    { 0x15D4, 13 },
    { 0x15D7, 13 },
    { 0x15D6, 13 },
    { 0x15D1, 13 },
    { 0x15D0, 13 },
    { 0x15D3, 13 },
    { 0x15D2, 13 },
    { 0xB, 4 },
    { 0x19, 5 },
    { 0xD, 5 },
    { 0x3E, 6 },
    { 0x31, 6 },
    { 0x7, 4 },
    { 0x5, 4 },
    { 0x3D, 6 },
    { 0x30, 6 }
  },
  {  /* AC bias group 4, table 3 */
    { 0x5, 3 },
    { 0x8, 4 },
    { 0x1A, 5 },
    { 0x0, 4 },
    { 0x36, 6 },
    { 0x11, 8 },
    { 0x106, 12 },
    { 0xA, 7 },
    { 0x6E, 7 },
    { 0x2, 3 },
    { 0x3, 3 },
    { 0x3, 4 },
    { 0x2, 4 },
    { 0x6F, 7 },
    { 0x21, 9 },
    { 0x20F, 13 },
    { 0x20E, 13 },
    { 0x101, 12 },
    { 0x100, 12 },
    { 0x103, 12 },
    { 0x102, 12 },
    { 0x105, 12 },
    { 0x104, 12 },
    { 0xC, 4 },
    { 0x1E, 5 },
    { 0x3, 5 },
    { 0x3E, 6 },
    { 0x3F, 6 },
    { 0x9, 4 },
    { 0xE, 4 },
    { 0xB, 7 },
    { 0x9, 7 }
  },
  {  /* AC bias group 4, table 4 */
    { 0x2, 3 },
    { 0xE, 4 },
    { 0x1E, 5 },
    { 0xC, 4 },
    { 0x1F, 5 },
    { 0x6E, 7 },
    { 0xAD, 10 },
    { 0xAF, 10 },
    { 0x14, 7 },
    { 0x4, 3 },
    { 0x3, 3 },
    { 0x1A, 5 },
    { 0x17, 5 },
    { 0x2A, 8 },
    { 0x576, 13 },
    { 0xAEF, 14 },
    { 0xAEE, 14 },
    { 0x571, 13 },
    { 0x570, 13 },
    { 0x573, 13 },
    { 0x572, 13 },
    { 0x575, 13 },
    { 0x574, 13 },
    { 0x3, 4 },
    { 0x16, 5 },
    { 0x4, 5 },
    { 0x36, 6 },
    { 0xB, 6 },
    { 0xA, 4 },
    { 0x0, 3 },
    { 0x6F, 7 },
    { 0xAC, 10 }
  },
  {  /* AC bias group 4, table 5 */
    { 0x4, 3 },
    { 0x5, 4 },
    { 0x3, 3 },
    { 0x1, 3 },
    { 0x4, 4 },
    { 0x2F, 6 },
    { 0x526, 11 },
    { 0x1495, 13 },
    { 0xA6, 8 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0x2D, 6 },
    { 0x2C, 6 },
    { 0x1494, 13 },
    { 0x1497, 13 },
    { 0x1496, 13 },
    { 0x1491, 13 },
    { 0x1490, 13 },
    { 0x1493, 13 },
    { 0x1492, 13 },
    { 0x293D, 14 },
    { 0x293C, 14 },
    { 0x293F, 14 },
    { 0x0, 3 },
    { 0x28, 6 },
    { 0xA5, 8 },
    { 0x148, 9 },
    { 0xA7, 8 },
    { 0x2E, 6 },
    { 0x15, 5 },
    { 0xA4E, 12 },
    { 0x293E, 14 }
  },
  {  /* AC bias group 4, table 6 */
    { 0x4, 3 },
    { 0x5, 4 },
    { 0x3, 3 },
    { 0x1, 3 },
    { 0x4, 4 },
    { 0x2F, 6 },
    { 0x526, 11 },
    { 0x1495, 13 },
    { 0xA6, 8 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0x2D, 6 },
    { 0x2C, 6 },
    { 0x1494, 13 },
    { 0x1497, 13 },
    { 0x1496, 13 },
    { 0x1491, 13 },
    { 0x1490, 13 },
    { 0x1493, 13 },
    { 0x1492, 13 },
    { 0x293D, 14 },
    { 0x293C, 14 },
    { 0x293F, 14 },
    { 0x0, 3 },
    { 0x28, 6 },
    { 0xA5, 8 },
    { 0x148, 9 },
    { 0xA7, 8 },
    { 0x2E, 6 },
    { 0x15, 5 },
    { 0xA4E, 12 },
    { 0x293E, 14 }
  },
  {  /* AC bias group 4, table 7 */
    { 0x4, 3 },
    { 0x5, 4 },
    { 0x3, 3 },
    { 0x1, 3 },
    { 0x4, 4 },
    { 0x2F, 6 },
    { 0x526, 11 },
    { 0x1495, 13 },
    { 0xA6, 8 },
    { 0x7, 3 },
    { 0x6, 3 },
    { 0x2D, 6 },
    { 0x2C, 6 },
    { 0x1494, 13 },
    { 0x1497, 13 },
    { 0x1496, 13 },
    { 0x1491, 13 },
    { 0x1490, 13 },
    { 0x1493, 13 },
    { 0x1492, 13 },
    { 0x293D, 14 },
    { 0x293C, 14 },
    { 0x293F, 14 },
    { 0x0, 3 },
    { 0x28, 6 },
    { 0xA5, 8 },
    { 0x148, 9 },
    { 0xA7, 8 },
    { 0x2E, 6 },
    { 0x15, 5 },
    { 0xA4E, 12 },
    { 0x293E, 14 }
  },
  {  /* AC bias group 4, table 8 */
    { 0x3, 3 },
    { 0x11, 5 },
    { 0x20, 6 },
    { 0x74, 7 },
    { 0x10D, 9 },
    { 0x863, 12 },
    { 0x860, 12 },
    { 0xA, 5 },
    { 0x75, 7 },
    { 0x1, 3 },
    { 0x0, 3 },
    { 0xB, 4 },
    { 0xA, 4 },
    { 0x18, 5 },
    { 0x38, 6 },
    { 0x42, 7 },
    { 0x10F, 9 },
    { 0x10E, 9 },
    { 0x219, 10 },
    { 0x10C3, 13 },
    { 0x10C2, 13 },
    { 0x10C5, 13 },
    { 0x10C4, 13 },
    { 0xF, 4 },
    { 0x4, 4 },
    { 0x19, 5 },
    { 0xB, 5 },
    { 0x39, 6 },
    { 0x9, 4 },
    { 0x1B, 5 },
    { 0x1A, 5 },
    { 0x3B, 6 }
  },
  {  /* AC bias group 4, table 9 */
    { 0x5, 3 },
    { 0x1, 4 },
    { 0x3E, 6 },
    { 0x1, 5 },
    { 0xE2, 8 },
    { 0x1C6F, 13 },
    { 0x38D9, 14 },
    { 0x39, 6 },
    { 0x1F, 6 },
    { 0x2, 3 },
    { 0x1, 3 },
    { 0x9, 4 },
    { 0x8, 4 },
    { 0x0, 5 },
    { 0x70, 7 },
    { 0x1C7, 9 },
    { 0x38C, 10 },
    { 0x71A, 11 },
    { 0x38D8, 14 },
    { 0x38DB, 14 },
    { 0x38DA, 14 },
    { 0x38DD, 14 },
    { 0x38DC, 14 },
    { 0xD, 4 },
    { 0x1D, 5 },
    { 0xE, 5 },
    { 0x3F, 6 },
    { 0x3C, 6 },
    { 0xC, 4 },
    { 0x6, 4 },
    { 0x3D, 6 },
    { 0x1E, 6 }
  },
  {  /* AC bias group 4, table 10 */
    { 0x6, 3 },
    { 0xB, 4 },
    { 0x11, 5 },
    { 0x1E, 5 },
    { 0x74, 7 },
    { 0x3AA, 10 },
    { 0x1D5C, 13 },
    { 0x1, 6 },
    { 0x21, 6 },
    { 0x1, 3 },
    { 0x2, 3 },
    { 0x7, 4 },
    { 0x6, 4 },
    { 0x3E, 6 },
    { 0xEB, 8 },
    { 0x1D4, 9 },
    { 0xEAF, 12 },
    { 0x3ABB, 14 },
    { 0x3ABA, 14 },
    { 0x1D59, 13 },
    { 0x1D58, 13 },
    { 0x1D5B, 13 },
    { 0x1D5A, 13 },
    { 0xA, 4 },
    { 0x1C, 5 },
    { 0x1, 5 },
    { 0x3F, 6 },
    { 0x3B, 6 },
    { 0x1, 4 },
    { 0x9, 4 },
    { 0x20, 6 },
    { 0x0, 6 }
  },
  {  /* AC bias group 4, table 11 */
    { 0x4, 3 },
    { 0xA, 4 },
    { 0x17, 5 },
    { 0x4, 4 },
    { 0x16, 6 },
    { 0x16A, 9 },
    { 0x16B1, 13 },
    { 0x17, 7 },
    { 0x5B, 7 },
    { 0x6, 3 },
    { 0x7, 3 },
    { 0x1, 4 },
    { 0x0, 4 },
    { 0xA, 6 },
    { 0x2D7, 10 },
    { 0xB5A, 12 },
    { 0x16B0, 13 },
    { 0x16B3, 13 },
    { 0x16B2, 13 },
    { 0x2D6D, 14 },
    { 0x2D6C, 14 },
    { 0x2D6F, 14 },
    { 0x2D6E, 14 },
    { 0x6, 4 },
    { 0xA, 5 },
    { 0x4, 5 },
    { 0x2C, 6 },
    { 0x17, 6 },
    { 0x3, 4 },
    { 0x7, 4 },
    { 0x16, 7 },
    { 0xB4, 8 }
  },
  {  /* AC bias group 4, table 12 */
    { 0x5, 3 },
    { 0xD, 4 },
    { 0x5, 4 },
    { 0x9, 4 },
    { 0x33, 6 },
    { 0x193, 9 },
    { 0x192C, 13 },
    { 0x61, 8 },
    { 0x31, 7 },
    { 0x0, 2 },
    { 0x7, 3 },
    { 0x10, 5 },
    { 0x11, 5 },
    { 0xC8, 8 },
    { 0x192F, 13 },
    { 0x325B, 14 },
    { 0x325A, 14 },
    { 0x1929, 13 },
    { 0x1928, 13 },
    { 0x192B, 13 },
    { 0x192A, 13 },
    { 0x325D, 14 },
    { 0x325C, 14 },
    { 0x18, 5 },
    { 0x1A, 6 },
    { 0x1B, 6 },
    { 0x65, 7 },
    { 0x19, 6 },
    { 0x4, 4 },
    { 0x7, 4 },
    { 0x60, 8 },
    { 0x324, 10 }
  },
  {  /* AC bias group 4, table 13 */
    { 0x6, 3 },
    { 0x0, 3 },
    { 0x2, 4 },
    { 0xF, 4 },
    { 0x39, 6 },
    { 0x1D9, 9 },
    { 0x1D82, 13 },
    { 0x761, 11 },
    { 0x3BE, 10 },
    { 0x1, 2 },
    { 0x2, 2 },
    { 0xF, 6 },
    { 0xE, 6 },
    { 0x762, 11 },
    { 0x3B07, 14 },
    { 0x3B06, 14 },
    { 0x3B1D, 14 },
    { 0x3B1C, 14 },
    { 0x3B1F, 14 },
    { 0x3B1E, 14 },
    { 0x3B19, 14 },
    { 0x3B18, 14 },
    { 0x3B1B, 14 },
    { 0x38, 6 },
    { 0x1DE, 9 },
    { 0xED, 8 },
    { 0x3BF, 10 },
    { 0xEE, 8 },
    { 0x3A, 6 },
    { 0x6, 5 },
    { 0xEC0, 12 },
    { 0x3B1A, 14 }
  },
  {  /* AC bias group 4, table 14 */
    { 0x0, 2 },
    { 0x2, 3 },
    { 0xF, 5 },
    { 0x6, 4 },
    { 0x1C, 6 },
    { 0x1D0, 10 },
    { 0xE8C, 13 },
    { 0x1D1B, 14 },
    { 0x1D1A, 14 },
    { 0x3, 2 },
    { 0x2, 2 },
    { 0xEA, 9 },
    { 0xE9, 9 },
    { 0xE89, 13 },
    { 0xE88, 13 },
    { 0xE8B, 13 },
    { 0xE8A, 13 },
    { 0x1D65, 14 },
    { 0x1D64, 14 },
    { 0x1D67, 14 },
    { 0x1D66, 14 },
    { 0x1D61, 14 },
    { 0x1D60, 14 },
    { 0x3AD, 11 },
    { 0x1D63, 14 },
    { 0x1D62, 14 },
    { 0x1D1D, 14 },
    { 0x1D1C, 14 },
    { 0x3B, 7 },
    { 0x1D7, 10 },
    { 0x1D1F, 14 },
    { 0x1D1E, 14 }
  },
  {  /* AC bias group 4, table 15 */
    { 0x2, 2 },
    { 0xF, 4 },
    { 0x1C, 5 },
    { 0xC, 4 },
    { 0x3B, 6 },
    { 0x1AC, 9 },
    { 0x1AD8, 13 },
    { 0x35B3, 14 },
    { 0x35B2, 14 },
    { 0x1, 2 },
    { 0x0, 2 },
    { 0x69, 7 },
    { 0x68, 7 },
    { 0x35BD, 14 },
    { 0x35BC, 14 },
    { 0x35BF, 14 },
    { 0x35BE, 14 },
    { 0x35B9, 14 },
    { 0x35B8, 14 },
    { 0x35BB, 14 },
    { 0x35BA, 14 },
    { 0x35B5, 14 },
    { 0x35B4, 14 },
    { 0x1A9, 9 },
    { 0x1A8, 9 },
    { 0x35A, 10 },
    { 0xD7, 8 },
    { 0xD5, 8 },
    { 0x3A, 6 },
    { 0x1B, 5 },
    { 0x35B7, 14 },
    { 0x35B6, 14 }
  }
};

#endif /* AVCODEC_VP3DATA_H */
