/*
 * $Id: chip_fau_camera.c,v 1.16 2010-12-10 11:36:51 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-main.h"
#include "system.h"

#include "chip_fau_camera.h"

#define COMP_(x) chip_fau_camera_ ## x

struct cpssp {
	/* Config */
	char name[1024];

	/* Ports */
	struct sig_video *port_video_out;

	/* State */
	unsigned int state_power;

	struct {
		uint8_t r;
		uint8_t g;
		uint8_t b;
	} pixel[1080][1920];
	unsigned int width;
	unsigned int height;

	/* Processes */
	struct process process;
};

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;
	
	cpssp->state_power = val;
}

static void
COMP_(pixel_set)(
	void *_cpssp,
	unsigned int x,
	unsigned int y,
	uint8_t r,
	uint8_t g,
	uint8_t b
)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->pixel[y][x].r = r;
	cpssp->pixel[y][x].g = g;
	cpssp->pixel[y][x].b = b;
}

static void
COMP_(size_set)(void *_cpssp, unsigned int width, unsigned int height)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->width = width;
	cpssp->height = height;
}

static void
COMP_(sync)(void *_cpssp)
{
	/* Nothing to do... */
}

static void
COMP_(image_set)(void *_cpssp, const char *image)
{
	struct cpssp *cpssp = _cpssp;
	char name[1024];

	/* Hide old image (if any). */
	system_port_disconnect(cpssp->name, "opt_in");

	if (*image == '\0') {
		return;
	}

	/* Show new image. */
	name[0] = ':';
	strcpy(&name[1], image);
	system_port_connect(cpssp->name, "opt_in", name, "connect");
}

static void __attribute__((__noreturn__))
COMP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int y;
	unsigned int x;

again:	;
	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		if (! cpssp->state_power) {
			/* No power => no video signal. */
			sig_video_no_sync(cpssp->port_video_out, cpssp);

		} else if (cpssp->width == 0
			|| cpssp->height == 0) {
			for (y = 0; y < 1; y++) {
				for (x = 0; x < 1; x++) {
					sig_video_out(cpssp->port_video_out, cpssp,
							0, 0, 0);
				}
				sig_video_hor_retrace(cpssp->port_video_out, cpssp);
			}
			sig_video_vert_retrace(cpssp->port_video_out, cpssp);

		} else {
			/* Send video signal. */
			for (y = 0; y < cpssp->height; y++) {
				for (x = 0; x < cpssp->width; x++) {
					sig_video_out(cpssp->port_video_out, cpssp,
							cpssp->pixel[y][x].r,
							cpssp->pixel[y][x].g,
							cpssp->pixel[y][x].b);
				}
				sig_video_hor_retrace(cpssp->port_video_out, cpssp);
			}
			sig_video_vert_retrace(cpssp->port_video_out, cpssp);
		}

		cpssp->process.inst_cnt += 1980*1080;
	}

	sched_to_scheduler();

	goto again;
}

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_boolean *port_power,
	struct sig_opt_rgb *port_opt_in,
	struct sig_video *port_video_out,
	struct sig_string *port_image,
	struct sig_string *port_movie
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = COMP_(power_set),
	};
	static const struct sig_opt_rgb_funcs in_funcs = {
		.pixel_set = COMP_(pixel_set),
		.size_set = COMP_(size_set),
		.sync = COMP_(sync),
	};
	static const struct sig_string_funcs image_funcs = {
		.set = COMP_(image_set),
	};
	struct cpssp *cpssp;

	system_name_push(name);

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	assert(strlen(system_path()) < sizeof(cpssp->name));
	strcpy(cpssp->name, system_path());

	cpssp->process.inst_hz = 1920*1080*10;

	cpssp->width = 0;
	cpssp->height = 0;

	/* Call */
	/* Out */
	cpssp->port_video_out = port_video_out;

	/* In */
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);
	sig_opt_rgb_connect(port_opt_in, cpssp, &in_funcs);
	sig_string_connect(port_image, cpssp, &image_funcs);

	sched_process_init(&cpssp->process, COMP_(step), cpssp);

	system_name_pop();

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */

	free(cpssp);
}
