/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"

#if defined(ISHELL) && defined(IEXTENSION)

#include "ishellfactory.h"

#include "icontrolmodule.h"
#include "iimagecomposer.h"
#include "irendertool.h"
#include "iviewmodule.h"
#include "ishell.h"


#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>

#include <string.h>

//
//  Include shell-specific headers 
//
#if ISHELL_INCLUDED(ISHELL_GG)
#include "iggcommoneventobservers.h"
#include "iggcontrolscript.h"
#include "iggrenderwindow.h"
#include "iggrenderwindowinteractor.h"
#include "iggshell.h"
#endif

#if ISHELL_INCLUDED(ISHELL_CL)
#include "iclcommoneventobservers.h"
#include "iclcontrolscript.h"
#include "iclshell.h"
#endif


const iString iShellFactory::qt = "qt";
const iString iShellFactory::cl = "cl";
const iString iShellFactory::fx = "fx";


void iShellFactory::GetSupportedShells(iString &list, iString &help)
{
	list.Clear();
	help.Clear();

#if ISHELL_INCLUDED(ISHELL_QT)
	list += "qt/";
	help += "Shell based on Qt Grapical User Interface Toolkit/";
#endif

#if ISHELL_INCLUDED(ISHELL_CL)
	list += "cl/";
	help += "Command-line based shell/";
#endif

#if ISHELL_INCLUDED(ISHELL_FX)
	list += "fx/";
	help += "Shell based on FOX Grapical User Interface Toolkit/";
#endif
}


iShell* iShellFactory::CreateShell(const iString &type, int argc, char **argv)
{ 
	if(type == qt)
	{
#if ISHELL_INCLUDED(ISHELL_QT)
		return new iggShell(qt,argc,argv);
#else
		return 0;
#endif
	}

	if(type == cl)
	{
#if ISHELL_INCLUDED(ISHELL_CL)
		return new iclShell(argc,argv);
#else
		return 0;
#endif
	}

	if(type == fx)
	{
#if ISHELL_INCLUDED(ISHELL_FX)
		return new iggShell(fx,argc,argv);
#else
		return 0;
#endif
	}

	return 0;
}


iEventObserver* iShellFactory::CreateEventObserver(const iString &type, iControlModule *cm)
{
	if(cm==0 || cm->GetShell()==0) return 0;
	iShell *shell = cm->GetShell();

	if(type == "ParallelUpdate") 
	{
		if(shell->Type() == qt)
		{
#if ISHELL_INCLUDED(ISHELL_QT)
			return new iggParallelUpdateEventObserver(cm);
#else
			return 0;
#endif
		}
		if(shell->Type() == cl)
		{
#if ISHELL_INCLUDED(ISHELL_CL)
			return new iclParallelUpdateEventObserver(cm);
#else
			return 0;
#endif
		}
		if(shell->Type() == fx)
		{
#if ISHELL_INCLUDED(ISHELL_FX)
			return new iggParallelUpdateEventObserver(cm);
#else
			return 0;
#endif
		}
	}

	return 0;
}


iEventObserver* iShellFactory::CreateEventObserver(const iString &type, iViewModule *vm)
{
	if(vm==0 || vm->GetControlModule()==0 || vm->GetControlModule()->GetShell()==0) return 0;
	iShell *shell = vm->GetControlModule()->GetShell();

	if(type == "Progress") 
	{
		if(shell->Type() == qt)
		{
#if ISHELL_INCLUDED(ISHELL_QT)
			return new iggProgressEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == cl)
		{
#if ISHELL_INCLUDED(ISHELL_CL)
			return new iclProgressEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == fx)
		{
#if ISHELL_INCLUDED(ISHELL_FX)
			return new iggProgressEventObserver(vm);
#else
			return 0;
#endif
		}
	}

	if(type == "AbortRender") 
	{
		if(shell->Type() == qt)
		{
#if ISHELL_INCLUDED(ISHELL_QT)
			return new iggAbortRenderEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == cl)
		{
#if ISHELL_INCLUDED(ISHELL_CL)
			return new iclAbortRenderEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == fx)
		{
#if ISHELL_INCLUDED(ISHELL_FX)
			return new iggAbortRenderEventObserver(vm);
#else
			return 0;
#endif
		}
	}

	if(type == "Pick") 
	{
		if(shell->Type() == qt)
		{
#if ISHELL_INCLUDED(ISHELL_QT)
			return new iggPickEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == cl)
		{
#if ISHELL_INCLUDED(ISHELL_CL)
			return new iclPickEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == fx)
		{
#if ISHELL_INCLUDED(ISHELL_FX)
			return new iggPickEventObserver(vm);
#else
			return 0;
#endif
		}
	}

	if(type == "Animator") 
	{
		if(shell->Type() == qt)
		{
#if ISHELL_INCLUDED(ISHELL_QT)
			return new iggAnimatorEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == cl)
		{
#if ISHELL_INCLUDED(ISHELL_CL)
			return new iclAnimatorEventObserver(vm);
#else
			return 0;
#endif
		}
		if(shell->Type() == fx)
		{
#if ISHELL_INCLUDED(ISHELL_FX)
			return new iggAnimatorEventObserver(vm);
#else
			return 0;
#endif
		}
	}

	return 0;
}


iControlScript* iShellFactory::CreateControlScript(iControlModule *cm, iScript *parent)
{
	if(cm==0 || cm->GetShell()==0) return 0;
	iShell *shell = cm->GetShell();

	if(shell->Type() == qt)
	{
#if ISHELL_INCLUDED(ISHELL_QT)
		return new iggControlScript(cm,parent);
#else
		return 0;
#endif
	}

	if(shell->Type() == cl)
	{
#if ISHELL_INCLUDED(ISHELL_CL)
		return new iclControlScript(cm,parent);
#else
		return 0;
#endif
	}

	if(shell->Type() == fx)
	{
#if ISHELL_INCLUDED(ISHELL_FX)
		return new iggControlScript(cm,parent);
#else
		return 0;
#endif
	}

	return 0;
}


vtkRenderWindow* iShellFactory::CreateRenderWindow(iRenderTool *rv)
{
	if(rv==0 || rv->GetViewModule()==0 || rv->GetViewModule()->GetControlModule()==0 || rv->GetViewModule()->GetControlModule()->GetShell()==0) return 0;
	iShell *shell = rv->GetViewModule()->GetControlModule()->GetShell();

	if(shell->Type() == qt)
	{
#if ISHELL_INCLUDED(ISHELL_QT)
		return new iggRenderWindow(rv->GetViewModule());
#else
		return 0;
#endif
	}

	if(shell->Type() == cl)
	{
#if ISHELL_INCLUDED(ISHELL_CL)
		return vtkRenderWindow::New();
#else
		return 0;
#endif
	}

	if(shell->Type() == fx)
	{
#if ISHELL_INCLUDED(ISHELL_FX)
		return vtkRenderWindow::New();
#else
		return 0;
#endif
	}

	return 0;
}


vtkRenderWindowInteractor* iShellFactory::CreateRenderWindowInteractor(iRenderTool *rv)
{
	if(rv==0 || rv->GetViewModule()==0 || rv->GetViewModule()->GetControlModule()==0 || rv->GetViewModule()->GetControlModule()->GetShell()==0) return 0;
	iShell *shell = rv->GetViewModule()->GetControlModule()->GetShell();

	if(shell->Type() == qt)
	{
#if ISHELL_INCLUDED(ISHELL_QT)
		return new iggRenderWindowInteractor(shell);
#else
		return 0;
#endif
	}

	if(shell->Type() == cl)
	{
#if ISHELL_INCLUDED(ISHELL_CL)
		return vtkRenderWindowInteractor::New();
#else
		return 0;
#endif
	}

	if(shell->Type() == fx)
	{
#if ISHELL_INCLUDED(ISHELL_FX)
		return vtkRenderWindowInteractor::New();
#else
		return 0;
#endif
	}

	return 0;
}

#endif
