/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icamera.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "irendertool.h"
#include "iviewmodule.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkRenderer.h>

IOBJECT_DEFINE_TYPE(iCamera,Camera,c,iObjectType::_Helper);


IOBJECT_DEFINE_KEY(iCamera,Azimuth,a,Float,1);
IOBJECT_DEFINE_KEY(iCamera,ClippingRange,cr,Double,2);
IOBJECT_DEFINE_KEY(iCamera,ClippingRangeAuto,cra,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,Elevation,e,Float,1);
IOBJECT_DEFINE_KEY(iCamera,EyeAngle,ea,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Pitch,p,Float,1);
IOBJECT_DEFINE_KEY(iCamera,ParallelProjection,pp,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,ParallelScale,ps,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Roll,r,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Reset,rs,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,ViewAngle,va,Float,1);
IOBJECT_DEFINE_KEY(iCamera,ViewAngleVertical,vav,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,ViewUp,u,Double,3);
IOBJECT_DEFINE_KEY(iCamera,Yaw,y,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Zoom,z,Float,1);

IOBJECT_DEFINE_POSITION_KEY(iCamera,FocalPoint,f);
IOBJECT_DEFINE_POSITION_KEY(iCamera,Position,x);

IOBJECT_DEFINE_KEY(iCamera,SyncProjections,-sp,Bool,1);


//
//  Helper classes
//
namespace iCamera_Private
{
	class CameraObserver : public iEventObserver
	{

	public:

		vtkTypeMacro(CameraObserver,iEventObserver);
		static CameraObserver* New(iCamera *c = 0)
		{
			IERROR_ASSERT(c);
			return new CameraObserver(c);
		}

	protected:

		CameraObserver(iCamera *c) : iEventObserver(c->GetViewModule()->GetControlModule()->GetShell())
		{
			mCamera = c;
		}

		virtual void ExecuteBody(vtkObject *, unsigned long event, void *)
		{
			switch(event)
			{
			case ModifiedEvent:
				{
					mCamera->Update();
					break;
				}
			}
		}

	private:

		iCamera *mCamera;
	};
};


using namespace iCamera_Private;


//
//  Main class
//
iCamera* iCamera::New(iRenderTool *rt)
{
	IERROR_ASSERT(rt);
	return new iCamera(rt);
}


iCamera::iCamera(iRenderTool *rt) : iObject("Camera"), mViewModule(rt->GetViewModule()), mPosition(rt->GetViewModule()), mFocalPoint(rt->GetViewModule())
{
	mRenderTool = rt;
	mDevice = vtkCamera::New(); IERROR_ASSERT(mDevice);
	mObserver = CameraObserver::New(this); IERROR_ASSERT(mObserver);

	mSyncProjections = false;

	mDevice->AddObserver(vtkCommand::ModifiedEvent,mObserver);

	mDevice->ParallelProjectionOn();

	mDevice->Modified();
}


iCamera::~iCamera()
{
	mObserver->Delete();
	mDevice->Delete();
}


void iCamera::Update()
{
	//
	//  Update position and focal point
	//
	mPosition = mDevice->GetPosition();
	mFocalPoint = mDevice->GetFocalPoint();
	this->ClearCache();
}


void iCamera::Reset()
{
	mRenderTool->ResetCamera();
	this->ClearCache();
}


void iCamera::OrthogonalizeView()
{
	static const double orts[6][3] = 
	{
		{ -1.0,  0.0,  0.0 },
		{  1.0,  0.0,  0.0 },
		{  0.0, -1.0,  0.0 },
		{  0.0,  1.0,  0.0 },
		{  0.0,  0.0, -1.0 },
		{  0.0,  0.0,  1.0 } 
	};

	//
	//  Orthogonalize the camera position.
	//
	int i, j, imax;
	double d, *v, dmax, pos[3];

	imax = 5;
	dmax = 0.0;
	v = mDevice->GetDirectionOfProjection();
	for(i=0; i<6; i++)
	{
		d = vtkMath::Dot(orts[i],v);
		if(d > dmax)
		{
			dmax = d;
			imax = i;
		}
	}

	d = mDevice->GetDistance();
	v = mDevice->GetFocalPoint();
	for(j=0; j<3; j++) pos[j] = v[j] - d*orts[imax][j];
	mDevice->SetPosition(pos);

	//
	//  Orthogonalize the view up.
	//
	imax = 3;
	dmax = 0.0;
	v = mDevice->GetViewUp();
	for(i=0; i<6; i++)
	{
		d = vtkMath::Dot(orts[i],v);
		if(d > dmax)
		{
			dmax = d;
			imax = i;
		}
	}
	mDevice->SetViewUp(orts[imax]);
}


void iCamera::ShiftTo(const iPosition& pos)
{
	int j;
	double cp[3], fp[3];
	mDevice->GetPosition(cp);
	mDevice->GetFocalPoint(fp);
	for(j=0; j<3; j++)
	{
		cp[j] += (pos[j]-fp[j]);
	}
	mDevice->SetPosition(cp);
	mDevice->SetFocalPoint(pos);
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iCamera::PackStateBody(iString &s) const
{
	double d, dv[2];

	//
	//  Pack projection first as changing projections may reset other properties
	//
	this->PackValue(s,KeyParallelProjection(),mDevice->GetParallelProjection()!=0);

	this->PackValuePosition(s,KeyPosition(),mPosition);
	this->PackValuePosition(s,KeyFocalPoint(),mFocalPoint);
	this->PackValue(s,KeyViewUp(),mDevice->GetViewUp(),3);

	this->PackValue(s,KeyParallelScale(),float(mDevice->GetParallelScale()));
	this->PackValue(s,KeyViewAngle(),float(mDevice->GetViewAngle()));
	this->PackValue(s,KeyViewAngleVertical(),mDevice->GetUseHorizontalViewAngle()==0);
	this->PackValue(s,KeyEyeAngle(),float(mDevice->GetEyeAngle()));
	this->PackValue(s,KeyClippingRangeAuto(),mRenderTool->GetAdjustCameraClippingRangeAutomatically());
	this->PackValue(s,KeySyncProjections(),mSyncProjections);

	mDevice->GetClippingRange(dv);
	d = mDevice->GetDistance();
	dv[0] /= d;
	dv[1] /= d;
	this->PackValue(s,KeyClippingRange(),dv,2);
}


void iCamera::UnPackStateBody(const iString &s)
{
	bool b; float f; double d, dv[3];

	iObject::ReportMissingKeys(false); //optional key
	if(this->UnPackValue(s,KeySyncProjections(),b)) mSyncProjections = b;
	iObject::ReportMissingKeys(true);

	//
	//  UnPack projection first as changing projections may reset other properties
	//
	if(this->UnPackValue(s,KeyParallelProjection(),b))
	{
		//
		//  We shouldn't reset unless we actually change projections
		//
		if(b != (mDevice->GetParallelProjection()!=0))
		{
			if(mSyncProjections) this->SyncProjections(); else mRenderTool->ResetCamera();
		}
		mDevice->SetParallelProjection(b?1:0);
	}

	//
	//  Camera orientation
	//
	if(this->UnPackValuePosition(s,KeyPosition(),mPosition)) this->SetPosition(mPosition);
	if(this->UnPackValuePosition(s,KeyFocalPoint(),mFocalPoint)) this->SetFocalPoint(mFocalPoint);
	if(this->UnPackValue(s,KeyViewUp(),dv,3)) mDevice->SetViewUp(dv);

	if(this->UnPackValue(s,KeyParallelScale(),f)) mDevice->SetParallelScale(f);
	if(this->UnPackValue(s,KeyViewAngle(),f)) mDevice->SetViewAngle(f);
	if(this->UnPackValue(s,KeyViewAngleVertical(),b)) mDevice->SetUseHorizontalViewAngle(b?0:1);
	if(this->UnPackValue(s,KeyEyeAngle(),f)) mDevice->SetEyeAngle(f);

	if(this->UnPackValue(s,KeyClippingRangeAuto(),b))
	{
		mRenderTool->SetAdjustCameraClippingRangeAutomatically(b);
		this->ClearCache();
	}

	mRenderTool->GetCameraClippingRange(dv);
	d = mDevice->GetDistance();
	dv[0] /= d;
	dv[1] /= d;
	if(this->UnPackValue(s,KeyClippingRange(),dv,2))
	{
		dv[0] *= d;
		dv[1] *= d;
		mRenderTool->SetCameraClippingRange(dv);
		this->ClearCache();
	}

	//
	//  "Action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeyReset(),b) && b) this->Reset();

	if(this->UnPackValue(s,KeyAzimuth(),f)) mDevice->Azimuth(f);
	if(this->UnPackValue(s,KeyElevation(),f)) mDevice->Elevation(f);
	if(this->UnPackValue(s,KeyYaw(),f)) mDevice->Yaw(f);
	if(this->UnPackValue(s,KeyPitch(),f)) mDevice->Pitch(f);
	if(this->UnPackValue(s,KeyRoll(),f)) mDevice->Roll(f);
	if(this->UnPackValue(s,KeyZoom(),f) && f>1.0e-30)
	{
		if(mDevice->GetParallelProjection() != 0)
		{
			mDevice->SetParallelScale(mDevice->GetParallelScale()/f);
		}
		else
		{
			mDevice->Dolly(f);
		}
	}

	iObject::ReportMissingKeys(true);
}


void iCamera::SetPosition(const iPosition &p)
{
	mDevice->SetPosition(p);	
}


void iCamera::SetFocalPoint(const iPosition &p)
{
	mDevice->SetFocalPoint(p);
}


void iCamera::SyncProjections()
{
	static const double minDistance = 0.0002; // idiotic VTK limit

	double r = 1.0e-35 + tan(0.5*iMath::Deg2Rad(mDevice->GetViewAngle()));

	if(mDevice->GetParallelProjection() != 0)
	{
		double d = mDevice->GetParallelScale()/r;
		if(d > minDistance) mDevice->Dolly(mDevice->GetDistance()/d);
	}
	else
	{
		mDevice->SetParallelScale(mDevice->GetDistance()*r);
	}
	mRenderTool->ResetCameraClippingRange();
}

