/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 363 $
 * $Date: 2012-01-22 18:45:43 +0800 (Sun, 22 Jan 2012) $
 */
#include "PostProcessPane.hpp"

#include "../../gui/Help.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

using namespace indii;

PostProcessPane::PostProcessPane(wxWindow* parent) :
    Pane(parent), model(NULL) {
  static const int PADDING = 4;

  wxGridBagSizer* sizer = new wxGridBagSizer();

  slideSatDecay = new wxSlider(this, ID_SLIDE_SAT_DECAY, 16, 0, 255,
      wxDefaultPosition, wxSize(100,-1));
  slideSatSoft = new wxSlider(this, ID_SLIDE_SAT_SOFT, 32, 0, 255,
      wxDefaultPosition, wxSize(100,-1));
  slideHard = new wxSlider(this, ID_SLIDE_HARD, 255, 0, 255,
      wxDefaultPosition, wxSize(100,-1));

  wxStaticText *labelSatDecay, *labelSatSoft, *labelHard;
  labelSatDecay = new wxStaticText(this, -1, _("Decay:"));
  labelSatSoft = new wxStaticText(this, -1, _("Edge:"));
  labelHard = new wxStaticText(this, -1, _("Hardness:"));

  sizer->Add(labelSatDecay, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelSatSoft, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelHard, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_RIGHT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT|wxBOTTOM, PADDING);

  sizer->Add(slideSatDecay, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideSatSoft, wxGBPosition(1,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideHard, wxGBPosition(2,1), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxTOP|wxRIGHT|wxBOTTOM, PADDING);

  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(1);
  SetSizerAndFit(sizer);
  setModel(NULL);
}

void PostProcessPane::setModel(ClusterModel* model) {
  this->model = model;
  ignore();

  Enable(model != NULL);
  
  /* must explicitly disable all controls, not just panel, on Mac */
  slideSatDecay->Enable(model != NULL);
  slideSatSoft->Enable(model != NULL);
  slideHard->Enable(model != NULL);
  
  if (model != NULL) {
    watch(model);
    notifySaturationDecayChange();
    notifySaturationSoftnessChange();
    notifyIsHardChange();
  }
}

void PostProcessPane::notifySaturationDecayChange() {
  /* pre-condition */
  assert (model != NULL);

  int val = ColourSpace::uround(model->getSaturationDecay()*255.0f);
  if (slideSatDecay->GetValue() != val) {
    slideSatDecay->SetValue(val);
  }
}

void PostProcessPane::notifySaturationSoftnessChange() {
  /* pre-condition */
  assert (model != NULL);

  int val = ColourSpace::uround(model->getSaturationSoftness()*255.0f);
  if (slideSatSoft->GetValue() != val) {
    slideSatSoft->SetValue(val);
  }
}

void PostProcessPane::notifyIsHardChange() {
  /* pre-condition */
  assert (model != NULL);

  int val = ColourSpace::uround(model->getHard()*255.0f);
  if (slideHard->GetValue() != val) {
    slideHard->SetValue(val);
  }
}

void PostProcessPane::OnSlideSatDecay(wxScrollEvent& evt) {
	if (model != NULL) {
	  model->setSaturationDecay(slideSatDecay->GetValue()/255.0f);
  }
}

void PostProcessPane::OnSlideSatSoft(wxScrollEvent& evt) {
	if (model != NULL) {
	  model->setSaturationSoftness(slideSatSoft->GetValue()/255.0f);
  }
}

void PostProcessPane::OnSlideHard(wxScrollEvent& evt) {
	if (model != NULL) {
	  model->setHard(slideHard->GetValue()/255.0f);
  }
}

void PostProcessPane::OnHelp(wxHelpEvent& evt) {
  Help::show(540);
}

BEGIN_EVENT_TABLE(PostProcessPane, Pane)

EVT_COMMAND_SCROLL(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL(ID_SLIDE_HARD, PostProcessPane::OnSlideHard)

EVT_HELP(wxID_ANY, PostProcessPane::OnHelp)

END_EVENT_TABLE()
