/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 432 $
 * $Date: 2012-03-08 22:17:25 +0800 (Thu, 08 Mar 2012) $
 */
#include "ClusterModel.hpp"

#include "ClusterModelObserver.hpp"

//#include "wx/app.h"
//#include "wx/busyinfo.h"

#include <cassert>
#include <ctime>
#include <limits>

namespace ublas = boost::numeric::ublas;

using namespace indii;
using namespace std;

ClusterModel::ClusterModel(ImageResource* res) : minClusterer(NULL),
    locked(false) {
  this->res = res;
  this->saturationThreshold = 16;
  this->maxPixels = 10000;
  this->reps = 8;
  this->k = 4;
  this->hard = 1.0f;
  this->saturationDecay = 16.0f / 255.0f;
  this->centroidDecay = 0.0f;
  this->saturationSoftness = 32.0f / 255.0f;
  this->centroidSoftness = 0.0f;
  this->colours.resize(k, *wxWHITE);
  this->hues.resize(k, 0.0f);
  this->sats.resize(k, -1.0f);
  this->lights.resize(k, 0.0f);
  this->minClusterer = new KMeansClusterer<>(k, time(NULL));
}

ClusterModel::~ClusterModel() {
  delete minClusterer;
}

void ClusterModel::setDefaults() {
  lock();
  setHard(1.0f);
  setSaturationThreshold(16);
  setMaxPixels(10000);
  setNumRepetitions(8);
  setNumClusters(4);
  setSaturationDecay(16.0f / 255.0f);
  setCentroidDecay(0.0f);
  setSaturationSoftness(32.0f / 255.0f);
  setCentroidSoftness(0.0f);
  setChannelMix(76.0f, 150.0f, 29.0f);
  setForDialog();
  unlock();
  prepare();
  cluster();

  notifyAll();
  notifyNumClustersChange();
}

void ClusterModel::setForDialog() {
  fill(hues.begin(), hues.end(), 0.0f);
  fill(sats.begin(), sats.end(), -1.0f);
  fill(lights.begin(), lights.end(), 0.0f);  
}

void ClusterModel::setNumClusters(const unsigned k) {
  if (this->k != k) {
    this->k = k;
    colours.resize(k, *wxWHITE);
    hues.resize(k);
    sats.resize(k);
    lights.resize(k);

    fill(sats.begin(), sats.end(), -1.0f);
    fill(hues.begin(), hues.end(), 0.0f);
    fill(lights.begin(), lights.end(), 0.0f);

    delete minClusterer;
    minClusterer = new KMeansClusterer<>(k, time(NULL));
    cluster();
    notifyAll();
    notifyNumClustersChange();
  }
}

void ClusterModel::setHard(const float hard) {
  /* pre-condition */
  assert (hard >= 0.0f && hard <= 1.0f);
  
  if (this->hard != hard) {
    this->hard = hard;
    notifyAll();
    notifyIsHardChange();
  }
}

void ClusterModel::setNumRepetitions(const unsigned reps) {
  if (this->reps != reps) {
    this->reps = reps;
    notifyAll();
    notifyNumRepetitionsChange();
  }
}

void ClusterModel::setSaturationThreshold(const unsigned char x) {
  if (this->saturationThreshold != x) {
    this->saturationThreshold = x;
    prepare();
    notifyAll();
    notifySaturationThresholdChange();
  }
}

void ClusterModel::setMaxPixels(const unsigned n) {
  if (this->maxPixels != n) {
    this->maxPixels = n;
    prepare();
    notifyAll();
    notifyMaxPixelsChange();
  }
}

void ClusterModel::setSaturationDecay(const float x) {
  /* pre-condition */
  assert (x >= 0.0f && x <= 1.0f);

  if (this->saturationDecay != x) {
    this->saturationDecay = x;
    notifyAll();
    notifySaturationDecayChange();
  }
}

void ClusterModel::setCentroidDecay(const float x) {
  /* pre-condition */
  assert (x >= 0.0f && x <= 1.0f);

  if (this->centroidDecay != x) {
    this->centroidDecay = x;
    notifyAll();
    notifyCentroidDecayChange();
  }
}

void ClusterModel::setSaturationSoftness(const float x) {
  /* pre-condition */
  assert (x >= 0.0 && x <= 1.0f);

  if (this->saturationSoftness != x) {
    this->saturationSoftness = x;
    notifyAll();
    notifySaturationSoftnessChange();
  }
}

void ClusterModel::setCentroidSoftness(const float x) {
  /* pre-condition */
  assert (x >= 0.0 && x <= 1.0f);

  if (this->centroidSoftness != x) {
    this->centroidSoftness = x;
    notifyAll();
    notifyCentroidSoftnessChange();
  }
}

void ClusterModel::setColour(const unsigned i, const wxColour& col) {
  /* pre-condition */
  assert (i < k);

  ClusterVector<>::type x(ClusterVector<>::N);
  x(0) = (float)col.Red();
  x(1) = (float)col.Green();
  x(2) = (float)col.Blue();
  PearsonDistance<>::prepare(x);

  colours[i] = col;
  minClusterer->setCentroid(i, x);
}

void ClusterModel::setHue(const unsigned i, const float x) {
  /* pre-condition */
  assert (i < k);
  assert (x >= -0.5f && x <= 0.5f);

  if (hues[i] != x) {
    hues[i] = x;
    notifyAll();
    notifyHueChange(i);
  }
}

void ClusterModel::setSat(const unsigned i, const float x) {
  /* pre-condition */
  assert (i < k);
  assert (x >= -1.0f && x <= 1.0f);

  if (sats[i] != x) {
    sats[i] = x;
    notifyAll();
    notifySatChange(i);
  }
}

void ClusterModel::setLight(const unsigned i, const float x) {
  /* pre-condition */
  assert (i < k);
  assert (x >= -255.0f && x <= 255.0f);

  if (lights[i] != x) {
    lights[i] = x;

    /* notify observers */
    notifyAll();
    notifyLightChange(i);
  }
}

void ClusterModel::show(const unsigned i, const bool on) {
  /* pre-condition */
  assert (i < k);

  if (isShown(i) != on) {
    if (on) {
      sats[i] = 0.0f;
    } else {
      sats[i] = -1.0f;
    }
    
    notifyAll();
    notifySatChange(i);
  }
}

void ClusterModel::showAll(const bool on) {
  unsigned i;
  for (i = 0; i < k; i++) {
    show(i, on);
  }
}

void ClusterModel::calcFg(const wxRect& rect,
    const unsigned width, const unsigned height, wxImage& o) {
  /* pre-conditions */
  assert (o.GetWidth() >= rect.width);
  assert (o.GetHeight() >= rect.height);

  wxImage img(res->calc(rect, width, height, true));

  #pragma omp parallel default(shared)
  {
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
    ClusterVector<>::type pixel(ClusterVector<>::N);
    std::vector<float> ds(k);

    int x, y;
    int cluster;
    unsigned char r, g, b;
    float decay, light;

    #pragma omp for schedule(dynamic)
    for (y = 0; y < rect.height; y++) {
      for (x = 0; x < rect.width; x++) {
        r = img.GetRed(x, y);
        g = img.GetGreen(x, y);
        b = img.GetBlue(x, y);

        rgb(0) = r;
        rgb(1) = g;
        rgb(2) = b;
        cs.rgb2hsl(rgb, hsl);

        pixel(0) = r;
        pixel(1) = g;
        pixel(2) = b;

        decay = threshold(hsl(1), saturationDecay, saturationSoftness);
        
        /* cluster adjustments */
        PearsonDistance<>::prepare(pixel);
        if (isHard()) {
          cluster = minClusterer->assign(pixel);

          hsl(0) = fmod(6.0f + hsl(0) + 6.0f*hues[cluster], 6.0f);
          hsl(1) = cs.bound(0.0f, decay*(sats[cluster] + hsl(1)), 1.0f);

          light = decay*lights[cluster];
          if (light >= 0.0) {
            hsl(2) *= 255.0f/(255.0f - light);
          } else {
            hsl(2) *= (255.0f + light)/255.0f;
          }
          hsl(2) = cs.bound(0.0f, hsl(2), 255.0f);
        } else {
          minClusterer->distances(pixel, ds);
          
          unsigned i;
          float h = 0.0f, s = 0.0f, l = 0.0f;
          float D = 0.0f;
          for (i = 0; i < ds.size(); ++i) {
            ds[i] = std::exp(hard*25.0f*(1.0f - ds[i]));
            D += ds[i];
          }
          for (i = 0; i < k; ++i) {
            if (D >= 0.0) {
	            ds[i] /= D;
            }
            h += ds[i]*hues[i];
            s += ds[i]*sats[i];
            l += ds[i]*lights[i];
          }
          
          hsl(0) = fmod(6.0f + hsl(0) + 6.0f*h, 6.0f);
          hsl(1) = cs.bound(0.0f, decay*(s + hsl(1)), 1.0f);

          light = decay*l;
          if (light >= 0.0) {
            hsl(2) *= 255.0f/(255.0f - light);
          } else {
            hsl(2) *= (255.0f + light)/255.0f;
          }
          hsl(2) = cs.bound(0.0f, hsl(2), 255.0f);
        }

        /* convert back */
        cs.hsl2rgb(hsl, rgb);
        o.SetRGB(x, y, rgb(0), rgb(1), rgb(2));
      }
    }
  }
}

void ClusterModel::calcFg(const unsigned width, const unsigned height,
    wxImage& o) {
  wxRect rect(0, 0, width, height);
	calcFg(rect, width, height, o);
}

void ClusterModel::calcAlpha(const unsigned i,
    const wxRect& rect, const unsigned width, const unsigned height,
    channel& c) {
  /* pre-conditions */
  assert (i < k);
  assert ((int)c.size1() == rect.height && (int)c.size2() == rect.width);

  wxImage img(res->calc(rect, width, height));

  #pragma omp parallel default(shared)
  {
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
    ClusterVector<>::type pixel(ClusterVector<>::N);

    int x, y;
    unsigned char r, g, b;

    #pragma omp for schedule(dynamic)
    for (y = 0; y < rect.height; y++) {
      for (x = 0; x < rect.width; x++) {
        r = img.GetRed(x, y);
        g = img.GetGreen(x, y);
        b = img.GetBlue(x, y);

        rgb(0) = r;
        rgb(1) = g;
        rgb(2) = b;
        cs.rgb2hsl(rgb, hsl);

        pixel(0) = r;
        pixel(1) = g;
        pixel(2) = b;
        
        /* cluster adjustments */
        PearsonDistance<>::prepare(pixel);
        if (minClusterer->assign(pixel) == i) {
          c(y,x) = cs.uround(255.0f*threshold(hsl(1), saturationDecay, saturationSoftness));
        } else {
          c(y,x) = 0;
        }
      }
    }
  }
}

sparse_mask ClusterModel::calcMask(const unsigned i,
    const wxRect& rect, const unsigned width, const unsigned height) {
  /* pre-condition */
  assert (i < k);

  sparse_mask m(rect.height, rect.width);
  wxImage img(res->calc(rect, width, height));

  #pragma omp parallel default(shared)
  {
    ClusterVector<>::type pixel(ClusterVector<>::N);
    unsigned char r, g, b;
    int x, y;

    #pragma omp for schedule(dynamic)
    for (y = 0; y < rect.height; y++) {
      for (x = 0; x < rect.width; x++) {
        r = img.GetRed(x,y);
        g = img.GetGreen(x,y);
        b = img.GetBlue(x,y);

        pixel(0) = static_cast<float>(r);
        pixel(1) = static_cast<float>(g);
        pixel(2) = static_cast<float>(b);

        PearsonDistance<>::prepare(pixel);
        if (minClusterer->assign(pixel) == i) {
          #pragma omp critical
          {
            /* sparse matrix, so need to control access, elements may be
             * rearranged */
            m(y,x) = true;
          }
        }
      }
    }
  }
  return m;
}

void ClusterModel::prepare() {
  if (locked) {
    return;
  }

  /* prepare data set for clustering */
  wxImage* original = res->get();
  wxImage* working;
  int originalWidth = original->GetWidth();
  int originalHeight = original->GetHeight();
  int pixels = originalWidth*originalHeight;
  int workingWidth, workingHeight;
  float factor = maxPixels / static_cast<float>(pixels);
    
  if (factor > 1.0) {
    /* image size doesn't exceed limit, cluster it directly */
    workingWidth = originalWidth;
    workingHeight = originalHeight;
    working = original;
  } else {
    /* image size exceeds limit, cluster scaled version */
    workingWidth = static_cast<int>(originalWidth*sqrt(factor));
    workingHeight = static_cast<int>(originalHeight*sqrt(factor));
    working = res->get(workingWidth, workingHeight);
  }

  data.clear();
  
  /* prepare data set for clustering */
  #pragma omp parallel default(shared)
  {
    ClusterVector<>::type pixel(ClusterVector<>::N);
    int x, y;
    unsigned char r, g, b, s;
    
    for (x = 0; x < workingWidth; x++) {
      #pragma omp for schedule(dynamic,16)
      for (y = 0; y < workingHeight; y++) {
        r = working->GetRed(x,y);
        g = working->GetGreen(x,y);
        b = working->GetBlue(x,y);
        
        /* saturation threshold */
        s = saturation(r,g,b);
        if (s >= saturationThreshold) {
          pixel(0) = r;
          pixel(1) = g;
          pixel(2) = b;
          PearsonDistance<>::prepare(pixel);
          #pragma omp critical
          {
            data.add(pixel);
          }
        }
      }
    }
  }
}

void ClusterModel::cluster() {
  if (locked) {
    return;
  }
  const unsigned MAX_ITERS = 100;
  const unsigned seed = time(NULL);
  float minError = std::numeric_limits<float>::max();

  #pragma omp parallel shared(minError)
  {
    /* cluster */
    KMeansClusterer<>* clusterer;
    float error;
    int i;
    ClusterVector<>::type pixel(ClusterVector<>::N);
    
    #pragma omp for schedule(dynamic)
    for (i = 0; i < (int)reps; i++) {
      clusterer = new KMeansClusterer<>(k, seed+i);
      clusterer->cluster(data, MAX_ITERS);
      error = clusterer->getError();
      #pragma omp critical
      {
        if (error < minError) {
          delete minClusterer;
          minError = error;
          minClusterer = clusterer;
        } else {
          delete clusterer;
        }
        //wxYield();
      }
    }
    assert (minClusterer != NULL);

    /* cluster colours */
    ColourSpace cs;
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
    #pragma omp for
    for (i = 0; i < (int)k; i++) {
      pixel = 127.0*minClusterer->getCentroid(i) +
          ublas::scalar_vector<float>(3,128.0);
      rgb(0) = static_cast<unsigned char>(pixel(0));
      rgb(1) = static_cast<unsigned char>(pixel(1));
      rgb(2) = static_cast<unsigned char>(pixel(2));
      cs.rgb2hsl(rgb, hsl);
      hsl(1) = 0.8f;
      hsl(2) = 190.0f;
      cs.hsl2rgb(hsl, rgb);

      colours[i].Set(rgb(0), rgb(1), rgb(2));
    }
  }
}
